/*  ocrad - Optical Character Recognition program
    Copyright (C) 2003 Antonio Diaz Diaz.

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include <cstdio>
#include <list>
#include <vector>
#include "common.h"
#include "rectangle.h"
#include "block.h"
#include "profile.h"


Profile::Profile( const Block & b, type t ) throw()
  : _block( &b ), _blockmap( b.blockmap() ), _type(t), valid(false),
    valid_max(false), valid_min(false), valid_mean(false),
    valid_range(false), valid_isconcave(false),
    valid_isconvex(false), valid_isflat(false), valid_isflats(false),
    valid_ispit(false), valid_isupit(false), valid_isvpit(false),
    valid_istip(false) {}


void Profile::initialize() throw()
  {
  const Block & b = *_block;

  valid = true;
  switch( _type )
    {
    case left :
      data.resize( b.height() ); _limit = b.width();
      for( int i = 0; i < samples(); ++i )
        {
        int j = b.left(), row = b.top() + i;
        while( j <= b.right() && b.id( row, j ) != b.id() ) ++j;
        data[i] = j - b.left();
        } break;
    case top :
      data.resize( b.width() ); _limit = b.height();
      for( int i = 0; i < samples(); ++i )
        {
        int j = b.top(), col = b.left() + i;
        while( j <= b.bottom() && b.id( j, col ) != b.id() ) ++j;
        data[i] = j - b.top();
        } break;
    case right :
      data.resize( b.height() ); _limit = b.width();
      for( int i = 0; i < samples(); ++i )
        {
        int j = b.right(), row = b.top() + i;
        while( j >= b.left() && b.id( row, j ) != b.id() ) --j;
        data[i] = b.right() - j;
        } break;
    case bottom :
      data.resize( b.width() ); _limit = b.height();
      for( int i = 0; i < samples(); ++i )
        {
        int j = b.bottom(), col = b.left() + i;
        while( j >= b.top() && b.id( j, col ) != b.id() ) --j;
        data[i] = b.bottom() - j;
        } break;
    case height :
      data.resize( b.width() ); _limit = b.height();
      for( int i = 0; i < samples(); ++i )
        {
        int u = b.top(), d = b.bottom(), col = b.left() + i;
        while( u <= d && b.id( u, col ) != b.id() ) ++u;
        while( u <= d && b.id( d, col ) != b.id() ) --d;
        data[i] = ( u <= d ) ? d - u + 1 : 0;
        } break;
    case width :
      data.resize( b.height() ); _limit = b.width();
      for( int i = 0; i < samples(); ++i )
        {
        int l = b.left(), r = b.right(), row = b.top() + i;
        while( l <= r && b.id( row, l ) != b.id() ) ++l;
        while( l <= r && b.id( row, r ) != b.id() ) --r;
        data[i] = ( l <= r ) ? r - l + 1 : 0;
        } break;
    case zheight :
      data.resize( b.width() ); _limit = b.height();
      for( int i = 0; i < samples(); ++i )
        {
        int col = b.left() + i;
        data[i] = 0;
        for( int j = b.top(); j <= b.bottom(); ++j )
          if( b.id( j, col ) == b.id() ) ++data[i];
        } break;
    case zwidth :
      data.resize( b.height() ); _limit = b.width();
      for( int i = 0; i < samples(); ++i )
        {
        int row = b.top() + i;
        data[i] = 0;
        for( int j = b.left(); j <= b.right(); ++j )
          if( b.id( row, j ) == b.id() ) ++data[i];
        } break;
    }
  }


int Profile::mean() throw()
  {
  if( !valid_mean )
    {
    valid_mean = true; if( !valid ) initialize();
    _mean = 0;
    for( int i = 0; i < samples(); ++i ) _mean += data[i];
    if( samples() > 1 ) _mean /= samples();
    }
  return _mean;
  }


int Profile::limit() throw()
  {
  if( !valid ) initialize(); return _limit;
  }


int Profile::max() throw()
  {
  if( !valid_max )
    {
    valid_max = true; if( !valid ) initialize();
    _max = data[0];
    for( int i = 1; i < samples(); ++i ) if( data[i] > _max ) _max = data[i];
    }
  return _max;
  }


int Profile::min() throw()
  {
  if( !valid_min )
    {
    valid_min = true; if( !valid ) initialize();
    _min = data[0];
    for( int i = 1; i < samples(); ++i ) if( data[i] < _min ) _min = data[i];
    }
  return _min;
  }


int Profile::operator[]( int i ) throw()
  {
  if( !valid ) initialize();
  if( i < 0 ) i = 0;
  else if( i >= samples() ) i = samples() - 1;
  return data[i];
  }

int Profile::pos( int p ) throw()
  { return ( ( samples() - 1 ) * p ) / 100; }


int Profile::range() throw()
  {
  if( !valid_range )
    {
    valid_range = true; if( !valid ) initialize();
    _range = 0;
    for( int i = 0; i < samples(); ++i ) if( data[i] > _range )
      _range = data[i];
    }
  return _range;
  }


bool Profile::increasing( int i ) throw()
  {
  if( !valid ) initialize();
  if( i < 0 || i > samples() - 2 || data[i] >= data[samples()-1] )
    return false;
  for( ++i; i < samples(); ++i ) if( data[i] < data[i-1] ) return false;
  return true;
  }


bool Profile::decreasing( int i ) throw()
  {
  if( !valid ) initialize();
  if( i < 0 || i > samples() - 2 || data[i] <= data[samples()-1] )
    return false;
  for( ++i; i < samples(); ++i ) if( data[i] > data[i-1] ) return false;
  return true;
  }


bool Profile::isconvex() throw()
  {
  if( !valid_isconvex )
    {
    valid_isconvex = true; _isconvex = false; if( !valid ) initialize();
    int min = _limit, min_begin = 0, min_end = 0;
    int lmin = _limit, rmax = -_limit, l = 0, r = 0;

    for( int i = 1; i < samples(); ++i )
      {
      int d = data[i] - data[i-1];
      if( d < lmin ) { lmin = d; l = i; }
      if( d >= rmax ) { rmax = d; r = i; }
      if( data[i] <= min )
        { min_end = i; if( data[i] < min ) { min = data[i]; min_begin = i; } }
      }
    if( l >= r || l >= pos( 25 ) || r <= pos( 75 ) ) return _isconvex;
    if( lmin >= 0 || rmax <= 0 ) return _isconvex;
    if( min_end - min_begin > samples() / 2 ) return _isconvex;
    if( l >= min_begin || r <= min_end ) return _isconvex;
    int imin = ( min_begin + min_end ) / 2;
    if( imin < pos( 25 ) || imin > pos( 80 ) ) return _isconvex;

    int dmax = -_limit;
    for( int i = l; i <= r; ++i )
      {
      if( i > min_begin && i < min_end )
        { if( data[i] <= 1 ) continue; else return _isconvex; }
      int d = data[i] - data[i-1];
      if( d > dmax ) dmax = d;
      else if( d < dmax - 2 || ( d < dmax - 1 && d != 0 ) ) return _isconvex;
      }
    _isconvex = true;
    }
  return _isconvex;
  }


bool Profile::isflat() throw()
  {
  if( !valid_isflat )
    {
    valid_isflat = true; _isflat = false; if( !valid ) initialize();
    int th = ( mean() < 1 ) ? 1 : mean();
    for( int i = 2; i < samples() - 2; ++i )
      if( abs( data[i] - th ) > 1 ) return _isflat;
    _isflat = true;
    }
  return _isflat;
  }


bool Profile::isflats() throw()
  {
  if( !valid_isflats )
    {
    valid_isflats = true; _isflats = false; if( !valid ) initialize();
    int begin = pos( 10 ); if( begin < 3 ) begin = 3;
    int end = pos( 90 ); if( end > samples() - 4 ) end = samples() - 4;
    if( 2 * ( end - begin ) <= samples() ) return _isflats;

    int mean1 = 0;
    for( int i = begin; i < end; ++i ) mean1 += data[i];
    mean1 /= ( end - begin );

    for( int i = 1; i < begin; ++i )
      if( data[i] - mean1 > 1 ) return _isflats;
    for( int i = begin; i < end; ++i )
      if( abs( data[i] - mean1 ) > 1 ) return _isflats;
    for( int i = end; i < samples() - 1; ++i )
      if( data[i] - mean1 > 1 ) return _isflats;
    _isflats = true;
    }
  return _isflats;
  }


bool Profile::ispit() throw()
  {
  if( !valid_ispit )
    {
    valid_ispit = true; _ispit = false; if( !valid ) initialize();
    if( samples() < 5 ) return _ispit;

    int th = ( mean() < 2 && range() > 2 ) ? 2 : mean();
    if( data[0] <= th && data[1] <= th ) return _ispit;
    if( data[samples()-1] <= th && data[samples()-2] <= th ) return _ispit;
    int status = 0;
    for( int i = 2; i < samples() - 2; ++i )
      switch( status )
        {
        case 0: if( data[i] < th ) status = 1; break;
        case 1: if( data[i] > th ) status = 2; break;
        case 2: if( data[i] < th ) return _ispit;
        }
    _ispit = ( status >= 1 );
    }
  return _ispit;
  }


bool Profile::isupit() throw()
  {
  if( !valid_isupit )
    {
    valid_isupit = true; _isupit = false; if( !valid ) initialize();
    if( samples() < 5 ) return _isupit;

    int th = ( mean() < 2 && range() > 2 ) ? 2 : mean();
    int status = 0, ucount = 0, lcount = 0, umean =0, lmean = 0;
    for( int i = 0; i < samples(); ++i )
      {
      int d = data[i];
      switch( status )
        {
        case 0: if( d < th )
                  { if( i < pos( 25 ) || i > pos( 70 ) ) return _isupit;
                  status = 1; break; }
                if( d > th ) { ++ucount; umean += d; }
          break;
        case 1: if( d > th )
                  { if( i < pos( 30 ) || i > pos( 75 ) ) return _isupit;
                  status = 2; break; }
                if( d < th ) { ++lcount; lmean += d; }
          break;
        case 2: if( d < th ) return _isupit;
                if( d > th ) { ++ucount; umean += d; }
          break;
        }
      }
    if( ucount > 1 ) umean /= ucount;
    if( lcount > 1 ) lmean /= lcount;
    _isupit = ( status == 2 && umean - lmean > range() / 2 );
    }
  return _isupit;
  }


bool Profile::isvpit() throw()
  {
  if( !valid_isvpit )
    {
    valid_isvpit = true; if( !valid ) initialize();
    if( !ispit() ) { _isvpit = false; return _isvpit; }
    int count1 = 0, count2 = 0;
    for( int i = 0; i < samples(); ++i )
      if( data[i] <= 2 ) { if( data[i] < 2 ) ++count1; else ++count2; }
    _isvpit = ( count1 * 4 < samples() ||
                ( count1 * 3 < samples() && count2 <= 2 ) );
    }
  return _isvpit;
  }


bool Profile::istip() throw()
  {
  if( !valid_istip )
    {
    valid_istip = true; _istip = false; if( !valid ) initialize();
    if( samples() < 5 ) return _istip;

    int th = ( mean() < 2 && range() > 2 ) ? 2 : mean();
    if( data[0] >= th && data[1] >= th ) return _istip;
    if( data[samples()-1] >= th && data[samples()-2] >= th ) return _istip;
    int status = 0;
    for( int i = 2; i < samples() - 2; ++i )
      switch( status )
        {
        case 0: if( data[i] > th ) status = 1; break;
        case 1: if( data[i] < th ) status = 2; break;
        case 2: if( data[i] > th ) return _istip;
        }
    _istip = ( status >= 1 );
    }
  return _istip;
  }


bool Profile::isctip( int cpos ) throw()
  {
  if( !valid ) initialize();
  if( samples() < 5 ) return false;
  int th = ( mean() < 2 ) ? 2 : mean();
  int imax = -1, mid = ( ( samples() - 1 ) * cpos ) / 100;

  for( int i = 0; i < samples() / 4; ++i )
    {
    if( data[mid+i] > th ) { imax = mid + i; break; }
    if( data[mid-i-1] > th ) { imax = mid - i - 1; break; }
    }
  if( imax < 0 ) return false;

  for( int i = imax + 1; i < samples(); ++i )
    if( data[i] < th )
      {
      for( int j = imax - 1; j >= 0; --j ) if( data[j] < th ) return true;
      break;
      }
  return false;
  }


int Profile::imaximum() throw()
  {
  if( !valid ) initialize();
  int mbegin = 0, mend, mvalue = 0;
  for( int i = 0; i < samples(); ++i )
    if( data[i] > mvalue ) { mvalue = data[i]; mbegin = i; }
  for( mend = mbegin + 1; mend < samples(); ++mend )
    if( data[mend] < mvalue ) break;
  return ( mbegin + mend - 1 ) / 2;
  }


int Profile::iminimum( int m, int th ) throw()
  {
  if( !valid ) initialize();
  if( samples() < 1 ) return false;
  if( th < 2 ) th = ( mean() < 2 ) ? 2 : mean();
  int minima = 0, status = 0;
  int begin = 0, end, value = _limit + 1;

  for( end = 0; end < samples(); ++end )
    {
    if( status == 0 )
      { if( data[end] < th ) { status = 1; ++minima; begin = end; } }
    else if( data[end] > th )
      { if( minima == m + 1 ) { --end; break; } else status = 0; }
    else if( end == samples() - 1 ) break;
    }
  if( minima != m + 1 ) return 0;

  for( int i = begin; i <= end; ++i )
    if( data[i] < value ) { value = data[i]; begin = i; }
  for( ; end >= begin; --end ) if( data[end] == value ) break;
  return ( begin + end ) / 2;
  }


int Profile::maxima() throw()
  {
  if( !valid ) initialize();
  int maxima = ( data[0] > mean() ) ? 1 : 0;
  for( int i = 1; i < samples(); ++i )
    if( data[i] > mean() && data[i-1] <= mean() ) ++maxima;
  return maxima;
  }


int Profile::minima( int th ) throw()
  {
  if( !valid ) initialize();
  if( samples() < 1 ) return false;
  if( th < 2 ) th = ( mean() < 2 ) ? 2 : mean();
  int minima = ( data[0] < th ) ? 1 : 0;
  int status = ( minima ) ? 1 : 0;

  for( int i = 1; i < samples(); ++i )
    switch( status )
      {
      case 0: if( data[i] < th ) { status = 1; ++minima; } break;
      case 1: if( data[i] > th ) status = 0; break;
      }
  return minima;
  }


bool Profile::straight( int * _dy ) throw()
  {
  if( !valid ) initialize();
  if( samples() < 5 ) return false;

  int xl = 1,             yl = data[1];
  int xr = samples() - 2, yr = data[samples()-2];
  int dx = xr - xl, dy = yr - yl;
  for( int i = xl + 1; i < xr; ++i )
    {
    int y = yl; if( dx ) y += ( dy * ( i - xl ) / dx );
    if( abs( y - data[i] ) > 1 ) return false;
    }
  *_dy = dy;
  return true;
  }
