/*  GNU OCRAD - Optical Character Recognition program
    Copyright (C) 2003 Antonio Diaz Diaz.

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include <cstdio>
#include <list>
#include <vector>
#include "common.h"
#include "rectangle.h"
#include "bitmap.h"
#include "block.h"
#include "blockmap.h"


Block * Blockmap::create_block( const Rectangle & r, int id ) throw()
  {
  _block_list.push_back( Block( r, *this, id ) );
  return &_block_list.back();
  }


void Blockmap::delete_block( int id ) throw()
  {
  std::list< Block >::reverse_iterator p = _block_list.rbegin();
  for( ; p != _block_list.rend(); ++p )
    if( id == p->id() )
      {
      for( int row = p->top(); row <= p->bottom(); ++row )
        for( int col = p->left(); col <= p->right(); ++col )
          if( data[row][col] == id ) data[row][col] = 0;
      _block_list.erase( --(p.base()) ); return;
      }

  internal_error( "delete_block, lost block" );
  }


Block * Blockmap::join_blocks( int id1, int id2 ) throw()
  {
  if( abs( id1 ) > abs( id2 ) )
    { int temp = id1; id1 = id2; id2 = temp; }

  std::list< Block >::reverse_iterator p1 = _block_list.rbegin();
  std::list< Block >::reverse_iterator p2 = _block_list.rbegin();
  while( p1 != _block_list.rend() && id1 != p1->id() ) ++p1;
  while( p2 != _block_list.rend() && id2 != p2->id() ) ++p2;
  if( p1 == _block_list.rend() || p2 == _block_list.rend() )
    internal_error( "join_blocks, lost block" );

  for( int row = p2->top(); row <= p2->bottom(); ++row )
    for( int col = p2->left(); col <= p2->right(); ++col )
      if( data[row][col] == id2 ) data[row][col] = id1;
  p1->add_rectangle( *p2 );
  _block_list.erase( --(p2.base()) );
  return &(*p1);
  }


Block * Blockmap::add_point_to_block( int row, int col, int id ) throw()
  {
  std::list< Block >::reverse_iterator p = _block_list.rbegin();
  for( ; p != _block_list.rend(); ++p )
    if( id == p->id() ) { p->add_point( row, col ); return &(*p); }

  internal_error( "add_point_to_block, lost block" );
  }


void delete_small_blocks( std::list< Block > & block_list ) throw()
  {
  std::list< Block >::iterator p = block_list.begin();
  while( p != block_list.end() )
    {
    std::list< Block >::iterator next = p;
    ++next;
    if( p->height() <= 4 && p->width() <= 4 &&
        ( ( p->height() <= 2 && p->width() <= 2 ) || p->area() <= 4 ) )
      block_list.erase( p );
    p = next;
    }
  }


void delete_white_blocks( std::list< Block > & block_list ) throw()
  {
  std::list< Block >::iterator p = block_list.begin();
  while( p != block_list.end() )
    {
    std::list< Block >::iterator next = p;
    ++next;
    if( p->id() < 0 ) block_list.erase( p );
    p = next;
    }
  }


void delete_wide_blocks( std::list< Block > & block_list, int width ) throw()
  {
  std::list< Block >::iterator p = block_list.begin();
  while( p != block_list.end() )
    {
    std::list< Block >::iterator next = p;
    ++next;
    if( 2 * p->width() > width ) block_list.erase( p );
    p = next;
    }
  }


void hierarchize_blocks( std::list< Block > & block_list ) throw()
  {
  std::list< Block >::iterator p1 = block_list.begin();
  for( ; p1 != block_list.end(); ++p1 ) if( p1->id() > 0 )
    {
    std::list< Block >::iterator p2 = p1; ++p2;
    while( p2 != block_list.end() )
      {
      if( p2->top() >= p1->bottom() ) break;
      std::list< Block >::iterator next = p2;
      ++next;
      if( p1->includes( *p2 ) )
        {
        if( p2->id() < 0 )
          p1->_block_list.splice( p1->_block_list.end(), block_list, p2 );
        else
          {
          std::list< Block >::iterator p3 = p1->_block_list.begin();
          for( ; p3 != p1->_block_list.end(); ++p3 )
            if( p3->includes( *p2 ) )
              p3->_block_list.splice( p3->_block_list.end(), block_list, p2 );
          }
        }
      p2 = next;
      }
    }
  }


void order_blocks( std::list< Block > & block_list ) throw()
  {
  std::list< Block >::iterator p1 = block_list.begin();
  while( p1 != block_list.end() )
    {
    std::list< Block >::iterator best = p1;
    std::list< Block >::iterator p2 = p1;
    while( ++p2 != block_list.end() && p2->top() <= best->bottom() )
      if( p2->left() < best->left() ) best = p2;
    if( best != p1 ) block_list.splice( p1, block_list, best );
    else ++p1;
    }
  }


void remove_top_bottom_noise( std::list< Block > & block_list ) throw()
  {
  std::list< Block >::iterator p = block_list.begin();
  for( ; p != block_list.end(); ++p )
    {
    Block & b = *p;
    if( b.height() <= 10 ) continue;

    int c = 0;
    for( int col = b.left(); col <= b.right(); ++col )
      if( b.id( b.top(), col ) == b.id() && ++c > 1 ) break;
    if( c <= 1 ) b.top( b.top() + 1 );

    c = 0;
    for( int col = b.left(); col <= b.right(); ++col )
      if( b.id( b.bottom(), col ) == b.id() && ++c > 1 ) break;
    if( c <= 1 ) b.bottom( b.bottom() - 1 );
    }
  }


int generate_black_id() { static int next = 0; return ++next; }

int generate_white_id() { static int next = 0; return --next; }

Blockmap::Blockmap( const Bitmap & page_image, int rindex, int debug_level )
                                                                     throw()
  {
  if( rindex < 0 ||
      rindex >= static_cast<int>(page_image.rectangle_vector().size()) )
    internal_error( "Blockmap, text rectangle index out of range" );

  const Rectangle & r = page_image.rectangle_vector()[rindex];
  int row0 = r.top();
  int col0 = r.left();
  _height =  r.height();
  _width =   r.width();
  data.resize( _height );

  for( int row = 0; row < _height; ++row )
    { data[row].resize( _width ); data[row][0] = 0; }
  for( int col = 0; col < _width; ++col ) data[0][col] = 0;

  for( int row = 1; row < _height; ++row )
    {
    Block *pb = 0, *pw = 0;
    for( int col = 1; col < _width; ++col )
      {
      if( page_image.get_bit( row0 + row, col0 + col ) )	// black point
        {
        if( data[row-1][col] > 0 )
          {
          int id = data[row][col] = data[row-1][col];
          if( pb && pb->id() == id ) pb->add_point( row, col );
          else pb = add_point_to_block( row, col, id );
          if( data[row][col-1] > 0 && data[row][col-1] != id )
            pb = join_blocks( data[row][col-1], id );
          }
        else if( data[row][col-1] > 0 )
          {
          int id = data[row][col] = data[row][col-1];
          if( pb && pb->id() == id ) pb->add_point( row, col );
          else pb = add_point_to_block( row, col, id );
          }
        else
          {
          int id = data[row][col] = generate_black_id();
          pb = create_block( Rectangle( col, row, col, row ), id );
          }
        }
      else						// white point
        {
        if( data[row][col-1] == 0 )
          {
          data[row][col] = 0; pw = 0;
          if( data[row-1][col] < 0 ) delete_block( data[row-1][col] );
          }
        else if( data[row-1][col] == 0 )
          {
          data[row][col] = 0; pw = 0;
          if( data[row][col-1] < 0 ) delete_block( data[row][col-1] );
          }
        else if( data[row-1][col] < 0 )
          {
          int id = data[row][col] = data[row-1][col];
          if( pw && pw->id() == id ) pw->add_point( row, col );
          else pw = add_point_to_block( row, col, id );
          if( data[row][col-1] < 0 && data[row][col-1] != id )
            pw = join_blocks( data[row][col-1], id );
          }
        else if( data[row][col-1] < 0 )
          {
          int id = data[row][col] = data[row][col-1];
          if( pw && pw->id() == id ) pw->add_point( row, col );
          else pw = add_point_to_block( row, col, id );
          }
        else
          {
          int id = data[row][col] = generate_white_id();
          pw = create_block( Rectangle( col, row, col, row ), id );
          }
        }
      }
    }
  delete_small_blocks( _block_list );
  delete_wide_blocks( _block_list, _width );
  if( debug_level <= 99 ) hierarchize_blocks( _block_list );
  delete_white_blocks( _block_list );
  remove_top_bottom_noise( _block_list );
  if( debug_level <= 97 ) order_blocks( _block_list );
  }


void Blockmap::print( FILE * outfile, int debug_level ) const throw()
  {
  fprintf( outfile, "%d blocks\n", _block_list.size() );
  fprintf( outfile, "%d %d\n\n", _width, _height );

  int sp = (debug_level & 1) ? 0 : -1;
  std::list< Block >::const_iterator p = _block_list.begin();
  for( ; p != _block_list.end(); ++p )
    p->print( outfile, sp );
  }
