/*  GNU OCRAD - Optical Character Recognition program
    Copyright (C) 2003 Antonio Diaz Diaz.

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include <cstdio>
#include <list>
#include <vector>
#include "common.h"
#include "rectangle.h"
#include "block.h"
#include "profile.h"
#include "iso_8859_1.h"
#include "features.h"


Features::Features( const Block & b ) throw()
  : _block( &b ), _blockmap( b.blockmap() ), _hbars( -1 ), _vbars( -1 ),
    hscan_valid( false ), vscan_valid( false ), lp( b, Profile::left ),
    tp( b, Profile::top ), rp( b, Profile::right ), bp( b, Profile::bottom ),
    hp( b, Profile::height ), wp( b, Profile::width )
  {}


int Features::hbars() const throw()
  {
  if( _hbars < 0 )
    {
    const Block & b = *_block;
    int state = 0, begin = 0, l = 0, r = 0, limit = wp.max() / 2;
    int count[b.height()];
    _hbars = 0;

    for( int row = b.top(); row <= b.bottom(); ++row )
      {
      int col, c = 0, lt = 0, rt = 0, x = 0;
      int & maxcount = count[row-b.top()];
      maxcount = 0;
      for( col = b.left(); col <= b.right(); ++col )
        {
        if( b.id( row, col ) == b.id() )
          { ++c; x = col; if( col < b.right() ) continue; }
        if( c > maxcount ) { maxcount = c; rt = x; lt = rt - c + 1; } c = 0;
        }
      switch( state )
        {
        case 0: if( maxcount > limit )
                  { state = 1; begin = row; l = lt; r = rt; }
                else break;
        case 1: if( maxcount > limit )
                  {
                  if( lt < l ) l = lt;
                  if( rt > r ) r = rt;
                  if( row < b.bottom() ) break;
                  }
                int end = ( maxcount <= limit ) ? row - 1 : row;
                int width = r - l + 1;
                while( begin <= end && 3 * count[begin-b.top()] < 2 * width )
                  ++begin;
                while( begin <= end && 3 * count[end-b.top()] < 2 * width )
                  --end;
                if( begin > end ) break;
                _hbar.push_back( Rectangle( l, begin, r, end ) );
                ++_hbars; state = 0; break;
        }
      }
    }
  return _hbars;
  }


int Features::vbars() const throw()	// FIXME small gaps not detected
  {
  if( _vbars < 0 )
    {
    const Block & b = *_block;
    int state = 0, begin = 0, limit = b.height();
    limit -= ( b.height() < 40 ) ? 3 : b.height() / 10;
    _vbars = 0;

    for( int col = b.left(); col <= b.right(); ++col )
      {
      int c = 0, c2 = 0, count = 0;
      for( int row = b.top() + 1; row < b.bottom(); ++row )
        {
        if( b.id( row, col ) == b.id() )
          { ++c; if( row < b.bottom() - 1 ) continue; }
        else if( ( col > b.left() && b.id( row, col - 1 ) == b.id() ) ||
                 ( col < b.right() && b.id( row, col + 1 ) == b.id() ) )
          { ++c; ++c2; if( row < b.bottom() - 1 ) continue; }
        if( c > count ) { count = c; } c = 0;
        }
      if( ( count - c2 ) * 3 < limit * 2 ) count = 0;
      switch( state )
        {
        case 0: if( count >= limit ) { state = 3; begin = col; }
                else if( count * 4 >= limit * 3 ) { state = 2; begin = col; }
                else if( count * 3 >= limit * 2 ) { state = 1; begin = col; }
                break;
        case 1: if( count >= limit ) state = 3;
                else if( count * 4 >= limit * 3 ) state = 2;
                else if( count * 3 < limit * 2 ) state = 0;
                else begin = col;
                break;
        case 2: if( count >= limit ) state = 3;
                else if( count * 3 < limit * 2 ) state = 0;
                else if( count * 4 < limit * 3 ) state = 1;
                break;
        case 3: if( count * 3 < limit * 2 || col == b.right() )
                  {
                  int end = ( col == b.right() ) ? col : col - 1;
                  _vbar.push_back( Rectangle( begin, b.top(), end, b.bottom() ) );
                  ++_vbars; state = 0;
                  }
        }
      }
    }
  return _vbars;
  }


// return the number of horizontal traces crossing every column
const std::vector< int > & Features::hscan() const throw()
  {
  if( !hscan_valid )
    {
    const Block & b = *_block;
    int state = 0;
    hscan_valid = true; _hscan.resize( b.width() );

    for( int col = b.left(); col <= b.right(); ++col )
      {
      _hscan[col-b.left()] = 0;
      for( int row = b.top(); row <= b.bottom(); ++row )
        {
        bool black = ( b.id( row, col ) == b.id() );
        switch( state )
          {
          case 0: if( black ) state = 1; break;
          case 1: if( !black || row == b.bottom() )
                    { state = 0; ++_hscan[col-b.left()]; }
          }
        }
      }
    }
  return _hscan;
  }


// return the number of vertical traces crossing every row
const std::vector< int > & Features::vscan() const throw()
  {
  if( !vscan_valid )
    {
    const Block & b = *_block;
    int state = 0;
    vscan_valid = true; _vscan.resize( b.height() );

    for( int row = b.top(); row <= b.bottom(); ++row )
      {
      _vscan[row-b.top()] = 0;
      for( int col = b.left(); col <= b.right(); ++col )
        {
        bool black = ( b.id( row, col ) == b.id() );
        switch( state )
          {
          case 0: if( black ) state = 1; break;
          case 1: if( !black || col == b.right() )
                    { state = 0; ++_vscan[row-b.top()]; }
          }
        }
      }
    }
  return _vscan;
  }


// Looks for three black sections in column hcenter()  1, then tests if
// upper and lower gaps are open to the right or to the left
unsigned char Features::test_235Esz() const throw()
  {
  const Block & b = *_block;
  int lrow1 = 0, urow2 = 0, lrow2 = 0, urow3 = 0, col, black_section = 0;

  for( col = b.hcenter() - 1; col <= b.hcenter() + 1; ++col )
    {
    bool prev_black = false;
    for( int row = b.top(); row <= b.bottom(); ++row )
      {
      bool black = ( b.id( row, col ) == b.id() );
      if( black )
        { if( !prev_black )
          { if( ++black_section == 2 ) urow2 = row - 1;
          else if( black_section == 3 ) urow3 = row - 1; }
        }
      else if( prev_black )
        { if( black_section == 1 ) lrow1 = row;
          else if( black_section == 2 ) lrow2 = row; }
      prev_black = black;
      }
    if( black_section == 3 && lrow1 <= b.vpos(30) && urow3 >= b.vpos(70) )
      break;
    black_section = 0;
    }

  if( black_section != 3 ) return 0;
  if( b.escape_left( lrow2, col ) )
    {
    if( b.escape_left( urow2, col ) )
      {
      if( b.escape_bottom( lrow2, col ) && b.escape_top( urow2, col ) ) return'*';
      return '3';
      }
    if( b.escape_right( urow2, col ) )
      {
      if( b.seek_right( lrow1 + 1, col ) >= b.right() ) return '5';
      if( urow2 > b.vcenter() + 1 && lrow2 > b.bottom() - (b.height() / 5) &&
          b.seek_right( urow2 - 1, col ) < b.right() )
        return '';
      return 's';
      }
    }
  else if( b.escape_right( lrow2, col ) )
    {
    if( b.escape_right( urow2, col ) )
      {
      if( b.escape_bottom( lrow2, col ) && b.escape_top( urow2, col ) )
        return '*';
      if( bp.minima( b.height() / 5 ) == 1 )
        {
        if( b.escape_bottom( urow3, col ) ) return 'F';
        if( b.escape_top( urow2, b.left() ) &&
            !b.escape_top( lrow2, b.left() ) ) return 'f';
        else return 'E';
        }
      }
    if( b.escape_left( urow2, col ) )
      {
      if( tp.isflat() ) return 'z';
      if( tp.isconvex() ) return '2';
      if( b.height() > 2 * wp.max() && bp.minima( b.height() / 5 ) == 1 )
        { if( b.height() >= 3 * wp.max() ) return 'l'; else return 'I'; }
      return 'z';
      }
    }
  return 0;
  }


// Tests if the lower half of character is open to the left, to the right,
// and/or to the bottom
unsigned char Features::test_49egpq() const throw()
  {
  const Block & b = *_block;
  const Block & h = b.block_list().front();		// hole

  int col = h.hcenter();
  int row = b.seek_bottom( h.bottom(), col, false ) + 1;
  if( row >= b.vpos(90) )
    { col = h.left(); row = b.seek_bottom( h.bottom(), col, false ) + 1; }
  if( row >= b.bottom() ) return 0;

  if( b.escape_right( row, col ) )
    {
    if( lp.ispit() && b.seek_bottom( row, h.right() ) < b.bottom() )
      return 'e';
    else return 'p';
    }

  if( b.escape_left( row, col ) )
    {
    Profile hlp( h, Profile::left );
    Profile htp( h, Profile::top );
    if( vbars() == 1 && vbar(0).hcenter() > b.hcenter() &&
        hlp.decreasing() && htp.decreasing() )
      return '4';
    if( rp.isconvex() && tp.isconvex() && rp.minima() == 1 )
      return '9';
    int hdiff;
    if( b.top_hook( &hdiff ) && hdiff > 0 ) return 's';
    if( b.bottom_hook( &hdiff ) && hdiff > 0 ) return 'g';
    if( row > b.vpos(80) ) return 'Q';
    if( b.seek_bottom( row, col ) < b.bottom() &&
        rp.increasing( row - b.top() ) ) return 'g';
    return 'q';
    }
  return 0;
  }


// Tests if the upper half of character is open to the left, to the right,
// and/or to the bottom
unsigned char Features::test_6abd() const throw()
  {
  const Block & b = *_block;
  const Block & h = b.block_list().front();		// hole

  int col = h.hcenter();
  int row = b.seek_top( h.top(), col, false ) - 1;
  if( row <= b.top() )
    { col = h.right(); row = b.seek_top( h.top(), col, false ) - 1; }
  if( row <= b.top() ) return 0;
  int rcol = ( b.right() + h.right() ) / 2;
  int urow = h.top() - ( b.bottom() - h.bottom() );
  bool oacute1 = ( b.seek_right( urow - 1, h.right() ) >= b.right() );

  if( b.escape_right( row, col ) )
    {
    bool oacute2 = ( b.seek_right( urow, b.left() ) > b.seek_right( h.top(), b.left() ) + 1 );
//    if( !oacute2 && lp.isconvex() && bp.isconvex() ) return '6';
    int row2 = b.seek_top( h.top(), rcol, false ) - 1;
    row2 = b.seek_top( row2, rcol );
    if( !oacute2 && lp.ispit() && bp.ispit() && row2 > b.top() ) return '6';
    bool oacute3 = ( b.seek_left( b.top() + 1, b.right() + 1 ) >= rcol ||
                     b.seek_right( h.top(), b.left() ) <= b.left() + 1 );
    if( oacute1 && oacute2 && oacute3 ) return '';
    if( row2 <= b.top() ) return 'b'; else return 's';
    }

  if( b.escape_left( row, col ) )
    {
    int row2 = b.seek_top( h.top(), h.left(), false ) - 1;
    row2 = b.seek_top( row2, h.left() );
    if( row2 > b.top() )
      { if( b.vcenter() < row ) return ''; else return 'a'; }
    if( oacute1 ) return ''; else return 'd';
    }
  return 0;
  }


unsigned char Features::test_CEFIJLlT() const throw()
  {
  const Block & b = *_block;

  if( tp.minima( b.height() / 4 ) != 1 || bp.minima( b.height() / 4 ) != 1 )
    return 0;

  int row, col;
  if( b.seek_right( b.vcenter(), b.left() - 1 ) >= b.hcenter() )
    col = b.hpos( 25 );
  else col = b.hpos( 75 );
  row = b.seek_top( b.vcenter(), col );
  if( row <= b.top() || ( row < b.vpos( 25 ) && b.escape_top( row, col ) ) )
    {
    int hdiff;
    if( b.bottom_hook( &hdiff ) )
      {
      if( hdiff > b.height() / 2 && rp.increasing( rp.pos(75) ) ) return 'J';
      if( -hdiff > b.height() / 2 ) return 'L';
      }
    }

  if( vbars() == 1 && vbar(0).width() >= 2 &&
      3 * vbar(0).width() <= b.width() + 2 )
    {
    if( abs( vbar(0).hcenter() - b.hcenter() ) <= 1 &&
        abs( (vbar(0).left() - b.left()) - (b.right() - vbar(0).right()) ) <= 2 )
      {
      if( hbars() == 1 )
        {
        if( hbar(0).top() <= b.top() + 1 ) return 'T';
        if( abs( hbar(0).vcenter() - b.vcenter() ) <= 1 ) return '+';
        }
      if( hbars() == 2 && hbar(0).top() <= b.top() + 1 &&
          3 * hbar(0).width() > 4 * hbar(1).width() )
        return 'T';
      }
    if( vbar(0).right() < b.hcenter() )
      {
      if( hbars() == 3 && hbar(0).top() <= b.top() + 1 &&
          hbar(0).width() + 1 >= hbar(1).width() &&
          vbar(0).h_overlaps( hbar(1) ) )
        {
        if( similar( hbar(0).width(), hbar(2).width(), 10 ) &&
            hbar(2).width() >= hbar(1).width() )
          return 'E';
        if( hbar(0).width() > hbar(2).width() &&
            hbar(1).includes_vcenter( b ) )
          return 'F';
        }
      if( hbars() == 2 )
        {
        if( hbar(0).top() <= b.top() + 1 &&
            hbar(0).width() > hbar(1).width() &&
            hbar(1).includes_vcenter( b ) )
          return 'F';
        if( hbar(1).bottom() >= b.bottom() - 1 &&
            b.height() > b.width() &&
            hbar(1).width() > hbar(0).width() &&
            abs( vbar(0).hcenter() - hbar(0).hcenter() ) <= 1 &&
            rp.iminimum() > rp.pos( 70 ) )
          return 'L';
        }
      }
    if( vbar(0).left() > b.hcenter() )
      {
      if( hbars() == 1 && hbar(0).top() <= b.top() + 1 &&
          hbar(0).width() + 1 >= b.width() )
        return '';
      }
    }

  if( vbars() == 1 && vbar(0).width() >= 2 &&
      vbar(0).width() * 2 < b.width() && vbar(0).right() < b.hcenter() &&
      hbars() == 1 && hbar(0).bottom() >= b.bottom() - 1 &&
      similar( hbar(0).width(), b.width(), 10 ) )
    return 'L';

  if( vbars() == 1 && vbar(0).width() >= 2 )
    {
    if( 3 * b.height() > 4 * b.width() &&
        abs( vbar(0).hcenter() - b.hcenter() ) <= 1 )
      {
      if( rp.istip() && ( lp.istip() || lp.isflats() ) ) return 'I';
      if( rp.isflats() && ( lp.istip() || lp.isflats() ) ) return 'l';
      if( rp.istip() && lp.ispit() && similar( lp.iminimum(), rp.pos(50), 10 ) )
        return '{';
      if( lp.istip() && rp.ispit() && similar( rp.iminimum(), rp.pos(50), 10 ) )
        return '}';
      if( 2 * b.height() > 3 * b.width() )
        if( vbar(0).right() >= b.hpos( 70 ) ||
            b.escape_top( b.vpos( 75 ), vbar(0).right() + 1 ) )
          for( int i = vbar(0).left() - 1; i > b.left(); --i )
            if( b.seek_bottom( b.vpos( 75 ), i ) < b.bottom() &&
                bp[i-b.left()] <= 1 ) return 'l';
      }
    if( vbar(0).left() <= b.left() + 1 && b.height() > 2 * b.width() &&
        rp.istip() )
      {
      int urow = b.seek_top( b.vcenter(), b.hcenter() );
      int lrow = b.seek_bottom( b.vcenter(), b.hcenter() );
      int ucol = b.seek_right( urow, b.hcenter() );
      int lcol = b.seek_right( lrow, b.hcenter() );
      if( ucol < b.right() && lcol < b.right() )
        return 'C';
      return '[';
      }
    if( vbar(0).right() >= b.right() - 1 && b.height() > 2 * b.width() )
      {
      if( lp.istip() )
        {
        if( 2 * vbar(0).width() <= wp.max() ) return ']';
        if( b.height() >= 3 * b.width() ) return 'l';
        }
      if( lp.isvpit() && lp.iminimum() < lp.pos(40) ) return '1';
      }
    }
  if( hbars() == 1 && abs( hbar(0).vcenter() - b.vcenter() ) <= 1 &&
      tp.isupit() && bp.isupit() ) return '+';
  return 0;
  }


unsigned char Features::test_c() const throw()
  {
  const Block & b = *_block;

  if( lp.isconvex() )
    {
    if( b.height() > 2 * b.width() ) return '(';
    if( b.escape_right( b.vcenter(), b.hcenter() ) ) return 'c';
    }
  if( lp.ispit() && bp.ispit() && tp.ispit() &&
      b.escape_right( b.vcenter(), b.hcenter() ) )
    return 'c';
  return 0;
  }


unsigned char Features::test_frst() const throw()
  {
  const Block & b = *_block;

  if( tp.minima( b.height() / 4 ) != 1 || bp.minima( b.height() / 4 ) != 1 )
    return 0;

    {
    int b_hdiff = 0, t_hdiff = 0;
    if( b.bottom_hook( &b_hdiff ) )
      {
      if( -b_hdiff > b.height() / 2 )
        {
        if( b.height() >= 3 * wp.max() &&
            ( hbars() == 0 || hbar(0).bottom() < b.vpos( 20 ) ) ) return 'l';
        if( 3 * wp[1] < b.width() && hbars() >= 1 && hbars() <= 2 &&
            hbar(0).top() >= b.vpos( 20 ) && hbar(0).bottom() < b.vcenter() &&
            similar( hbar(0).width(), wp.max(), 10 ) ) return 't';
        }
      }
    if( b.top_hook( &t_hdiff ) && t_hdiff > b.height() / 2 )
      {
      if( b.height() >= 2 * wp.max() &&
          tp.iminimum() > tp.pos( 50 ) && bp.iminimum() < bp.pos( 50 ) )
        return 'f';
      if( b_hdiff > b.height() / 2 ) return 's';
      }
    }

  if( vbars() != 1 || vbar(0).width() < 2 ) return 0;
  if( vbar(0).hcenter() <= b.hcenter() )
    {
    if( similar( b.height(), b.width(), 40 ) )
      {
      int col = b.right() - rp[b.vcenter()-b.top()] + 2;
      int row = b.seek_bottom( b.vcenter(), col );
      if( row >= b.bottom() || b.escape_bottom( row - 1, col ) )
        {
        if( rp.minima() == 3 ) return 'f';
        if( tp.minima( b.height() / 8 ) == 2 &&
            bp.minima( b.height() / 8 ) == 2 ) return 'x';
        int row = b.vpos(75);
        int col = b.seek_right( row, b.hcenter(), false ) + 1;
        if( b.seek_right( row, col ) >= b.right() )
          { if( lp.isconvex() ) return 'c'; else  return 'r'; }
        }
      if( b.seek_right( row - 1, col ) < b.right() && lp.isctip() )
        return 'x';
      if( lp.isvpit() ) return 't';
      }
    else if( b.height() > b.width() && vbar(0).left() > b.left() &&
             rp.minima() <= 2 )
      {
      int col = b.right() - rp[b.vcenter() - b.top()] + 1;
      if( !b.escape_bottom( b.vcenter(), col ) )
        { if( tp.ispit() && lp.iminimum() < lp.pos(40) ) return 't'; }
      else if( 2 * wp.max() > b.width() )
        {
        if( rp.iminimum() < rp.pos(20) )
          {
          if( rp.increasing( rp.pos(20) ) || bp.increasing() ||
              tp.minima( 1 ) == 2 ) return 'r';
          else return 'f';
          }
        else return 't';
        }
      }
    if( b.seek_bottom( b.vcenter(), b.hpos( 60 ) + 1 ) >= b.bottom() )
      { if( rp.minima() == 2 ) return 'f'; else return 'r'; }
    }
  return 0;
  }


unsigned char Features::test_G() const throw()
  {
  const Block & b = *_block;

  if( lp.isconvex() || lp.ispit() )
    {
    int col = 0, row = 0;
    for( int i = rp.pos( 40 ); i <= rp.pos( 60 ); ++i )
      if( rp[i] > col ) { col = rp[i]; row = i; }
    if( col == 0 ) return 0;
    row += b.top(); col = b.right() - col + 1;
    if( col <= b.left() || col >= b.hcenter() ) return 0;

    col = ( col + b.hcenter() ) / 2;
    row = b.seek_bottom( row, col );
    if( row < b.bottom() && b.escape_right( row, col ) &&
        !b.escape_bottom( row, b.hcenter() ) )
      {
      int lrow, urow;
      for( lrow = row - 1 ; lrow > b.top(); --lrow )
        if( b.seek_right( lrow, b.hcenter() ) >= b.right() ) break;
      for( urow = lrow - 1 ; urow > b.top(); --urow )
        if( b.seek_right( urow, b.hcenter() ) < b.right() ) break;
      lrow += 2; urow -= 1;
      if( lrow < row && urow > b.top() )
        {
        int uwidth = b.seek_left( urow, b.right() ) - b.seek_right( urow, b.hcenter() );
        int lwidth = b.seek_left( lrow, b.right() ) - b.seek_right( lrow, b.hcenter() );
        if( lrow - 1 <= b.vcenter() || lwidth > uwidth + 2 )
          return 'G';
        }
      }
    }
  return 0;
  }


unsigned char Features::test_HKMNUuvwYy() const throw()
  {
  const Block & b = *_block;

  if( tp.minima( b.height() / 4 ) == 2 && tp.isctip() )
    {
    if( b.height() >= 10 && bp.minima( b.height() / 5 ) == 1 )
      {
      int lg = lp[lp.samples()-2];
      if( lg > 1 && bp.isvpit() && tp.minima( b.height()/2 ) == 2 &&
          lp[lp.pos(75)] <= lg ) return 'v';
      int hdiff;
      if( b.bottom_hook( &hdiff ) )
        {
        if( abs( hdiff ) <= b.height() / 8 )
          {
          if( vscan()[b.height()/4] >= 3 ) return 'v';
          if( bp.isconvex() ) return 'u';
          }
        if( hdiff > b.height() / 2 ) return 'y';
        }
      int rg = rp[rp.samples()-2];
      if( bp.ispit() && 7 * tp.range() < 4 * b.height() )
        if( ( similar( lg, rg, 20 ) &&
              5 * wp[wp.pos(98)] >= 4 * wp[wp.pos(70)] ) ||
            ( lg > 1 && lg < rg && !rp.increasing() ) ) return 'Y';
      if( 2 * lg < rg && lp.isctip() ) return 'y';
      if( b.escape_top( b.vcenter(), b.hcenter() ) ) return 'u';
      if( lg < rg + 1 && !lp.increasing() && rp.increasing() &&
          tp.minima( b.height()/2 ) == 1 ) return 'y';
      if( lg > 1 && bp.ispit() && tp.minima( b.height()/2 ) == 2 ) return 'v';
      if( lg <= 1 && 2 * ( b.width() - rg - lg ) < b.width() &&
          rp.increasing() && tp.minima( b.height()/2 ) == 2 ) return 'v';
      return 0;
      }
    if( bp.minima( b.height() / 4 ) == 2 && bp.isctip() )
      {
      if( vscan()[rp.pos(40)] == 4 ) return 'M';
      int lg = lp[lp.pos(50)];
      int rg = rp[rp.pos(50)];
      if( similar( lg, rg, 80 ) && lg < b.width() / 4 && rg < b.width() / 4 )
        {
        if( lg > 1 && rg > 1 && lp.increasing() && rp.increasing() )
          return 'w';
        if( hbars() == 1 && abs( hbar(0).vcenter() - b.vcenter() ) <= 1 &&
            abs( hbar(0).hcenter() - b.hcenter() ) <= 1 )
          return 'H';
        if( vscan()[rp.pos(60)] == 4 || vscan()[rp.pos(70)] == 4 ) return 'w';
        if( b.width() > 2 * b.height() ) return '~'; else return 'N';
        }
      if( 3 * lg < 2 * rg && lg < b.width() / 4 && rg > b.width() / 4 &&
          rp.isctip() ) return 'K';
      return 0;
      }
    if( bp.minima() == 3 ) return 'M';
    }
  return 0;
  }


// Looks for the nearest frontier in column hcenter(), then tests if
// gap is open downwards
unsigned char Features::test_hnw() const throw()
  {
  const Block & b = *_block;

  int col = 0, row = 0;
  for( int i = bp.pos( 40 ); i <= bp.pos( 60 ); ++i )
    if( bp[i] > row ) { row = bp[i]; col = i; }
  if( row < 2 ) return 0;
  row = b.bottom() - row + 1; col += b.left();
  if( row <= b.top() ) return 0;
  //FIXME
  { int c = col; col = b.seek_right( row, col ); if( col > c ) --col;
  row = b.seek_top( row, col ); }

  if( tp.isconvex() && b.width() >= b.height() && lp.decreasing() ) return '^';
  int urow = b.seek_top( row - 1, col, false );
  if( urow > b.vpos( 20 ) )
    {
    if( tp.minima() >= 2 && tp.minima( b.height() / 5 ) >= 2 ) return 'w';
    if( tp.minima( b.height() / 5 ) == 1 )
      { if( rp.isctip( 66 ) ) return 'k'; else return 'h'; }
    }
  if( similar( b.height(), b.width(), 40 ) && row > b.vcenter() &&
      urow < b.vcenter() && tp.minima( b.height() / 5 ) == 2 &&
      bp.minima( urow + 1 ) == 3 ) return 'w';
  if( urow <= b.vpos( 20 ) && tp.minima( b.height() / 5 ) == 1 &&
      ( b.width() > b.height() || similar( b.height(), b.width(), 40 ) ) )
    return 'n';
  return 0;
  }


bool Features::test_comma() const throw()
  {
  const Block & b = *_block;

  if( b.block_list().size() > 0 ) return 0;
  if( 4 * b.height() < 5 * b.width()) return false;
  if( b.height() > 3 * b.width() ) return false;

  if( b.width() >= 3 && b.height() >= 3 )
    {
    int upper_area = 0;
    for( int row = b.top(); row < b.top() + b.width(); ++row )
      for( int col = b.left(); col <= b.right(); ++col )
        if( b.id( row, col ) == b.id() ) ++upper_area;
    if( upper_area < (b.width() - 2) * (b.width() - 2) ) return false;
    int count1 = 0, count2 = 0;
    for( int col = b.left(); col <= b.right(); ++col )
      { if( b.id( b.top() + 1, col ) == b.id() ) ++count1;
      if( b.id( b.bottom() - 1, col ) == b.id() ) ++count2; }
    if( count1 <= count2 ) return false;
    }
  return true;
  }


// Recognizes single line, non-rectangular characters without holes.
// '/<>\^`
unsigned char Features::test_line( int charbox_vcenter ) const throw()
  {
  const Block & b = *_block;
  int slope1, slope2;

  if( lp.straight( &slope1 ) && rp.straight( &slope2 ) )
    {
    if( slope1 < 0 )
      {
      if( slope2 < 0 && bp.minima() == 2 ) return '^';
      if( b.v_includes( charbox_vcenter ) )
        {
        if( 4 * b.area() < b.height() * b.width() ) return '/';
        else return 'l';
        }
      if( b.top() >= charbox_vcenter ) return ',';
      return '\'';
      }
    if( slope1 > 0 && slope2 < 0 )
      {
      if( b.bottom() > charbox_vcenter ) return '\\';
      return '`';
      }
    }
  if( tp.straight( &slope1 ) && bp.straight( &slope2 ) )
    {
    if( slope1 < 0 && slope2 < 0 ) return '<';
    if( slope1 > 0 && slope2 > 0 ) return '>';
    }
  return 0;
  }


unsigned char Features::test_solid( int charbox_vcenter ) const throw()
  {
  const Block & b = *_block;
  if( b.blocks() ) return 0;
  int inner_area, inner_space;

  if( b.width() >= 3 && b.height() >= 3 )
    {
    inner_space = ( b.width() - 2 ) * ( b.height() - 2 );
    inner_area = 0;
    for( int row = b.top() + 1; row < b.bottom(); ++row )
      {
      int holes = 0;
      for( int col = b.left() + 1; col < b.right(); ++col )
        { if( b.id( row, col ) == b.id() ) ++inner_area; else ++holes; }
      if( 5 * holes >= b.width() ) return 0;
      }
    if( inner_area * 100 < inner_space * 75 ) return 0;
    }
  else { inner_space = 0; inner_area = b.area(); }

  if( abs( b.height() - wp.max() ) <= 2 ||
      similar( b.height(), wp.max(), 20 ) ) return '.';
  if( inner_area * 100 < inner_space * 85 ) return 0;
  if( b.width() >= 6 * b.height() &&
      b.top() - b.height() > charbox_vcenter ) return '_';
  if( b.width() > b.height() ) return '-';
  if( b.height() > 9 && b.width() <= 8 && b.v_includes( charbox_vcenter ) )
    {
    int l = 0, r = 0;
    for( int row = b.vcenter() - 1; row <= b.vcenter() + 1; ++row )
      { if( b.id( row, b.left() ) == b.id() ) ++l;
      if( b.id( row, b.right() ) == b.id() ) ++r; }
    if( l <= 1 && r <= 1 )
      {
      bool lt = false, lb = false, rt = false, rb = false;
      for( int row = b.top(); row <= b.top() + 2; ++row )
        { if( b.id( row, b.left() ) == b.id() ) lt = true;
        if( b.id( row, b.right() ) == b.id() ) rt = true; }
      for( int row = b.bottom() - 2; row <= b.bottom(); ++row )
        { if( b.id( row, b.left() ) == b.id() ) lb = true;
        if( b.id( row, b.right() ) == b.id() ) rb = true; }
      if( lt && lb && rb ) { if( rt ) return 'I'; else return 'l'; }
      }
    }
  if( b.height() > b.width() )
    {
//    if( b.height() >= 5 * b.width() ) return '|';
    if( b.top() > charbox_vcenter ) return ',';
    if( b.bottom() <= charbox_vcenter ) return '\'';
    return '|';
    }
  return 0;
  }


unsigned char Features::test_misc() const throw()
  {
  const Block & b = *_block;

  if( hbars() == 1 && hbar(0).top() <= b.top() + (b.height() / 10) &&
      5 * hbar(0).width() >= 4 * b.width() &&
      rp.increasing( hbar(0).vcenter() - b.top() ) &&
      rp[hbar(0).bottom()-b.top()+2] - rp[hbar(0).bottom()-b.top()] < b.width() / 4 )
    return '7';

  if( b.height() > b.width() && rp.increasing( 1 ) &&
      b.seek_left( b.vcenter(), b.hcenter() ) <= b.left() )
    return '7';

  if( tp.minima( b.height() / 4 ) == 1 && bp.minima( b.height() / 4 ) == 1 )
    {
    if( b.height() > 2 * wp.max() )
      {
      if( hbars() == 1 && hbar(0).bottom() >= b.bottom() - 1 ) return 'l';
      if( hbars() == 2 && hbar(0).bottom() < b.vpos( 25 ) &&
          hbar(1).bottom() >= b.bottom() - 1 )
        {
        if( 3 * hbar(0).width() <= 2 * hbar(1).width() ) return 'l';
        else return 'I';
        }
      }
    if( 3 * wp[wp.pos(50)] < b.width() )
      {
      if( hbars() == 1 && hbar(0).top() <= b.top() + 1 ) return 'T';
      if( hbars() == 2 && hbar(0).top() <= b.top() + 1 &&
          hbar(1).bottom() >= b.bottom() - 1 &&
          3 * hbar(0).width() > 4 * hbar(1).width() ) return 'T';
      }
    if( ( hbars() == 2 || hbars() == 3 ) && hbar(0).top() <= b.top() + 1 &&
        hbar(1).includes_vcenter( b ) &&
        3 * hbar(0).width() > 4 * hbar(1).width() &&
        ( hbars() == 2 ||
          ( hbar(2).bottom() >= b.bottom() - 1 &&
            3 * hbar(0).width() > 4 * hbar(2).width() ) ) ) return 'F';
    }

  if( b.height() > 2 * wp.max() )
    {
    if( rp.istip() && lp.ispit() ) return '{';
    if( lp.istip() && rp.ispit() ) return '}';
    }

  return 0;
  }
