#!/usr/bin/perl -w
#
# Copyright (C) 2003 Dmitry Fedorov <fedorov@inp.nsk.su>
#
# This file is part of Offmirror.
#
# Offmirror is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# Offmirror is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with Offmirror; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA


=head1 NAME

offmirror-Sources - convert Debian 'Sources' files to offmirror list

=head1 SYNOPSIS

offmirrror-Sources file ... | -

=head1 DESCRIPTION

This program parses Debian 'Sources' files and output to stdout
in offmirrror list form.

=head1 OPTIONS

=over 4

=item file ...

List of 'Sources' files to convert or '-' for stdin

=back

=head1 SEE ALSO

L<offmirror(1)>

L<offmirror-Packages(1)>

=cut


require 5.004;
use strict;

use FindBin;
use lib "$FindBin::Bin";
use OffMirror::FileAttrRecord;


my $progname = 'offmirror-Sources';

sub usage {
	warn "\n".join(" ", @_)."\n" if @_;
	warn <<EOF;

Usage: $progname file ... | -

For details, see man page.
EOF
	exit(1);
}


usage("no Sources files specified") if scalar(@ARGV) == 0;

my @source_files = @ARGV;

local $/="\n\n";

my ($directory, $filename, $size, $md5sum);

foreach my $file (@source_files)
{
    local *IN;

    if ($file ne '-')
    {
	$file =~ s/^(.*\.(gz|z|Z|zip))$/gzip -dc < $1|/
	    or
	$file =~ s/^(.*\.(bz2|bz))$/bzip2 -dc < $1|/
	    or
	$file =~ s/^(.*)$/< $1/;

	open IN, "$file"
	    or die "Can't open file $file: $!";
    }
    else
    {
	open IN, "<&STDIN"
	    or die die "Can't redirect stdin: $!";
    }

    while (<IN>)
    {
	($directory) = m/^Directory:\s+(.*)/im;
	die "error parsing $file"
	    unless $directory;

	while (m/^ ([A-Za-z0-9]{32} .*)/mg)
	{
	    ($md5sum, $size, $filename)=split(' ', $1, 3);
	    die "error parsing $file"
		unless $filename and $size and $md5sum;
	    $filename="$directory/$filename";

	    my $r = OffMirror::FileAttrRecord->make_record_from(
					$size,
					$md5sum,
					$filename);

	    print ( $r->make_record_line($filename, 'l', 1) . "\n" );
	}
    }
}

