/***************************************************************************
 *   Copyright (C) 2004 by Rick L. Vinyard, Jr.                            *
 *   rvinyard@cs.nmsu.edu                                                  *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU Lesser General Public License as        *
 *   published by the Free Software Foundation version 2.1.                *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU Lesser General Public      *
 *   License along with this library; if not, write to the                 *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA              *
 ***************************************************************************/
#include "simple.h"
#include <iostream>

#define ANIMATE_ROTATE_STEP 0.05
#define ANIMATE_FRAME_DELAY 40

#define ROTATE_MAX 6.28         /* 2*pi */

#define MARGIN 20
#define INITIAL_SIZE 200

Simple::Simple()
{
  set_title("Papyrus Group Transformation Demo");

  m_vbox.set_border_width(12);
  m_frame.set_shadow_type(Gtk::SHADOW_IN);

  m_slider.set_range(0, ROTATE_MAX);
  m_slider.set_increments(0.05, 0.05);
  m_zoom.set_range(0.1, 4.0);
  m_zoom.set_increments(0.1, 0.1);
  m_zoom.set_value(1.0);

  m_canvas.set_size_request(INITIAL_SIZE, INITIAL_SIZE);

  adjust_canvas(INITIAL_SIZE, INITIAL_SIZE);

  // This is the method that demonstrates the Papyrus objects
  draw_scene();

  m_frame.add(m_canvas);
  m_vbox.pack_start(m_frame);

  m_slider.set_draw_value(false);
  m_slider.signal_value_changed().connect(sigc::mem_fun(*this, &Simple::slider_changed));
  m_vbox.pack_start(*Gtk::manage(new Gtk::Label("Rotate")), Gtk::PACK_SHRINK);
  m_vbox.pack_start(m_slider, Gtk::PACK_SHRINK);

  m_animate.set_label("Animate Rotation");
  m_animate.signal_toggled().connect(sigc::mem_fun(*this, &Simple::animate_toggled));
  m_vbox.pack_start(*Gtk::manage(new Gtk::HSeparator()), Gtk::PACK_SHRINK);
  m_vbox.pack_start(m_animate, Gtk::PACK_SHRINK);

  m_zoom.set_draw_value(false);
  m_zoom.signal_value_changed().connect(sigc::mem_fun(*this, &Simple::zoom_changed));
  m_vbox.pack_start(*Gtk::manage(new Gtk::HSeparator()), Gtk::PACK_SHRINK);
  m_vbox.pack_start(*Gtk::manage(new Gtk::Label("Zoom")), Gtk::PACK_SHRINK);
  m_vbox.pack_start(m_zoom, Gtk::PACK_SHRINK);

  signal_size_allocate().connect(sigc::mem_fun(*this, &Simple::on_sig_size_allocate));

  add(m_vbox);
  m_vbox.show_all();
  show();
}

Simple::~Simple()
{
}

void Simple::draw_scene( )
{
  // set an initial rotation of the box to the value of the slider
  m_rect.rotate(m_slider.get_value());

  // set the box color to red
  m_rect.set_fill(1.0, 0.0, 0.0, 0.8);

  m_circle.set_fill(0.0, 0.0, 1.0);
  m_circle.set_outline(0.0, 1.0, 0.0);

  m_text.set_text("Text");
  m_text.set_size(40);
  m_text.set_fill(1.0, 1.0, 0.0);

  m_png.set_filename("romedalen.png");

  m_lines.add_vertex(0, 0);
  m_lines.add_vertex(40, 0);
  m_lines.add_vertex(0, 40);
  m_lines.add_vertex(40, 40);
  m_lines.set_fill(0.0, 1.0, 1.0);
  m_lines.set_outline(1.0, 0.0, 0.0);

  m_polygon.add_vertex(50, 50);
  m_polygon.add_vertex(90, 50);
  m_polygon.add_vertex(50, 90);
  m_polygon.add_vertex(90, 90);
  m_polygon.set_fill(1.0, 0.0, 1.0);
  m_polygon.set_outline(0.0, 1.0, 0.0);

  // add the box to the root of the canvas
  m_canvas.root().add(m_text);
  m_canvas.root().add(m_lines);
  m_canvas.root().add(m_rect);
  m_canvas.root().add(m_circle);
  m_canvas.root().add(m_polygon);
  m_canvas.root().add(m_png);
}

void Simple::slider_changed()
{
  m_rect.set_rotate(m_slider.get_value());
}

void Simple::zoom_changed()
{
  m_canvas.root().set_scale(m_zoom.get_value());
}


bool
Simple::animate_step ()
{
  double    newval = m_slider.get_value() + ANIMATE_ROTATE_STEP;
  if (newval > ROTATE_MAX)
    newval -= ROTATE_MAX;
  m_slider.set_value(newval);
  return true;
}

void
Simple::animate_toggled()
{
  static sigc::connection timer;
  bool  active = m_animate.get_active();
  if (active && !timer.connected())
      timer = Glib::signal_timeout().connect(sigc::mem_fun(*this, &Simple::animate_step), ANIMATE_FRAME_DELAY);
  else if (!active && timer.connected())
      timer.disconnect();
}

void Simple::on_sig_size_allocate(Gtk::Allocation& a) {
  adjust_canvas(m_canvas.get_width(), m_canvas.get_height());
}

void Simple::adjust_canvas(int width, int height) {
  int box_size = (width + height) / 6;

  // set the canvas origin to the center of the screen
  m_canvas.root().set_xy(width/2, height/2);

  // adjust the box size
  m_rect.set_xy(-box_size, -box_size);
  m_rect.set_width_height(box_size, box_size);

  m_circle.set_radius(box_size/2);
  m_circle.set_xy(-box_size/2, 0);
}

