/***************************************************************************
 *   Copyright (C) 2004 by Rick L. Vinyard, Jr.                            *
 *   rvinyard@cs.nmsu.edu                                                  *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU Lesser General Public License as        *
 *   published by the Free Software Foundation version 2.1.                *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU Lesser General Public      *
 *   License along with this library; if not, write to the                 *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA              *
 ***************************************************************************/
#include "bbox.h"

#include <algorithm>

using namespace Papyrus;

BBox::BBox(double x, double y, double w, double h):
    m_x(x),
    m_y(y),
    m_w(w),
    m_h(h)
{
}


BBox::~BBox() {}

    double BBox::get_x() { return m_x; }
    double BBox::get_y() { return m_y; }
    double BBox::get_width() { return m_w; }
    double BBox::get_height() { return m_h; }

    double BBox::get_xywh(double& x, double& y, double& w, double& h) {
      x = m_x;
      y = m_y;
      w = m_w;
      h = m_h;
    }

    void BBox::set_x(double x) {
      m_x = x;
      on_changed(BBOX_X);
    }

    void BBox::set_y(double y) {
      m_y = y;
      on_changed(BBOX_Y);
    }

    void BBox::set_width(double width) {
      m_w = width;
      on_changed(BBOX_WIDTH);
    }

    void BBox::set_height(double height) {
      m_h = height;
      on_changed(BBOX_HEIGHT);
    }

    void BBox::set_xywh(double x, double y, double w, double h) {
      m_x = x;
      m_y = y;
      m_w = w;
      m_h = h;
      on_changed(BBOX_X|BBOX_Y|BBOX_WIDTH|BBOX_HEIGHT);
    }

BBox& BBox::operator=(const BBox& other) {
  if (*this == other)
    return *this;
  m_x = other.m_x;
  m_y = other.m_y;
  m_w = other.m_w;
  m_h = other.m_h;
  on_changed(BBOX_X|BBOX_Y|BBOX_WIDTH|BBOX_HEIGHT);
  return *this;
}

bool BBox::operator==(const BBox& other) const {
  return (m_x == other.m_x && m_y == other.m_y && m_w == other.m_w && m_h == other.m_h);
}

bool BBox::operator<(const BBox& other) const {
  return (
      m_x < other.m_x &&
      m_y < other.m_y &&
      m_w > other.m_w &&
      m_h > other.m_h
         );
}

bool BBox::operator<=(const BBox& other) const {
  return (
      m_x <= other.m_x &&
      m_y <= other.m_y &&
      m_w >= other.m_w &&
      m_h >= other.m_h
         );
}

bool BBox::operator>(const BBox& other) const {
  return (! (*this <= other ));
}

bool BBox::operator>=(const BBox& other) const {
  return (! (*this < other ));
}

int BBox::update(const BBox& other) {
  int changed = 0;
    if (other.m_x < m_x) {
      m_w = m_x + m_w - other.m_x;
      m_x = other.m_x;
      changed |= BBOX_X | BBOX_WIDTH;
    }
    if (other.m_y < m_y) {
      m_h = m_y + m_h - other.m_y;
      m_y = other.m_y;
      changed |= BBOX_Y | BBOX_HEIGHT;
    }
    if (other.m_x + other.m_w > m_x + m_w) {
      m_w = other.m_x + other.m_w - m_x;
      changed |= BBOX_WIDTH;
    }
    if (other.m_y + other.m_h > m_y + m_h) {
      m_h = other.m_y + other.m_h - m_y;
      changed |= BBOX_HEIGHT;
    }

    if (changed)
      on_changed(changed);

    return changed;
}

int BBox::update(double x, double y) {
  int changed = 0;
  if (x < m_x) {
    m_w += m_x - x;
    m_x = x;
    changed |= BBOX_X | BBOX_WIDTH;
  }
  else if (x > m_x + m_w) {
    m_w = x - m_x;
    changed |= BBOX_X;
  }

  if (y < m_y) {
    m_h += m_y - y;
    m_y = y;
    changed |= BBOX_Y | BBOX_HEIGHT;
  }
  else if (y > m_y + m_h) {
    m_h = y - m_y;
    changed |= BBOX_Y;
  }

  if (changed)
    on_changed(changed);

  return changed;
}

int BBox::update_x(double x) {
    int changed = 0;
    if (x < m_x) {
      m_w += m_x - x;
      m_x = x;
      changed = BBOX_X | BBOX_WIDTH;
    }
    else if (x > m_x + m_w) {
      m_w = x - m_x;
      changed = BBOX_X;
    }

    if (changed)
      on_changed(changed);

    return changed;
}

int BBox::update_y(double y) {
    int changed = 0;
    if (y < m_y) {
      m_h += m_y - y;
      m_y = y;
      changed = BBOX_Y|BBOX_HEIGHT;
    }
    else if (y > m_y + m_h) {
      m_h = y - m_y;
      changed = BBOX_HEIGHT;
    }

    if (changed)
      on_changed(changed);

    return changed;
}

bool BBox::bounded( double x, double y ) {
  return (
      x >= m_x        &&
      x <= m_x + m_w  &&
      y >= m_y        &&
      y <= m_y + m_h
         );
}

bool BBox::inside( double x, double y ) {
  return (
      x > m_x        &&
      x < m_x + m_w  &&
      y > m_y        &&
      y < m_y + m_h
         );
}

bool BBox::boundary( double x, double y ) {
  return (
      x == m_x        ||
      x == m_x + m_w  ||
      y == m_y        ||
      y == m_y + m_h
         );
}

void BBox::transform( const cairo_matrix_t& matrix )
{
  double tx, ty;
  double x[4] = {m_x, m_x,     m_x+m_w, m_x+m_w};
  double y[4] = {m_y, m_y+m_h, m_y,     m_y+m_h};
  double xmin, xmax, ymin, ymax;

  for (int i=0; i < 4; i++)
    cairo_matrix_transform_point(&matrix, x+i, y+i);

  xmin = xmax = x[0];
  ymin = ymax = y[0];

  for (int i=1; i < 4; i++) {
    xmin = std::min(xmin, x[i]);
    xmax = std::max(xmax, x[i]);
    ymin = std::min(ymin, y[i]);
    ymax = std::max(ymax, y[i]);
  }

  m_x = xmin;
  m_y = ymin;
  m_w = xmax - xmin;
  m_h = ymax - ymin;

  on_changed(BBOX_X|BBOX_Y|BBOX_WIDTH|BBOX_HEIGHT);
}

void BBox::on_changed(int i) {
  m_signal_changed.emit(this, i);
}

double BBox::get_left( )
{
  return m_x;
}

double Papyrus::BBox::get_right( )
{
  return m_x + m_w;
}

double Papyrus::BBox::get_top( )
{
  return m_y;
}

double Papyrus::BBox::get_bottom( )
{
  return m_y + m_h;
}

void Papyrus::BBox::get_lrtb( double & l, double & r, double & t, double & b )
{
  l = m_x;
  r = m_x + m_w;
  t = m_y;
  b = m_y + m_h;
}
