/***************************************************************************
 *   Copyright (C) 2004 by Rick L. Vinyard, Jr.                            *
 *   rvinyard@cs.nmsu.edu                                                  *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU Lesser General Public License as        *
 *   published by the Free Software Foundation version 2.1.                *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU Lesser General Public      *
 *   License along with this library; if not, write to the                 *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA              *
 ***************************************************************************/
#include "group.h"

#include <assert.h>

#include <iostream>
#include <algorithm>

using namespace Papyrus;

Group::Group():
m_parent(NULL) {}

Group::Group(Group& parent):
m_parent(&parent) {}

Group::~Group() {}

Group* Group::get_parent() {
    return m_parent;
}

bool Group::set_parent(Group& parent) {
    m_parent = &parent;
    on_changed(RENDER_BBOX);
}


bool Group::add
    (Drawable& item) {
    m_children.push_back(&item);
    m_connections[&item] = item.signal_need_redraw().connect(sigc::mem_fun(*this, &Group::on_child_need_redraw));
    m_child_connections[&item] = item.signal_changed().connect(sigc::mem_fun(*this, &Group::on_child_changed));
    m_bbox.update(item.get_bbox_transformed());
    on_changed(RENDER_BBOX);
}

bool Group::remove
    (Drawable& item) {
    Connections::iterator i;
    m_children.remove(&item);
    i = m_connections.find(&item);
    if (i != m_connections.end()) {
        m_connections[&item].disconnect();
        m_connections.erase(&item);
        m_child_connections[&item].disconnect();
        m_child_connections.erase(&item);
        rebuild_bbox();
        on_changed(RENDER_BBOX);
    }
}

bool Group::clear() {
    m_children.clear();
}

void Group::draw(cairo_t* cairo) {
    Children::reverse_iterator i;
    for (i=m_children.rbegin(); i != m_children.rend(); i++)
        (*i)->render(cairo);
}

void Group::on_child_need_redraw(double x, double y, double w, double h) {
    BBox region(x, y, w, h);
    recalculate_matrix();
    region.transform(m_matrix);
    region.get_xywh(x, y, w, h);
    m_signal_need_redraw.emit(x, y, w, h);
}

bool Group::raise(Drawable& item) {
    Children::iterator i, j;
    j = std::find(m_children.begin(), m_children.end(), &item);
    if (j != m_children.end() && j != m_children.begin()) {
        i = j;
        i--;
        m_children.erase(j);
        m_children.insert(i, &item);
        item.on_changed(RENDER_BBOX);
    }
}

bool Group::raise_to_top(Drawable& item) {
    Children::iterator i;
    i = std::find(m_children.begin(), m_children.end(), &item);
    if (i != m_children.end() && i != m_children.begin()) {
        m_children.erase(i);
        m_children.push_front(&item);
        item.on_changed(RENDER_BBOX);
    }
}

bool Group::lower(Drawable& item) {
    Children::iterator i, j;
    i = std::find(m_children.begin(), m_children.end(), &item);
    if (i != m_children.end() && (&item != m_children.back()) ) {
        j = i;
        j++;
        j++;
        m_children.erase(i);
        m_children.insert(j, &item);
        item.on_changed(RENDER_BBOX);
    }
}

bool Group::lower_to_bottom(Drawable& item) {
    Children::iterator i;
    i = std::find(m_children.begin(), m_children.end(), &item);
    if (i != m_children.end() && (&item != m_children.back()) ) {
        m_children.erase(i);
        m_children.push_back(&item);
        item.on_changed(RENDER_BBOX);
    }
}

Drawable * Group::pick( double x, double y ) {
    Drawable* result;

    // it's not worth checking the children individually
    // if the pick point isn't even in the group bbox
    if (intersects(x, y)) {

        // if the group is pickable, then return the group
        if (m_pickable)
            return this;

        // otherwise transform the points according to the group
        // affine matrix and check each of the children
        cairo_matrix_t inverse = get_matrix();
        cairo_matrix_invert(&inverse);
        cairo_matrix_transform_point(&inverse, &x, &y);
        for (Children::iterator i = m_children.begin(); i != m_children.end(); i++) {

            result = (*i)->pick(x, y);
            if (result != NULL)
                return result;
        }
    }
    return NULL;
}

void Group::rebuild_bbox()
{
  m_bbox.set_xywh(0.0, 0.0, 0.0, 0.0);
  if (m_children.size() > 0) {
    m_bbox = m_children.front()->get_bbox_transformed();
    for (Children::iterator i = m_children.begin(); i != m_children.end(); i++)
      m_bbox.update((*i)->get_bbox_transformed());
  }
}

void Papyrus::Group::on_child_changed( Object * child, int changed )
{
  if (changed & RENDER_BBOX) {
    rebuild_bbox();
    on_changed(RENDER_BBOX);
  }
}

