/***************************************************************************
 *   Copyright (C) 2004 by Rick L. Vinyard, Jr.                            *
 *   rvinyard@cs.nmsu.edu                                                  *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU Lesser General Public License as        *
 *   published by the Free Software Foundation version 2.1.                *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU Lesser General Public      *
 *   License along with this library; if not, write to the                 *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA              *
 ***************************************************************************/
#ifndef PAPYRUS_INTERFACE_VERTICES_H
#define PAPYRUS_INTERFACE_VERTICES_H

#include <sigc++/sigc++.h>
#include <vector>

namespace Papyrus {

class Vertex: public virtual sigc::trackable {
  public:
    Vertex(double vx=0.0, double vy=0.0): x(vx), y(vy) {
    }

    virtual ~Vertex() { }

  double x, y;
};

//typedef Interface::XY Vertex;
typedef std::vector<Vertex> Vertices;

namespace Interface {

/**
@author Rick L Vinyard Jr
*/
class Verts: public virtual sigc::trackable {
public:
Verts(Vertices vertices=Vertices()): m_vertices(vertices) {
        set_vertices(vertices);
    }

    virtual ~Verts() { }

    Vertex get_vertex(unsigned n) {
      if (n < m_vertices.size())
            return m_vertices[n];
        else
            return Vertex();
    }

    const Vertices& get_vertices() {
        return m_vertices;
    }

  void add_vertex(Vertex v) {
    m_vertices.push_back(v);
    interface_on_vertex_changed(m_vertices.size()-1);
    interface_on_vertices_changed();
  }

  void add_vertex(double x, double y) {
    m_vertices.push_back(Vertex(x,y));
    interface_on_vertex_changed(m_vertices.size()-1);
    interface_on_vertices_changed();
  }

  void set_vertex(unsigned n, Vertex v) {
      if (n < m_vertices.size()) {
            m_vertices[n] = v;
          interface_on_vertex_changed(n);
          interface_on_vertices_changed();
        }
    }

  void set_vertex(unsigned n, double x, double y) {
    if (n < m_vertices.size()) {
      m_vertices[n] = Vertex(x, y);
      interface_on_vertex_changed(n);
      interface_on_vertices_changed();
    }
  }

  void set_vertices(Vertices vertices) {
      m_vertices = vertices;
      interface_on_vertices_changed();
    }

    sigc::signal<void, unsigned>& signal_vertex_changed() {
        return m_signal_vertex_changed;
    }

    sigc::signal<void>& signal_vertices_changed() {
        return m_signal_vertices_changed;
    }

protected:
    Vertices m_vertices;
    sigc::signal<void, unsigned> m_signal_vertex_changed;
    sigc::signal<void> m_signal_vertices_changed;

    virtual void on_vertex_changed(int n) { }
    virtual void on_vertices_changed() { }

    virtual void interface_on_vertex_changed(int n) {
        m_signal_vertex_changed.emit(n);
        m_signal_vertices_changed.emit();
        on_vertex_changed(n);
        on_vertices_changed();
    }

    virtual void interface_on_vertices_changed() {
        m_signal_vertices_changed.emit();
        on_vertices_changed();
    }

};

}

}

#endif
