/***************************************************************************
 *   Copyright (C) 2004 by Rick L. Vinyard, Jr.                            *
 *   rvinyard@cs.nmsu.edu                                                  *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU Lesser General Public License as        *
 *   published by the Free Software Foundation version 2.1.                *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU Lesser General Public      *
 *   License along with this library; if not, write to the                 *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA              *
 ***************************************************************************/
#include "rgba.h"

#include <math.h>

using namespace Papyrus;

RGBA::RGBA( ) {
    memset(m_rgba, 0x00, sizeof(double)*3);
    m_rgba[3] = 1.0;
}

RGBA::RGBA(const RGBA* other ) {
    if (other)
        memcpy(m_rgba, other->m_rgba, sizeof(double)*4);
    else {
        memset(m_rgba, 0x00, sizeof(double)*3);
        m_rgba[3] = 1.0;
    }
}

RGBA::RGBA(int r, int g, int b, int a) {
    m_rgba[0] = static_cast<double>(r/255.0);
    m_rgba[1] = static_cast<double>(g/255.0);
    m_rgba[2] = static_cast<double>(b/255.0);
    m_rgba[3] = static_cast<double>(a/255.0);
}

RGBA::RGBA(double r, double g, double b, double a) {
    m_rgba[0] = r;
    m_rgba[1] = g;
    m_rgba[2] = b;
    m_rgba[3] = a;
}

RGBA::RGBA(double rgba[4]) {
    memcpy(m_rgba, rgba, sizeof(double)*4);
}

void RGBA::get_rgba(double rgba[4]) const {
    memcpy(rgba, m_rgba, sizeof(double)*4);
}

void RGBA::get_rgba(float rgba[4]) const {
    for (int i=0; i<4; i++)
        rgba[i] = static_cast<float>(m_rgba[i]);
}

void RGBA::get_rgba(double& r, double& g, double& b, double& a) const {
    r = m_rgba[0];
    g = m_rgba[1];
    b = m_rgba[2];
    a = m_rgba[3];
}

void RGBA::get_rgba(float& r, float& g, float& b, float& a) const {
    r = static_cast<float>(m_rgba[0]);
    g = static_cast<float>(m_rgba[1]);
    b = static_cast<float>(m_rgba[2]);
    a = static_cast<float>(m_rgba[3]);
}

void RGBA::get_hls(double& h, double& l, double& s) {
    double rgb[3];
    memcpy(rgb, m_rgba, sizeof(double)*3);
    rgb_to_hls(rgb[0], rgb[1], rgb[2], h, l, s);
}

void RGBA::get_hsv(double& h, double& s, double& v) {
    double rgb[3];
    memcpy(rgb, m_rgba, sizeof(double)*3);
    rgb_to_hsv(rgb[0], rgb[1], rgb[2], h, s, v);
}

void RGBA::set_rgba(float rgba[4]) {
    for (int i=0; i<4; i++)
        m_rgba[i] = rgba[i];
    m_signal_changed.emit();
}

void RGBA::set_rgba(double r, double g, double b, double a) {
    m_rgba[0] = r;
    m_rgba[1] = g;
    m_rgba[2] = b;
    m_rgba[3] = a;
    m_signal_changed.emit();
}

void RGBA::set_hlsa(double h, double l, double s, double a) {
    double r, g, b;
    hls_to_rgb(h, l, s, r, g, b);
    m_rgba[0] = r;
    m_rgba[1] = g;
    m_rgba[2] = b;
    m_rgba[3] = a;
    m_signal_changed.emit();
}

void RGBA::set_hsva(double h, double s, double v, double a) {
    double r, g, b;
    hsv_to_rgb(h, s, v, r, g, b);
    m_rgba[0] = r;
    m_rgba[1] = g;
    m_rgba[2] = b;
    m_rgba[3] = a;
    m_signal_changed.emit();
}

double RGBA::operator[](int index) {
    if (index < 0 || index > 3)
        index = 0;
    return m_rgba[index];
}

bool RGBA::operator==(const RGBA other) const {
    if (memcmp(m_rgba, other.m_rgba, sizeof(double)*4) == 0)
        return true;
    else
        return false;
}

  bool RGBA::operator!=(const RGBA other) const { return !(*this == other); }

bool RGBA::operator<(const RGBA other) const {
    if (memcmp(m_rgba, other.m_rgba, sizeof(double)*4) < 0)
        return true;
    else
        return false;
}

    RGBA& RGBA::operator=(const RGBA& other) {
      for (int i=0; i<4; i++)
        m_rgba[i] = other.m_rgba[i];
      m_signal_changed.emit();
      return *this;
    }


// From: Computer Graphics: Principles and Practice
void RGBA::rgb_to_hls(double r, double g, double b,
                      double& h, double& l, double& s) {
    // Given r,g,b, each in [0,1]
    // Desired: h in [0,360]; l and s in [0,1], except if s=0, h=UNDEF
    double max;
    double min;

    max = std::max(r, g);
    max = std::max(max, b);
    min = std::min(r, g);
    min = std::min(min, b);

    l = (max+min)/2.0;
    if (max == min) {
        s = 0;
        h = 0; // h is undefined
    } else {
        double delta = max - min;
        s = (l <= 0.5)?(delta/(max+min)):(delta/(2.0-(max+min)));
        if (r==max)
            h = (g-b)/delta;
        else if (g==max)
            h = 2.0 + (b-r)/delta;
        else if (b==max)
            h = 4.0+(r-g)/delta;
        h *= 60.0;
        if (h < 0.0)
            h += 360.0;
    }
}

// From: Computer Graphics: Principles and Practice
double RGBA::Value(double n1, double n2, double hue) {
    if (hue > 360.0)
        hue -= 360.0;
    else if (hue < 0.0)
        hue += 360.0;
    if (hue < 60.0)
        return n1 + (n2-n1)*hue/60.0;
    else if (hue < 180.0)
        return n2;
    else if (hue < 240.0)
        return n1 + (n2-n1)*(240.0-hue)/60.0;
    else
        return n1;
}

// From: Computer Graphics: Principles and Practice
void RGBA::hls_to_rgb(double h, double l, double s,
                      double& r, double& g, double& b) {
    double m1, m2;
    m2 = (l <= 0.5)?(l*(l+s)):(l+s-l*s);
    m1 = 2.0 * l - m2;
    if (s == 0.0) {
        if (h == 0)
            r = g = b = l;
    } else {
        r = Value(m1, m2, h+120.0);
        g = Value(m1, m2, h);
        b = Value(m1, m2, h-120.0);
    }
}






// From: Computer Graphics: Principles and Practice
void RGBA::rgb_to_hsv(double r, double g, double b,
                      double& h, double& s, double& v) {
    // Given r,g,b, each in [0,1]
    // Desired: h in [0,360]; s and v in [0,1], except if s=0, h=UNDEF
    double max;
    double min;

    max = std::max(r, g);
    max = std::max(max, b);
    min = std::min(r, g);
    min = std::min(min, b);

    v = max;

    s = (max != 0.0)?((max-min)/max):0.0;

    if (s == 0)
        h = 0;
    else {
        double delta = max - min;
        if (r == max)
            h = (g-b)/delta;
        else if (g == max)
            h = 2.0 + (b-r)/delta;
        else if (b == max)
            h = 4.0 + (r-g)/delta;
        h *= 60.0;
        if (h < 0.0)
            h += 360.0;
    }
}

// From: Computer Graphics: Principles and Practice
void RGBA::hsv_to_rgb(double h, double s, double v,
                      double& r, double& g, double& b) {
    if (s == 0.0)
        r = g = b = v;
    else {
        double f, p, q, t;
        int i;

        if (h == 360.0)
            h = 0.0;
        h /= 60.0;
        i = (int) floor(h);
        f = h - i;
        p = v * (1.0 - s);
        q = v * (1.0 - (s*f));
        t = v * (1.0 - (s*(1.0-f)));
        switch(i) {
        case 0:
            r = v;
            g = t;
            b = p;
            break;
        case 1:
            r = q;
            g = v;
            b = p;
            break;
        case 2:
            r = p;
            g = v;
            b = t;
            break;
        case 3:
            r = p;
            g = q;
            b = v;
            break;
        case 4:
            r = t;
            g = p;
            b = v;
            break;
        case 5:
            r = v;
            g = p;
            b = q;
            break;
        }
    }
}

std::ostream& operator<<(std::ostream& out, RGBA color) {
  double r, g, b, a;
  color.get_rgba(r, g, b, a);
  return out << r << " " << g << " " << b << " " << a;
}
