/***************************************************************************
 *   Copyright (C) 2004 by Rick L. Vinyard, Jr.                            *
 *   rvinyard@cs.nmsu.edu                                                  *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU Lesser General Public License as        *
 *   published by the Free Software Foundation version 2.1.                *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU Lesser General Public      *
 *   License along with this library; if not, write to the                 *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA              *
 ***************************************************************************/
#include "textbox.h"

using namespace Papyrus;

Textbox::Textbox(double x, double y, double w, double h):
        Drawable(),
        m_hjustification(HJUSTIFY_FULL),
m_vjustification(VJUSTIFY_TOP) { }

Textbox::~Textbox() { }

const std::string& Textbox::get_text() {
    return m_text;
}

void Textbox::set_text(const std::string text) {
    m_text = text;
    on_changed(RENDER_WIDTH|RENDER_HEIGHT|RENDER_BBOX);
}

void Textbox::draw_item(cairo_t* cairo) {
    cairo_text_extents_t extents;
    double textw, texth;
    double xoffset=0.0, yoffset=0.0, scale;

    cairo_set_font_size(cairo, 20.0);
    cairo_text_extents(cairo, m_text.c_str(), &extents);
    textw = extents.width;
    texth = extents.height;

    if (HJUSTIFY_FULL)
      scale = m_width/textw;
    if (VJUSTIFY_FULL && scale < m_height/texth)
      scale = m_height/texth;
    if (textw*scale > m_width)
        scale = m_width/textw;
    if (texth*scale > m_height)
      scale = m_height/texth;

    switch (m_hjustification) {
    case HJUSTIFY_RIGHT:
        xoffset = m_width - textw * scale;
        break;
    case HJUSTIFY_CENTER:
      case HJUSTIFY_FULL:
        xoffset = (m_width - textw * scale) / 2.0;
        break;
    }

    switch (m_vjustification) {
    case VJUSTIFY_BOTTOM:
        yoffset = m_height - texth * scale;
        break;
    case VJUSTIFY_CENTER:
      case VJUSTIFY_FULL:
        yoffset = (m_height - texth * scale) / 2;
        break;
    }

    cairo_move_to(cairo, m_x+xoffset, m_y+yoffset+texth*scale);
    cairo_scale(cairo, scale, scale);
//     get_color().render(cairo);
    cairo_show_text(cairo, m_text.c_str());
}

void Textbox::on_xywh_changed() {
  on_changed(RENDER_WIDTH|RENDER_HEIGHT|RENDER_BBOX);
}

HJustification Textbox::get_hjustification() {
    return m_hjustification;
}

void Textbox::set_hjustification(HJustification hj) {
    m_hjustification = hj;
    on_changed(RENDER_WIDTH|RENDER_HEIGHT|RENDER_BBOX);
}

VJustification Textbox::get_vjustification() {
    return m_vjustification;
}

void Textbox::set_vjustification(VJustification vj) {
    m_vjustification = vj;
    on_changed(RENDER_WIDTH|RENDER_HEIGHT|RENDER_BBOX);
}

