<?php
/* vim: set expandtab tabstop=4 shiftwidth=4 foldmethod=marker: */
/* @(#) $Header: /sources/phpprintipp/phpprintipp/CupsPrintIPP.php,v 1.6 2006/01/06 18:08:22 harding Exp $
 *
 * Version: 0.1
 * Class PrintIPP - Send extended IPP requests.
 *
 *   Copyright (C) 2005-2006  Thomas HARDING
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Library General Public
 *   License as published by the Free Software Foundation; either
 *   version 2 of the License, or (at your option) any later version.
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *   Library General Public License for more details.
 *
 *   You should have received a copy of the GNU Library General Public
 *   License along with this library; if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 *   mailto:thomas.harding@laposte.net
 *   Thomas Harding, 56 rue de la bourie rouge, 45 000 ORLEANS -- FRANCE
 *   
 */
    
/*

    This class is intended to implement Internet Printing Protocol on client side.

    References needed to debug / add functionnalities:
        - RFC 2910
        - RFC 2911
        - RFC 3995
        - RFC 3996
        - RFC 3997
        - RFC 3998
        - ...
        _ CUPS-IPP-1.1
*/


class CupsPrintIPP extends ExtendedPrintIPP {

    // {{{ variables declaration
    
    public $printers_attributes;
    
    // }}}

    // {{{ constructor
    public function __construct() {
        parent::__construct();
    }
    // }}}
    
    // {{{ getPrinters() [CUPS SPECIFIC]
    public function getPrinters() {
        // THIS WORKS THIS CUPS, AS IT USE A VENDOR EXTENSION
    
        $this->jobs = array_merge($this->jobs,array(""));
        $this->jobs_uri = array_merge($this->jobs_uri,array(""));
        
        unset ($this->printers_attributes);
        
        if (!isset($this->setup->charset))
            self::setCharset('us-ascii');
        
        if (!isset($this->setup->language))
            self::setLanguage('en');
            
        self::_setOperationId();

        $this->stringjob = chr(0x01) . chr(0x01) // IPP version 1.1 
                         . chr(0x40). chr(0x02) // operation:  cups vendor extension: get printers
                         . $this->meta->operation_id //           request-id
                         . chr(0x01) // start operation-attributes | operation-attributes-tag
                         . $this->meta->charset
                         . $this->meta->language
                         . chr(0x03); // end operations attribute
                        
        $this->output = $this->stringjob;
           
        $post_values = array( "Content-Type" => "application/ipp",
                              "Data" => $this->output);
            
        if (self::_sendHttp ($post_values,'/')) {
            
            if(self::_parseServerOutput())
                $this->_getAvailablePrinters();

            }
        
        if (isset($this->serveroutput) && isset($this->serveroutput->status)) {
            
            $this->status = array_merge($this->status,array($this->serveroutput->status));
            if ($this->serveroutput->status == "successfull-ok")
                self::_errorLog("getting printers: ".$this->serveroutput->status,3);
            else
                self::_errorLog("getting printers: ".$this->serveroutput->status,1);
            return $this->serveroutput->status; 
            
        } else {
            $this->status = array_merge($this->status,array("OPERATION FAILED"));
            self::_errorLog("getting printers : OPERATION FAILED",1);
            }
    return false;   
    }
    // }}}
    
    // {{{ getPrinterAttributes()
    public function getPrinterAttributes() {
        
        if(!$result = parent::getPrinterAttributes())
            return FALSE;
        if(!isset($this->printer_attributes))
            return FALSE;
            
        $printer_type = $this->printer_attributes->printer_type->_value0;
        $table = self::_interpretPrinterType($printer_type);
        
        for($i = 0 ; $i < count($table) ; $i++ ) {
            $index = '_value'.$i;
            $this->printer_attributes->printer_type->$index = $table[$i];
            }

    return $result;
    }
    // }}}
    
    // {{{ _getAvailablePrinters ()
    private function _getAvailablePrinters () {
      
        $this->available_printers = array();
        $k = 0;
        
        for ($i = 0 ; (array_key_exists($i,$this->serveroutput->response)) ; $i ++)
            if (($this->serveroutput->response[$i]['attributes']) == "start printer-attributes") {
                for ($j = 0 ; array_key_exists($j,$this->serveroutput->response[$i]) ; $j++) {
                    $value = $this->serveroutput->response[$i][$j]['value'];
                    $phpname = "_printer".$k;
                    switch ($name = $this->serveroutput->response[$i][$j]['name']) {
                        case "printer-uri-supported":
                            $this->available_printers = array_merge($this->available_printers,array($value));
                            $this->printers_attributes->$phpname->printer_uri = $value;
                            $this->printers_uri = array_merge($this->printers_uri,array($value));
                            break;
                        case "printer-name":
                            $this->printers_attributes->$phpname->printer_name =  $value;
                            break;
                        case "printer-location":
                            $this->printers_attributes->$phpname->printer_location =  $value;
                            break;
                        case "printer-type":
                            $this->printers_attributes->$phpname->printer_type =  self::_interpretPrinterType($value);
                            break;
                        case "printer-state":
                            $this->printers_attributes->$phpname->printer_state =  parent::_interpretEnum('printer-state',$value);
                            break;
                        case "printer-info":
                            $this->printers_attributes->$phpname->printer_info =  $value;
                            break;
                        default:
                            break;
                        }
                    }
                $k ++;
                }
        //print_r($this->printers_attributes);
    }
    // }}}
    
    // {{{ _getEnumVendorExtensions
    protected function _getEnumVendorExtensions ($value_parsed) {
        switch ($value_parsed) {
            case 0x4002:
                $value = 'Get-Availables-Printers';
                break;
            default:
                $value = sprintf('Unknown(Cups extension for operations): 0x%x',$value_parsed);
                break;
        }

    if (isset($value))
        return ($value);
  
    return sprintf('Unknown: 0x%x',$value_parsed);
    }
    // }}}

    // {{{ _interpretPrinterType($type)
    private function _interpretPrinterType($value) {
        $value_parsed = 0;
        for ($i = strlen($value) ; $i > 0 ; $i --)
            $value_parsed += pow(256,($i - 1)) * ord($value[strlen($value) - $i]);
        
        $type[0] = $type[1] = $type[2] = $type[3] = $type[4] = $type[5] = '';
        $type[6] = $type[7] = $type[8] = $type[9] = $type[10] = '';
        $type[11] = $type[12] = $type[13] = $type[14] = $type[15] = '';
        $type[16] = $type[17] = $type[18] = $type[19] = '';
        
        if ($value_parsed %2 == 1) {
            $type[0] = 'printer-class';
            $value_parsed -= 1;
            }
        if ($value_parsed %4 == 2 )  {
            $type[1] = 'remote-destination';
            $value_parsed -= 2;
            }
        if ($value_parsed %8 == 4 )  {
            $type[2] = 'print-black';
            $value_parsed -= 4;
            }
        if ($value_parsed %16 == 8 )  {
            $type[3] = 'print-color';
            $value_parsed -= 8;
            }
        if ($value_parsed %32 == 16) {
            $type[4] = 'hardware-print-on-both-sides';
            $value_parsed -= 16;
            }
        if ($value_parsed %64 == 32) {
            $type[5] = 'hardware-staple-output';
            $value_parsed -= 32;
            }
        if ($value_parsed %128 == 64) {
            $type[6] = 'hardware-fast-copies';
            $value_parsed -= 64;
            }
        if ($value_parsed %256 == 128) {
            $type[7] = 'hardware-fast-copy-collation';
            $value_parsed -= 128;
            }
        if ($value_parsed %512 == 256) {
            $type[8] = 'punch-output';
            $value_parsed -= 256;
            }
        if ($value_parsed %1024 == 512) {
            $type[9] = 'cover-output';
            $value_parsed -= 512;
            }
        if ($value_parsed %2048 == 1024) {
            $type[10] = 'bind-output';
            $value_parsed -= 1024;
            }
        if ($value_parsed %4096 == 2048) {
            $type[11] = 'sort-output';
            $value_parsed -= 2048;
            }
        if ($value_parsed %8192 == 4096) {
            $type[12] = 'handle-media-up-to-US-Legal-A4';
            $value_parsed -= 4096;
            }
        if ($value_parsed %16384 == 8192) {
            $type[13] = 'handle-media-between-US-Legal-A4-and-ISO_C-A2';
            $value_parsed -= 8192;
            }
        if ($value_parsed %32768 == 16384) {
            $type[14] = 'handle-media-larger-than-ISO_C-A2';
            $value_parsed -= 16384;
            }
        if ($value_parsed %65536 == 32768) {
            $type[15] = 'handle-user-defined-media-sizes';
            $value_parsed -= 32768;
            }
        if ($value_parsed %131072 == 65536) {
            $type[16] = 'implicit-server-generated-class';
            $value_parsed -= 65536;
            }
        if ($value_parsed %262144 == 131072) {
            $type[17] = 'network-default-printer';
            $value_parsed -= 131072;
            }
        if ($value_parsed %524288 == 262144) {
            $type[18] = 'fax-device';
            $value_parsed -= 262144;
            }
    return $type;   
    }
    // }}}
}

/*
 * Local variables:
 * mode: php
 * tab-width: 4
 * c-basic-offset: 4
 * End:
 */
?>
