/*
 * Copyright (C) 2003 INRIA
 *
 *	INRIA
 *	Domaine de Voluceau
 *	Rocquencourt - B.P. 105
 *	78153 Le Chesnay Cedex - France
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * Author: Loic Dachary <loic@gnu.org>
 * 
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif /* HAVE_CONFIG_H */

#include <unistd.h>
#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include <signal.h>

#include "rfid_error.h"
#include "s6350.h"

static void report_error(rfid_reader_t* reader, const char* message)
{
  if(reader) {
    char* str = reader->strerror_f(reader);
    fprintf(stderr, "%s: %s\n", message, str);
    free(str);
  } else {
    fprintf(stderr, "no RFID reader found\n");
  }
}

static s6350_t* s6350;
static rfid_reader_t* reader;

static RETSIGTYPE alarm_f(int sig)
{
  fprintf(stderr, "\
\n\
FAILURE: Timeout attempting to establish communication with RFID reader at %s.\n\
FAILURE: Make sure a RFID reader is plugged in and turned on.\n\
\n\
", reader->io->device);
  exit(-1);
}

int s6350_reader_alloc(rfid_reader_t** reader);

int main(int argc, char** argv)
{
  int timeout = 5;
  int verbose = 0;
  rfid_io_t io;

  if(s6350_reader_alloc(&reader) < 0)
    return -1;

  s6350 = s6350_from_reader(reader);
  s6350->verbose = verbose;
  reader->io = &io;

  {
    char* device;
    if(argc > 1)
      device = argv[1];
    else
      device = "/dev/ttyS0";
    rfid_io_init(&io, device);
  }

  if(signal(SIGALRM, alarm_f) < 0) {
    perror("signal(SIGALRM)");
    return -1;
  }

  alarm(timeout);
  if(reader->probe_f(reader, "s6350") < 0) {
    report_error(reader, "probe_f");
    if(reader->error_f(reader) == RFID_ERROR_READER_NOT_FOUND) {
      fprintf(stderr, "\
\n\
FAILURE: The test program was not able to dialog with a S6350 RFID reader\n\
FAILURE: using the %s device, please make sure a S6350 RFID reader\n\
FAILURE: is connected to it.\n\
\n\
", reader->io->device);
    }
    return -1;
  }
  alarm(0);

  alarm(timeout);
  if(reader->init_f(reader, "s6350") < 0) {
    report_error(reader, "init_f");
    return -1;
  }
  alarm(0);

  {
    s6350_transponder_t transponder;
    rfid_block_t block = { 0 };
   
    if(s6350_transponder_details(s6350, &transponder)) {
      if(reader->error_f(reader) == RFID_ERROR_TRANSPONDER_NOT_FOUND) {
	fprintf(stderr, "\
\n\
FAILURE: The test program will attempt to read / write on a transponder,\n\
FAILURE: please put a transponder on the RFID reader.\n\
\n\
");
      }
      report_error(reader, "s6350_transponder_details");
      return -1;
    }

    block.data = (u_int8_t*)malloc(transponder.bytes_per_block);

    block.data[0] = block.data[1] = block.data[2] = block.data[3] = 'B';
    block.block_number = 0;

    if(s6350_write_block(s6350, &transponder, &block)) {
      report_error(reader, "s6350_write_block");
      return -1;
    }

    for(block.block_number = 0; block.block_number < transponder.blocks; block.block_number++) {
	
      if(s6350_read_block(s6350, &transponder, &block)) {
	report_error(reader, "s6350_read_block");
	return -1;
      }
    }

    free(block.data);

    {
      rfid_block_t blocks[32];
      int i;

      for(i = 0; i < 32; i++)
	blocks[i].data = (u_int8_t*)malloc(transponder.bytes_per_block);

      if(s6350_special_read_block(s6350, &transponder, 0xFF, blocks)) {
	report_error(reader, "s6350_special_read_block");
	return -1;
      }

      for(i = 0; i < 32; i++)
	free(blocks[i].data);

    }

  }

#if 0
  S6350 say "Command not supported" (EPERM).
S6350 request : 0x01 0x0a 0x00 0x00 0x00 0x00 0xff 0x08 0xfc 0x03 
S6350 answer: 0x01 0x0a 0x00 0x00 0x00 0x10 0xff 0x02 0xe6 0x19 

  if(s6350_baud_rate_configuration(s6350, S6350_BAUD_RATE_CONFIGURATION_38400) < 0) {
    report_error(reader, "s6350_baud_rate_configuration");
    return -1;
  }
#endif /* 0 */

  reader->end_f(reader);
  rfid_free(reader);
  
  return 0;
}
