/*
 * Copyright (C) 2003 INRIA
 *
 *	INRIA
 *	Domaine de Voluceau
 *	Rocquencourt - B.P. 105
 *	78153 Le Chesnay Cedex - France
 *
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * Author: Loic Dachary <loic@gnu.org>
 * 
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif /* HAVE_CONFIG_H */

#include <unistd.h>
#include <stdlib.h>
#include <stdio.h>
#include <signal.h>
#include <memory.h>
#include <string.h>
#include <errno.h>

#include <rfid.h>
#include <cgic.h>

#define DEVICE_SIZE 256
#define DRIVER_SIZE 64

/*
 * Context of the rfid_cgi.
 */
typedef struct {
  rfid_t* rfid;
  int verbose;
  int max_count;
} rfid_cgi_t;

/*
 */
static void report_error(rfid_t* rfid, const char* message)
{
  char* str = rfid_strerror(rfid);
  fprintf(stderr, "%s: %s\n", message, str);
  fprintf(cgiOut, "ERROR %s: %s\n", message, str);
  free(str);
}

/*
 */

static int transponder_callback(rfid_t* rfid, int event, const rfid_transponder_t* transponder, void* data)
{
  rfid_cgi_t* rfid_cgi = (rfid_cgi_t*)data;
  char* id = 0;
  char* event_string = 0;

  if(rfid_cgi->verbose) {
    int i;
    fprintf(cgiOut, "# ");
    for(i = 0; i < transponder->data_length; i++)
      fprintf(cgiOut, "%c(0x%02x)/", transponder->data[i], transponder->data[i]);
    fprintf(cgiOut, "\n");
  }

  switch(event) {
  case RFID_EVENT_NEW:
    event_string = "N";
    break;
  case RFID_EVENT_DELETE:
    event_string = "D";
    break;
  case RFID_EVENT_AGAIN:
    event_string = "A";
    break;
  default:
    event_string = "(unknown event)";
    break;
  }
  
  rfid_transponder_id_get(rfid, transponder, &id);

  fprintf(cgiOut, "%s\t%s\t%04d\t%s\n",
	  event_string,
	  id,
	  transponder->blocks * transponder->bytes_per_block,
	  transponder->data);

  fflush(cgiOut);

  free(id);

  return RFID_CALLBACK_OK;
}

static RETSIGTYPE keepalive(int sig)
{
  fprintf(cgiOut, "#\n");
  fflush(cgiOut);
}

static int inventory_callback(rfid_t* rfid, rfid_transponder_t** transponders, int transponders_length, void* data)
{
  int i;
  rfid_cgi_t* rfid_cgi = (rfid_cgi_t*)data;

  if(rfid_cgi->max_count > 0)
    rfid_cgi->max_count--;

  fprintf(cgiOut, "I");

  for(i = 0; i < transponders_length; i++) {
    char* id = 0;
    rfid_transponder_id_get(rfid, transponders[i], &id);
    fprintf(cgiOut, "\t%s", id);
    free(id);
  }

  fprintf(cgiOut, "\n");

  fflush(cgiOut);

  /* Trigger keepalive if nothing happens within 60 seconds to cope with
     web server timeouts. */
  alarm(60);

  if(rfid_cgi->max_count >= 0)
    return rfid_cgi->max_count == 0 ? RFID_CALLBACK_END : RFID_CALLBACK_OK;
  else
    return RFID_CALLBACK_OK;
}

/*
 * Implements read=1, infinite loop reading commands from RFID transponders.
 */
static int read_daemon(rfid_cgi_t* rfid_cgi)
{
  if(rfid_event_loop(rfid_cgi->rfid, RFID_EVENT_NEW|RFID_EVENT_DELETE|RFID_EVENT_INVENTORY, transponder_callback, inventory_callback, rfid_cgi) < 0) {
    report_error(rfid_cgi->rfid, "rfid_event_loop");
    return -1;
  }

  return 0;
}

/*
 */
static int write_string(rfid_cgi_t* rfid_cgi, char* id, char* data)
{
  rfid_t* rfid = rfid_cgi->rfid;
  rfid_transponder_t* transponder = rfid_transponder_alloc(rfid);

  strncpy(transponder->data, data, RFID_TRANSPONDER_DATA_SIZE);
  transponder->data_length = RFID_TRANSPONDER_DATA_SIZE;
  if(strlen(id) > 0) {
    if(rfid_transponder_id_set(rfid, transponder, id) < 0) {
      report_error(rfid, "rfid_transponder_id_set");
      return -1;
    }
  }

  if(rfid_cgi->verbose) {
    fprintf(cgiOut, "# Attempting to write '%s' to a %s transponder ... ", transponder->data, (strlen(id) > 0 ? id : "any"));
    fflush(cgiOut);
  }
  
  if(rfid_write(rfid, transponder) < 0) {
    report_error(rfid, "rfid_write");
    return -1;
  }

  rfid_transponder_free(rfid, transponder);

  if(rfid_cgi->verbose) {
    fprintf(cgiOut, "done\n");
    fflush(cgiOut);
  }

  return read_daemon(rfid_cgi);
}

/*
 * Terse manual, see rfid_rfid_cgi(3) for more.
 */
static void usage(const char* message)
{
  char* usage = "\
# usage:\n\
#\n\
# rfid-cgi?[help=<n>&][device=<special>&][driver=<driver>&][verbose=1&]read=<n>\n\
# rfid-cgi?[help=<n>&][device=<special>&][driver=<driver>&][verbose=1&][id=<hex>&]write=<data>\n\
# rfid-cgi?kill=1\n\
#    read=<n>           Wait for a transponder and display its content.\n\
#                       Loop <n> times. Loop forever if <n> == -1.\n\
#    write=<data>       Wait for a transponder and store the data\n\
#                       given in argument in its memory.\n\
#    id=<hex>           II.II... RFID transponder identifier.\n\
#    kill=1             Kill all running rfid-cgi.\n\
#    device=<special>   Dialog with RFID reader connected to special\n\
#                       file <special>. Use %2F instead of /\n\
#                       (default is guessing).\n\
#    driver=<driver>    Driver to load: iso15693/m2xxh, iso15693/s6350,\n\
#                       s6350 (default is guessing).\n\
#    verbose=1          Set verbosity level to 1 (default 0).\n\
#    help=1             Print this message.\n\
#\n\
#    When read=<n> is set, each tag is reported using the following format,\n\
#    one per line. Newlines are forbiden and will break the format.\n\
#    \n\
#    [DN] <tab>II.II...<tab>BBBB<tab>CCCCCCC...\n\
#    \n\
#    Where:\n\
#    \n\
#    [DN]         is one letter describing the event: \n\
#                 N means the RFID transponder entered the field of the reader,\n\
#                 D means the RFID transponder left the field of the reader.\n\
#    II.II...     is a transponder identification in dotted hexadecimal.\n\
#    BBBB         is a is the size of the memory of the transponder as\n\
#                 4 digit decimal number\n\
#    CCCCC....    is the content of the memory of the transponder in characters\n\
#                 until a newline is found.\n\
#    <tab>        is a tabulation used as a separator.\n\
#\n\
#    At the end of each read=<n> loop iteration, an inventory of tags\n\
#    currently in the range of the RFID reader is printed as follows:\n\
#\n\
#    I<tab>II.II...<tab>II.II...<tab>II.II... ...\n\
#\n\
";
  fprintf(stderr, "usage: %s\n", message);
  fprintf(cgiOut, "# %s\n%s", message, usage);
}

/*
 */
int cgiMain() {
  int retval = 0;
  int option_read = 0;
  int option_kill = 0;
  int option_write = 0;
  char data[RFID_TRANSPONDER_DATA_SIZE + 1];
  char id[RFID_TRANSPONDER_HEX_ID_SIZE + 1];
  rfid_t rfid;
  rfid_cgi_t rfid_cgi;
  char device_string[DEVICE_SIZE];
  char* device = device_string;
  char driver_string[DRIVER_SIZE];
  char* driver = driver_string;

  memset(id, '\0', RFID_TRANSPONDER_HEX_ID_SIZE + 1);
  memset(data, '\0', RFID_TRANSPONDER_DATA_SIZE + 1);
  memset(&rfid, '\0', sizeof(rfid_t));

  rfid_cgi.rfid = &rfid;

  cgiHeaderContentType("text/plain");
  fflush(cgiOut);

  cgiFormInteger("verbose", &rfid_cgi.verbose, 0);

  {
    int dummy;
    if(cgiFormInteger("help", &dummy, 0) == cgiFormSuccess) {
      usage("Help");
      return 0;
    }
  }
  
  if(cgiFormString("device", device_string, DEVICE_SIZE) != cgiFormSuccess)
    device = 0;

  if(cgiFormString("driver", driver_string, DRIVER_SIZE) != cgiFormSuccess)
    driver = 0;

  if(cgiFormString("id", id, RFID_TRANSPONDER_HEX_ID_SIZE) != cgiFormSuccess)
    strcpy(id, "");

  if(cgiFormInteger("read", &rfid_cgi.max_count, 1) == cgiFormSuccess) {
    option_read++;
  }

  cgiFormInteger("kill", &option_kill, 0);

  switch(cgiFormString("write", data, RFID_TRANSPONDER_DATA_SIZE)) {
  case cgiFormSuccess:
    option_write++;
    break;

  case cgiFormTruncated:
    fprintf(cgiOut, "# Absolute maximum write size is %d\n", RFID_TRANSPONDER_DATA_SIZE);
    usage("Absolute maximum write size overflow");
    return -1;
    break;

  default:
    break;
  }

  if(!option_read && !option_write && !option_kill) {
    usage("Specify at least one of read=<n> write=<data> or kill=1");
    return -1;
  }

  if(option_kill) {
    system("killall --wait rfid-cgi");
    return 0;
  }

  if(option_read && option_write) {
    usage("Options read=<n> and write=<data> are mutually exclusive");
    return -1;
  }

  if(strlen(id) > 0 && !option_write) {
    usage("Options id=<hex> requires write=<data>");
    return -1;
  }

  rfid.verbose = rfid_cgi.verbose;

  if(rfid_init(&rfid, driver, device) < 0) {
    report_error(&rfid, "rfid_init");
    return -1;
  }
  
  if(option_read) {
    signal(SIGALRM, keepalive);

    retval = read_daemon(&rfid_cgi);
  } else if(option_write) {
    retval = write_string(&rfid_cgi, id, data);
  } else {
    usage("You must specify either write=<data> or read=<n>");
    retval = -1;
  }

  rfid_end(&rfid);

  return retval;
}

