/*
 * html.c: code concerning the dump as an HTML base.
 *
 * See Copyright for the status of this software.
 *
 * $Id: html.c,v 1.115 2005/03/08 14:38:28 hany Exp $
 */

#include <config.h>
#include <sys/types.h>
#include <sys/stat.h>
#ifdef HAVE_FCNTL_H
#include <fcntl.h>
#endif
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif
#include <time.h>
#include <errno.h>
#include <ctype.h>

#include <rpmlib.h>

#include "rpm2html.h"
#include "rpmdata.h"
#include "html.h"
#include "language.h"

#ifndef HAVE_SNPRINTF
#error You really need snprintf ...
#endif

/*
 * Global variables concerning the dump environment
 */

static char buf[500];

/*
 * The entities list for the extended ASCII charset.
 */
static char *html_flags[] = {
    "",
    "&lt;",
    "&lt;=",
    "&gt;",
    "&gt;=",
    "="
};

/*
 * The entities list for the extended ASCII charset.
 */
static char *entities[256] = {
NULL,		/* 000 */
NULL,		/* 001 */
NULL,		/* 002 */
NULL,		/* 003 */
NULL,		/* 004 */
NULL,		/* 005 */
NULL,		/* 006 */
NULL,		/* 007 */
NULL,		/* 008 */
NULL,		/* 009 */
NULL,		/* 010 */
NULL,		/* 011 */
NULL,		/* 012 */
NULL,		/* 013 */
NULL,		/* 014 */
NULL,		/* 015 */
NULL,		/* 016 */
NULL,		/* 017 */
NULL,		/* 018 */
NULL,		/* 019 */
NULL,		/* 020 */
NULL,		/* 021 */
NULL,		/* 022 */
NULL,		/* 023 */
NULL,		/* 024 */
NULL,		/* 025 */
NULL,		/* 026 */
NULL,		/* 027 */
NULL,		/* 028 */
NULL,		/* 029 */
NULL,		/* 030 */
NULL,		/* 031 */
NULL,		/* 032 */
NULL,		/* 033 */
"&quot;",	/* 034 */
NULL,		/* 035 */
NULL,		/* 036 */
NULL,		/* 037 */
"&amp;",	/* 038 */
NULL,		/* 039 */
NULL,		/* 040 */
NULL,		/* 041 */
NULL,		/* 042 */
NULL,		/* 043 */
NULL,		/* 044 */
NULL,		/* 045 */
NULL,		/* 046 */
NULL,		/* 047 */
NULL,		/* 048 */
NULL,		/* 049 */
NULL,		/* 050 */
NULL,		/* 051 */
NULL,		/* 052 */
NULL,		/* 053 */
NULL,		/* 054 */
NULL,		/* 055 */
NULL,		/* 056 */
NULL,		/* 057 */
NULL,		/* 058 */
NULL,		/* 059 */
"&lt;",		/* 060 */
NULL,		/* 061 */
"&gt;",		/* 062 */
NULL,		/* 063 */
NULL,		/* 064 */
NULL,		/* 065 */
NULL,		/* 066 */
NULL,		/* 067 */
NULL,		/* 068 */
NULL,		/* 069 */
NULL,		/* 070 */
NULL,		/* 071 */
NULL,		/* 072 */
NULL,		/* 073 */
NULL,		/* 074 */
NULL,		/* 075 */
NULL,		/* 076 */
NULL,		/* 077 */
NULL,		/* 078 */
NULL,		/* 079 */
NULL,		/* 080 */
NULL,		/* 081 */
NULL,		/* 082 */
NULL,		/* 083 */
NULL,		/* 084 */
NULL,		/* 085 */
NULL,		/* 086 */
NULL,		/* 087 */
NULL,		/* 088 */
NULL,		/* 089 */
NULL,		/* 090 */
NULL,		/* 091 */
NULL,		/* 092 */
NULL,		/* 093 */
NULL,		/* 094 */
NULL,		/* 095 */
NULL,		/* 096 */
NULL,		/* 097 */
NULL,		/* 098 */
NULL,		/* 099 */
NULL,		/* 100 */
NULL,		/* 101 */
NULL,		/* 102 */
NULL,		/* 103 */
NULL,		/* 104 */
NULL,		/* 105 */
NULL,		/* 106 */
NULL,		/* 107 */
NULL,		/* 108 */
NULL,		/* 109 */
NULL,		/* 110 */
NULL,		/* 111 */
NULL,		/* 112 */
NULL,		/* 113 */
NULL,		/* 114 */
NULL,		/* 115 */
NULL,		/* 116 */
NULL,		/* 117 */
NULL,		/* 118 */
NULL,		/* 119 */
NULL,		/* 120 */
NULL,		/* 121 */
NULL,		/* 122 */
NULL,		/* 123 */
NULL,		/* 124 */
NULL,		/* 125 */
NULL,		/* 126 */
NULL,		/* 127 */
NULL,		/* 128 */
NULL,		/* 129 */
NULL,		/* 130 */
NULL,		/* 131 */
NULL,		/* 132 */
NULL,		/* 133 */
NULL,		/* 134 */
NULL,		/* 135 */
NULL,		/* 136 */
NULL,		/* 137 */
NULL,		/* 138 */
NULL,		/* 139 */
NULL,		/* 140 */
NULL,		/* 141 */
NULL,		/* 142 */
NULL,		/* 143 */
NULL,		/* 144 */
NULL,		/* 145 */
NULL,		/* 146 */
NULL,		/* 147 */
NULL,		/* 148 */
NULL,		/* 149 */
NULL,		/* 150 */
NULL,		/* 151 */
NULL,		/* 152 */
NULL,		/* 153 */
NULL,		/* 154 */
NULL,		/* 155 */
NULL,		/* 156 */
NULL,		/* 157 */
NULL,		/* 158 */
NULL,		/* 159 */
"&nbsp;",	/* 160 */
"&iexcl;",	/* 161 */
"&cent;",	/* 162 */
"&pound;",	/* 163 */
"&curren;",	/* 164 */
"&yen;",	/* 165 */
"&brvbar;",	/* 166 */
"&sect;",	/* 167 */
"&uml;",	/* 168 */
"&copy;",	/* 169 */
"&ordf;",	/* 170 */
"&laquo;",	/* 171 */
"&not;",	/* 172 */
"&shy;",	/* 173 */
"&reg;",	/* 174 */
"&macr;",	/* 175 */
"&deg;",	/* 176 */
"&plusmn;",	/* 177 */
"&sup;",	/* 178 */
"&sup;",	/* 179 */
"&acute;",	/* 180 */
"&micro;",	/* 181 */
"&para;",	/* 182 */
"&middot;",	/* 183 */
"&cedil;",	/* 184 */
"&sup;",	/* 185 */
"&ordm;",	/* 186 */
"&raquo;",	/* 187 */
"&frac;",	/* 188 */
"&frac;",	/* 189 */
"&frac;",	/* 190 */
"&iquest;",	/* 191 */
"&Agrave;",	/* 192 */
"&Aacute;",	/* 193 */
"&Acirc;",	/* 194 */
"&Atilde;",	/* 195 */
"&Auml;",	/* 196 */
"&Aring;",	/* 197 */
"&AElig;",	/* 198 */
"&Ccedil;",	/* 199 */
"&Egrave;",	/* 200 */
"&Eacute;",	/* 201 */
"&Ecirc;",	/* 202 */
"&Euml;",	/* 203 */
"&Igrave;",	/* 204 */
"&Iacute;",	/* 205 */
"&Icirc;",	/* 206 */
"&Iuml;",	/* 207 */
"&ETH;",	/* 208 */
"&Ntilde;",	/* 209 */
"&Ograve;",	/* 210 */
"&Oacute;",	/* 211 */
"&Ocirc;",	/* 212 */
"&Otilde;",	/* 213 */
"&Ouml;",	/* 214 */
"&times;",	/* 215 */
"&Oslash;",	/* 216 */
"&Ugrave;",	/* 217 */
"&Uacute;",	/* 218 */
"&Ucirc;",	/* 219 */
"&Uuml;",	/* 220 */
"&Yacute;",	/* 221 */
"&THORN;",	/* 222 */
"&szlig;",	/* 223 */
"&agrave;",	/* 224 */
"&aacute;",	/* 225 */
"&acirc;",	/* 226 */
"&atilde;",	/* 227 */
"&auml;",	/* 228 */
"&aring;",	/* 229 */
"&aelig;",	/* 230 */
"&ccedil;",	/* 231 */
"&egrave;",	/* 232 */
"&eacute;",	/* 233 */
"&ecirc;",	/* 234 */
"&euml;",	/* 235 */
"&igrave;",	/* 236 */
"&iacute;",	/* 237 */
"&icirc;",	/* 238 */
"&iuml;",	/* 239 */
"&eth;",	/* 240 */
"&ntilde;",	/* 241 */
"&ograve;",	/* 242 */
"&oacute;",	/* 243 */
"&ocirc;",	/* 244 */
"&otilde;",	/* 245 */
"&ouml;",	/* 246 */
"&divide;",	/* 247 */
"&oslash;",	/* 248 */
"&ugrave;",	/* 249 */
"&uacute;",	/* 250 */
"&ucirc;",	/* 251 */
"&uuml;",	/* 252 */
"&yacute;",	/* 253 */
"&thorn;",	/* 254 */
"&yuml;"	/* 255 */
};

/*
 * Converting extended ASCII charset to valid HTML text.
 * The string returned is a shared location.
 */

static unsigned char *buffer = NULL;
static int buffer_size = 2000;
char *convertHTML(const char *str) {

    unsigned char *cur, *end;
    unsigned char c;

    if (buffer == NULL) {
        buffer = (char *) xmlMalloc(buffer_size * sizeof(char));
	if (buffer == NULL) {
	    perror("xmlMalloc failed");
	    exit(1);
	}
    }
    cur = &buffer[0];
    end = &buffer[buffer_size - 20];

    while (*str != '\0') {
        if (cur >= end) {
	    int delta = cur - buffer;

	    buffer_size *= 2;
	    buffer = (char *) xmlRealloc(buffer, buffer_size * sizeof(char));
	    if (buffer == NULL) {
		perror("xmlMalloc failed");
		exit(1);
	    }
	    end = &buffer[buffer_size - 20];
	    cur = &buffer[delta];
	}
        c = (unsigned char) *(str++);
	if (entities[(unsigned int) c] == NULL) {
	    *(cur++) = c;
	} else {
	    strcpy(cur, entities[(unsigned int) c]);
	    cur += strlen(entities[(unsigned int) c]);
	}
    }
    *cur = '\0';
    return(buffer);
}

/*
 * Converting extended ASCII charset to valid XML text.
 * The string returned is a shared location.
 */

char *convertXML(const char *str) {

    unsigned char *cur, *end;
    unsigned char c;

    if (buffer == NULL) {
        buffer = (char *) xmlMalloc(buffer_size * sizeof(char));
	if (buffer == NULL) {
	    perror("xmlMalloc failed");
	    exit(1);
	}
    }
    cur = &buffer[0];
    end = &buffer[buffer_size - 20];

    while (*str != '\0') {
        if (cur >= end) {
	    int delta = cur - buffer;

	    buffer_size *= 2;
	    buffer = (char *) xmlRealloc(buffer, buffer_size * sizeof(char));
	    if (buffer == NULL) {
		perror("xmlMalloc failed");
		exit(1);
	    }
	    end = &buffer[buffer_size - 20];
	    cur = &buffer[delta];
	}
        c = (unsigned char) *(str++);
	if (((c) == 0x09) || ((c) == 0x0A) || ((c) == 0x0D) ||
	    (((c) >= 0x20) && ((c) <= 0x8F))) {
	    if (entities[(unsigned int) c] == NULL) {
		*(cur++) = c;
	    } else {
		strcpy(cur, entities[(unsigned int) c]);
		cur += strlen(entities[(unsigned int) c]);
	    }
	}
    }
    *cur = '\0';
    return(buffer);
}

/*
 * Generates signature info (extended ASCII charset)
 * The string returned is a shared location.
 */
#define X(_x)   (unsigned)((_x) & 0xff)

static unsigned char *sigInfoMD5 = "internal MD5: ";
#if defined(WITH_GPG)
static unsigned char *sigInfoGPG = "GPG:\n";
static unsigned char *sigInfoPGP = "PGP:\n";
#else
static unsigned char *sigInfoGPG = "GPG";
static unsigned char *sigInfoPGP = "PGP";
#endif
static unsigned char *sigInfoU = "unknown";

static unsigned char *sbuffer = NULL;
static int sbuffer_size = 2000;
static unsigned char *sbuffer2 = NULL;
static int sbuffer2_size = 3;
char *convertSIG(rpmSigPtr sig) {
    int n1;
    unsigned char *md5sum = (unsigned char *) sig->sig;

    if (sbuffer == NULL) {
        sbuffer = (char *) xmlMalloc(sbuffer_size * sizeof(char));
	if (sbuffer == NULL) {
	    perror("xmlMalloc failed");
	    exit(1);
	}
    }
    if (sbuffer2 == NULL) {
        sbuffer2 = (char *) xmlMalloc(sbuffer2_size * sizeof(char));
	if (sbuffer2 == NULL) {
	    perror("xmlMalloc failed");
	    exit(1);
	}
    }

    switch(sig->tag) {
        case RPMSIGTAG_MD5:
            strncpy(sbuffer, sigInfoMD5, sbuffer_size);
            for(n1 = 0; n1 < sig->size; n1++) {
                snprintf(sbuffer2, sbuffer2_size, "%02x", X(md5sum[n1]));
                strncat(sbuffer, sbuffer2, sbuffer_size);
            }
            return(sbuffer);
        case RPMSIGTAG_GPG:
            #if defined(WITH_GPG)
            strncpy(sbuffer, sigInfoGPG, sbuffer_size);
            if (sig->resolve != NULL)
                strncat(sbuffer, sig->resolve, sbuffer_size);
            return(sbuffer);
            #else
            return(sigInfoGPG);
            #endif
        case RPMSIGTAG_PGP:
            #if defined(WITH_GPG)
            strncpy(sbuffer, sigInfoPGP, sbuffer_size);
            if (sig->resolve != NULL)
                strncat(sbuffer, sig->resolve, sbuffer_size);
            return(sbuffer);
            #else
            return(sigInfoPGP);
            #endif
        default:
            return(sigInfoU);
    }
}

/*
 * Cleanup the global varibales of this module
 */
void htmlCleanup(void) {
    if (buffer != NULL)
        xmlFree(buffer);
    if (sbuffer != NULL)
        xmlFree(sbuffer);
    if (sbuffer2 != NULL)
        xmlFree(sbuffer2);
    buffer = NULL;
    buffer_size = 2000;
    sbuffer = NULL;
    sbuffer_size = 2000;
    sbuffer2 = NULL;
    sbuffer2_size = 3;
}

/*
 * Converting extended ASCII charset to valid HTML text.
 * A new memory area is allocated for the returned value.
 */

char *xmlStrdupHTML(const char *str) {
    return(xmlStrdup(convertHTML(str)));
}

/*
 * createDirectory : create a directory. It does recurse and create the
 *                   father dir if needed.
 */

void createDirectory(const char *dirname) {
    static char last_dir[2000] = "";

    /*
     * avoid costly syscalls.
     */
    if (!strcmp(last_dir, dirname)) return;

    if (mkdir(dirname, 0777) != 0) {
        switch (errno) {
#ifdef EEXIST
            case EEXIST:
		if (rpm2htmlVerbose > 1)
		    fprintf(stderr, "Directory \"%s\" already exists\n",
		            dirname);
		return;
#endif
	    case ENOENT: {
	        char *father = xmlStrdup(dirname);
		char *cur = &father[strlen(father)];

		while (cur > father) {
		    if (*cur == '/') {
		        *cur = '\0';
			break;
	            }
		    cur--;
		}
		if (cur > father) {
		    createDirectory(father);
		    if (mkdir(dirname, 0777) != 0) {
			fprintf(stderr, "createDirectory \"%s\" failed\n",
			        dirname);
			perror("mkdir failed:");
			return;
		    }
		}
		xmlFree(father);
		
	        break;
	    }
            default:
	        fprintf(stderr, "createDirectory \"%s\" failed\n", dirname);
	        perror("mkdir failed:");
		return;
	}
    } else if (rpm2htmlVerbose)
        fprintf(stderr, "Created directory \"%s\"\n", dirname);
}

/*
 * Stuff needed for the icon creation.
 */

#include <zlib.h>
#include "dir.png.h"
#include "new.png.h"

void dumpDirIcon(void) {
    char path[500];
    char *content;
    int fd;
    /* struct stat *buf; !!!!!! */

    if (!rpm2html_dump_html) return;

    createDirectory(rpm2html_dir);
    snprintf(path, sizeof(path), "%s/dir.png", rpm2html_dir);
    content = read_data_buffer_dir_png();
    if ((fd = creat(path, 0644)) < 0) {
        fprintf(stderr, "creat() failed on %s\n", path);
	return;
    }
    write(fd, content, data_buffer_dir_png_size);
    close(fd);
    snprintf(path, sizeof(path), "%s/new.png", rpm2html_dir);
    content = read_data_buffer_new_png();
    if ((fd = creat(path, 0644)) < 0) {
        fprintf(stderr, "creat() failed on %s\n", path);
	return;
    }
    write(fd, content, data_buffer_new_png_size);
    close(fd);
}

/*
 * checkDate : check whether the last modification time of a file
 *             is anterior to a given time
 */

int checkDate(const char *filename, time_t stamp) {
    struct stat buf;

    if (force) return(0);
    if ((stat(filename, &buf)) != 0) {
        return(0);
    }
    if (buf.st_size < 10) {
        return(0);
    }
    return(buf.st_mtime > stamp);
}

/*
 * checkDirectory : check if this directory exist
 */

int checkDirectory(const char *filename) {
    struct stat buf;

    if (force) return(0);
    if ((stat(filename, &buf)) != 0) {
        return(0);
    }
    if ((S_ISDIR(buf.st_mode)) || (S_ISLNK(buf.st_mode)))
        return(1);
    return(0);
}

/*
 * checkFile : check if this file exist
 */

int checkFile(const char *filename) {
    struct stat buf;

    if (force) return(0);
    if ((stat(filename, &buf)) != 0) {
        return(0);
    }
    return(1);
}

/*
 * Transformation function from rpm to filename.
 */

const char *rpmName(rpmDataPtr cur) {
    static char rbuf[500];

    if (cur->arch != NULL)
	snprintf(rbuf, sizeof(rbuf), "%s-%s-%s.%s", 
	    cur->name, cur->version, cur->release, cur->arch);
    else
	snprintf(rbuf, sizeof(rbuf), "%s-%s-%s", 
	    cur->name, cur->version, cur->release);
    return(rbuf);
}

/*
 * Transformation function from rpm to Software name.
 */

const char *rpmSoftwareName(rpmDataPtr cur) {
    static char rbuf[500];

    snprintf(rbuf, sizeof(rbuf), "%s-%s-%s", 
	cur->name, cur->version, cur->release);
    return(rbuf);
}

/*
 * remove symbols that we do not want in filenames..
 */
const char *cleanName (const char *name) {
    static char cbuf[500];
    char *cur = cbuf;
 
    while (*name != '\0') {
        if ((*name == '/')
	 || (*name == ' ')
	 || (*name == '/')
	 || (*name == '"')
	 || (*name == '<')
	 || (*name == '>')
	 || (*name == ':')
	 || (*name == '|')
	 || (*name == '@')
	 || (*name == '\t')
	 || (*name == '\r')
	 || (*name == '\n')) {
	    *cur++ = '_';
	    name++;
	} else *cur++ = *name++;
    }
    *cur = '\0';
    return (cbuf);
}

/*
 * do the URI escaping needed for that name.
 */
const char *escapeName (const char *name) {
    static char cbuf[500];
    char *cur = cbuf, *end = &cbuf[495];
    char tmp;
 
    while ((*name != '\0') && (cur < end)) {
	tmp = *name++;
        if (((tmp >= 'a') && (tmp <= 'z')) ||
            ((tmp >= 'A') && (tmp <= 'Z')) ||
	    ((tmp >= '0') && (tmp <= '9')) ||
	    (tmp == '-') || (tmp == '_') || (tmp == '.') ||
	    (tmp == '!') || (tmp == '~') || (tmp == '*') || (tmp == '\'') ||
	    (tmp == '(') || (tmp == ')')) {
	    *cur++ = tmp;

	} else {
	    int val = (unsigned char)tmp;
	    int hi = val / 0x10, lo = val % 0x10;
	    *cur++ = '%';
	    *cur++ = hi + (hi > 9? 'A'-10 : '0');
	    *cur++ = lo + (lo > 9? 'A'-10 : '0');
	}
    }
    *cur = '\0';
    return (cbuf);
}

/*
 * Transformation function from group to filename.
 */

const char *groupName(const char *group) {
    static char gbuf[500];

    strncpy(gbuf, cleanName (group), 500);
    strncat(gbuf, localizedStrings[LANG_HTML_SUFFIX], 500);
    return(gbuf);
}

/*
 * Transformation function from group to filename.
 */

const char *distribName(const char *distrib) {
    static char dbuf[500];

    strncpy(dbuf, cleanName (distrib), 500);
    strncat(dbuf, localizedStrings[LANG_HTML_SUFFIX], 500);
    return(dbuf);
}

/*
 * Generate a full URL
 */

void fullURL(char *buf, int len, rpmDirPtr dir, const char *subdir, const char *filename) {
    if (dir == NULL) {
	snprintf(buf, len, "http://%s%s", rpm2html_host, rpm2html_url);
    } else {
	    if (subdir == NULL)
	        snprintf(buf, len, "http://%s%s/%s",
			 rpm2html_host, rpm2html_url,
			 filename);
            else
	        snprintf(buf, len, "http://%s%s/%s/%s",
			 rpm2html_host, rpm2html_url,
			 subdir, filename);
    }
}

/*
 * Generate a full directory path.
 */

void fullPathName(char *buf, int len, rpmDirPtr dir, const char *subdir, const char *filename) {
    if (dir == NULL) {
	snprintf(buf, len, "%s/%s", rpm2html_dir, filename);
    } else {
	    if (subdir == NULL)
	        snprintf(buf, len, "%s/%s", dir->dir, filename);
            else
	        snprintf(buf, len, "%s/%s/%s", dir->dir, subdir, filename);
    }
}

/*
 * Generate a full directory path plus ditinguish with a letter.
 */

void fullPathNameLr(char *buf, int len, rpmDirPtr dir, char *subdir,
                    char *filename, char letter) {
    if (dir == NULL) {
	snprintf(buf, len, "%s/%c%s", rpm2html_dir, letter, filename);
    } else {
	if (subdir == NULL)
	    snprintf(buf, len, "%s/%c%s", dir->dir, letter, filename);
	else
	    snprintf(buf, len, "%s/%s/%c%s", dir->dir, subdir, letter, filename);
    }
}

/*
 * Generate a full directory path plus ditinguish with a number.
 */

void fullPathNameNr(char *buf, int len, rpmDirPtr dir, char *subdir,
                    char *filename, int number) {
    if (dir == NULL) {
	snprintf(buf, len, "%s/%d%s", rpm2html_dir, number, filename);
    } else {
	if (subdir == NULL)
	    snprintf(buf, len, "%s/%d%s", dir->dir, number, filename);
	else
	    snprintf(buf, len, "%s/%s/%d%s", dir->dir, subdir, number, filename);
    }
}

/*
 * Transformation function from vendor to filename.
 */

const char *vendorName(const char *vendor) {
    static char vbuf[500];

    strncpy(vbuf, cleanName (vendor), 500);
    strncat(vbuf, localizedStrings[LANG_HTML_SUFFIX], 500);
    return(vbuf);
}

/*
 * Transformation function from resource to filename.
 */

const char *resourceName(const char *resource) {
    static char rbuf[500];

    strncpy(rbuf, cleanName (resource), 500);
    strncat(rbuf, localizedStrings[LANG_HTML_SUFFIX], 500);
    return(rbuf);
}

/*
 * Generate an HTML header
 */

void generateHtmlHeader(FILE *html, char *title, char *color) {
    fprintf(html, "<!DOCTYPE HTML PUBLIC \"-//W3C//DTD HTML 4.0 Transitional//EN\">\n");
    fprintf(html, "<html>\n<head>\n<title>%s</title>\n", title);
    fprintf(html, "<meta name=\"GENERATOR\" content=\"%s %s\">\n",
            rpm2html_rpm2html_name, rpm2html_rpm2html_ver);
    if (color == NULL)
        fprintf(html,
	        "</head>\n<body bgcolor=\"#ffffff\" text=\"#000000\">\n");
    else 
        fprintf(html, "</head>\n<body bgcolor=\"%s\" text=\"#000000\">\n",
	        color);
}

/*
 * Generate an HTML footer
 */

void generateHtmlFooter(FILE *html) {
    struct tm * tstruct;

    tstruct = localtime(&currentTime);

    fprintf(html, "<hr>\n");
    fprintf(html, "<p>%s <a href=\"%s\">%s %s</a>\n",
            localizedStrings[LANG_GENERATED],
	    rpm2html_rpm2html_url, rpm2html_rpm2html_name,
	    rpm2html_rpm2html_ver);
    if (rpm2html_help != NULL) {
	fprintf(html, "<p><a href=\"%s\">%s</a>, %s\n",
		rpm2html_help, rpm2html_maint, asctime(tstruct));
    } else {
	fprintf(html, "<p><a href=\"mailto:%s\">%s</a>, %s\n",
		rpm2html_mail, rpm2html_maint, asctime(tstruct));
    }
    fprintf(html, "</body>\n</html>\n");
}

/*
 * Generate a the opening anchor tag for a package
 */
void generateHtmlRpmAnchor(FILE *html, rpmDataPtr cur) {
    if (cur->dir == NULL) {
	fprintf(html, "<a href=\"\">");
	return;
    }
    if ((cur->dir->subdir != NULL) && (cur->dir->subdir[0] != '\0')) {
        /*
	 * More than one mirror, there is an HTML subdir
	 */
	if ((cur->subdir != NULL) && (cur->subdir[0] != '\0')) {
	    if (cur->dir->url)
		fprintf(html, "<a href=\"%s/%s/%s/%s.html\">",
			cur->dir->url, cur->dir->subdir,
			cur->subdir, rpmName(cur));
	    else
		fprintf(html, "<a href=\"%s/%s/%s.html\">",
			cur->subdir, cur->dir->subdir, rpmName(cur));
	} else {
	    if (cur->dir->url)
		fprintf(html, "<a href=\"%s/%s/%s.html\">",
		        cur->dir->url, cur->dir->subdir, rpmName(cur));
	    else
		fprintf(html, "<a href=\"%s/%s.html\">",
		        cur->dir->subdir, rpmName(cur));
	}
    } else {
        /*
	 * Only one mirror, no HTML subdir
	 */
	if ((cur->subdir != NULL) && (cur->subdir[0] != '\0')) {
	    if (cur->dir->url)
		fprintf(html, "<a href=\"%s/%s/%s.html\">",
			cur->dir->url, cur->subdir, rpmName(cur));
	    else
		fprintf(html, "<a href=\"%s/%s.html\">",
			cur->subdir, rpmName(cur));
	} else {
	    if (cur->dir->url)
		fprintf(html, "<a href=\"%s/%s.html\">",
		        cur->dir->url, rpmName(cur));
	    else
		fprintf(html, "<a href=\"%s.html\">",
		        rpmName(cur));
	}
    }
}

/*
 * Generate an RSS link element for a package
 */
void generateRSSLink(FILE *RSS, rpmDataPtr cur) {
    fprintf(RSS, "    <link>http://%s", rpm2html_host);
    if ((cur->dir->subdir != NULL) && (cur->dir->subdir[0] != '\0')) {
        /*
	 * More than one mirror, there is an HTML subdir
	 */
	if ((cur->subdir != NULL) && (cur->subdir[0] != '\0')) {
	    if (cur->dir->url)
		fprintf(RSS, "%s/%s/%s/%s.html",
			cur->dir->url, cur->dir->subdir,
			cur->subdir, rpmName(cur));
	    else
		fprintf(RSS, "%s/%s/%s.html",
			cur->subdir, cur->dir->subdir, rpmName(cur));
	} else {
	    if (cur->dir->url)
		fprintf(RSS, "%s/%s/%s.html",
		        cur->dir->url, cur->dir->subdir, rpmName(cur));
	    else
		fprintf(RSS, "%s/%s.html",
		        cur->dir->subdir, rpmName(cur));
	}
    } else {
        /*
	 * Only one mirror, no HTML subdir
	 */
	if ((cur->subdir != NULL) && (cur->subdir[0] != '\0')) {
	    if (cur->dir->url)
		fprintf(RSS, "%s/%s/%s.html",
			cur->dir->url, cur->subdir, rpmName(cur));
	    else
		fprintf(RSS, "%s/%s.html",
			cur->subdir, rpmName(cur));
	} else {
	    if (cur->dir->url)
		fprintf(RSS, "%s/%s.html",
		        cur->dir->url, rpmName(cur));
	    else
		fprintf(RSS, "%s.html",
		        rpmName(cur));
	}
    }
    fprintf(RSS, "</link>\n");
}

/*
 * Generate a line in a table for a RPM.
 * Don't list the source RPMs
 */

void generateHtmlRpmRow(FILE *html, rpmDataPtr cur, int shownew) {
#ifdef SHOW_DATE
    static char buf[500];
    struct tm * tstruct;
#endif

    if (!strcmp(cur->arch, "src")) return;

#ifdef SHOW_DATE
    tstruct = localtime(&(cur->date));
#ifdef HAVE_STRFTIME
    strftime(buf, sizeof(buf) - 1, "%c", tstruct);
#else
#error "no strftime, please check !"
#endif
#endif

    if (cur->dir != NULL) {
	fprintf(html, "<tr bgcolor=\"%s\"><td width=\"%d\">",
		cur->dir->color, PACKAGE_FIELD_WIDTH);
    } else {
	fprintf(html, "<tr><td width=\"%d\">",
		PACKAGE_FIELD_WIDTH);
    }
    generateHtmlRpmAnchor(html, cur);
    fprintf(html, "%s</a>", rpmName(cur));
    if ((shownew) && (cur->date > 0) &&
        ((currentTime - cur->date) < (24 * 60 * 60 * 15)))
	fprintf(html, "<img src=\"%s/new.png\" alt=\"New\">", rpm2html_url);
    fprintf(html, "</td>\n<td width=\"%d\">%s</td>\n",
            DESCRIPTION_FIELD_WIDTH,
	    convertHTML(cur->summary));
#ifdef SHOW_DATE
    fprintf(html, "<td>%s</td></tr>\n", buf);
#else
    if ((cur->dir != NULL) && (cur->dir->name != NULL))
	fprintf(html, "<td>%s</td></tr>\n", cur->dir->name);
#endif
}

/*
 * Generate an entry in an RSS channel for an RPM.
 */
void generateRSSItem(FILE *RSS, rpmDataPtr cur) {
    fprintf(RSS, "  <item>\n");
    fprintf(RSS, "    <title>%s</title>\n", rpmName(cur));
    if (cur->summary != NULL)
	fprintf(RSS, "    <description>%s</description>\n",
		convertHTML(cur->summary));
    generateRSSLink(RSS, cur);
    fprintf(RSS, "  </item>\n");
}

/*
 * Generate a line in a table for an RPM software and all it's architectures.
 */

void generateHtmlRpmArchRow(FILE *html, rpmDataPtr cur) {
    rpmDataPtr tmp;

    fprintf(html, "<tr><td width=\"%d\">", PACKAGE_FIELD_WIDTH);
    fprintf(html, "%s</td>\n", rpmSoftwareName(cur));
    fprintf(html, "<td width=\"%d\">%s</td>\n",
            DESCRIPTION_FIELD_WIDTH, convertHTML(cur->summary));
    /* dump the archs list */
    tmp = cur;
    while (tmp != NULL) {
	if (strcmp(tmp->arch, "src")) {
	    if (tmp->dir != NULL)
		fprintf(html, "<td bgcolor=\"%s\" width=\"%d\">",
			tmp->dir->color, SYSTEM_FIELD_WIDTH);
	    else
		fprintf(html, "<td width=\"%d\">",
			SYSTEM_FIELD_WIDTH);
	    generateHtmlRpmAnchor(html, tmp);
	    fprintf(html, "%s/%s</a>", tmp->os, tmp->arch);
	    if ((currentTime - tmp->date) < (24 * 60 * 60 * 15))
		fprintf(html, "<img src=\"%s/new.png\" alt=\"New\">",
		        rpm2html_url);
	    fprintf(html, "</td>");
	}
	tmp = tmp->nextArch;
    }
    fprintf(html, "\n</tr>\n");
}

/*
 * Generate the Links for the main pages
 */

void generateLinks(FILE *html, int installed) {
    int i;

    fprintf(html, "<table border=5 cellspacing=5 cellpadding=5>\n");
    fprintf(html, "<tbody>\n<tr>\n");

    fprintf(html, "<td><a href=\"%s/%s\">%s</a></td>\n",
            rpm2html_url, localizedStrings[LANG_INDEX_HTML],
	    localizedStrings[LANG_INDEX]);
    fprintf(html, "<td><a href=\"%s/%s\">%s</a></td>\n",
            rpm2html_url, localizedStrings[LANG_GROUP_HTML],
	    localizedStrings[LANG_SORTED_BY_GROUP]);
    fprintf(html, "<td><a href=\"%s/%s\">%s</a></td>\n",
            rpm2html_url, localizedStrings[LANG_DISTRIB_HTML],
	    localizedStrings[LANG_SORTED_BY_DISTRIB]);
    fprintf(html, "<td><a href=\"%s/%s\">%s</a></td>\n",
            rpm2html_url, localizedStrings[LANG_VENDOR_HTML],
	    localizedStrings[LANG_SORTED_BY_VENDOR]);
    if (installed)
	fprintf(html, "<td><a href=\"%s/%s\">%s</a></td>\n",
		rpm2html_url, localizedStrings[LANG_BYDATE_HTML],
		localizedStrings[LANG_SORTED_BY_IDATE]);
    else
	fprintf(html, "<td><a href=\"%s/%s\">%s</a></td>\n",
		rpm2html_url, localizedStrings[LANG_BYDATE_HTML],
		localizedStrings[LANG_SORTED_BY_CDATE]);
    fprintf(html, "<td><a href=\"%s/%s\">%s</a></td>\n",
            rpm2html_url, localizedStrings[LANG_BYNAME_HTML],
	    localizedStrings[LANG_SORTED_BY_NAME]);

    fprintf(html, "<td><a href=\"http://rpmfind.net/linux/rpm2html/mirrors.html\">%s</a></td>\n",
	    localizedStrings[LANG_MIRRORS]);

    for (i = 0;i < rpm2html_nb_extra_headers;i++) {
        if ((*rpm2html_headers_url[i] == '/') ||
	    (!strncmp(rpm2html_headers_url[i], "http://", 7)) ||
	    (!strncmp(rpm2html_headers_url[i], "ftp://", 6)) ||
	    (!strncmp(rpm2html_headers_url[i], "mailto", 6)))
	    fprintf(html, "<td><a href=\"%s\">%s</a></td>\n",
	            rpm2html_headers_url[i], rpm2html_headers_name[i]);
	else
	    fprintf(html, "<td><a href=\"%s/%s\">%s</a></td>\n",
	            rpm2html_url, rpm2html_headers_url[i],
		    rpm2html_headers_name[i]);
    }

    fprintf(html, "</tr>\n</tbody></table>\n");
}

/*
 * Generate a color indicator
 */

void generateColorIndicator(FILE *html) {
#ifdef SHOW_DATE
    int nb = 0;
    rpmDirPtr dir = dirList;

    fprintf(html, "<table align=\"center\"><tbody>\n<tr>\n");
    while (dir != NULL) {
	if (strcasecmp(dir->color, "#ffffff")) {
	    if ((nb > 0) && ((nb % MAX_COLOR_PER_LINE) == 0))
		fprintf(html, "</tr><tr>\n");
	    fprintf(html, "<td bgcolor=\"%s\">", dir->color);
	    if (dir->name != NULL)
		fprintf(html, "%s</td>", dir->name);
	    else if (dir->mirrors[0] != NULL)
		fprintf(html, "%s</td>", dir->mirrors[0]);
	    else
		fprintf(html, "%s</td>", dir->ftp);
	    nb++;
	}
	dir = dir->next;
    }
    fprintf(html, "</tr>\n</tbody></table>\n");
#endif
}

/*
 * Dump a subtree in an HTML page with all the links
 */

void generateHtmlTree(FILE *html, rpmSubdirPtr tree, int level, int full) {
    int i;

    if (tree->html == 0) return;
    if ((tree->nb_subdirs > 0) || (tree->parent != NULL)) {
	if (level == 0)
	    fprintf(html, "<h3>%s</h3>\n", localizedStrings[LANG_SUBDIRS]);
	if (tree->color)
	    fprintf(html, "<blockquote style=\"background : %s\">\n",
	            tree->color);
	else
	    fprintf(html, "<blockquote>\n");
	if ((level == 0) && (tree->parent != NULL)) {
	    if (tree->parent->htmlpath[0] != '\0') {
		fprintf(html, "<p><strong><a href=\"%s/%s/%s\">",
			rpm2html_url, tree->parent->htmlpath,
			localizedStrings[LANG_INDEX_HTML]);
	    } else {
		fprintf(html, "<p><strong><a href=\"%s/%s\">",
			rpm2html_url, localizedStrings[LANG_INDEX_HTML]);
	    }
	    fprintf(html,
	        "<img src=\"%s/dir.png\" alt=\"parent-directory\" border=0>",
	            rpm2html_url);
	    fprintf(html, " ..</a></strong></p>\n");
	}
	for (i = 0;i < tree->nb_subdirs;i++) {
	    if (tree->subdirs[i]->html == 0) continue;
	    fprintf(html, "<p><strong><a href=\"%s/%s/%s\">",
		    rpm2html_url, tree->subdirs[i]->htmlpath,
		    localizedStrings[LANG_INDEX_HTML]);
	    fprintf(html,
	        "<img src=\"%s/dir.png\" alt=\"sub-directory\" border=0>",
	            rpm2html_url);
	    fprintf(html, " %s</a></strong></p>\n", tree->subdirs[i]->name);
	    if (full)
		generateHtmlTree(html, tree->subdirs[i], level + 1, full);
	}
	fprintf(html, "</blockquote>\n");
    }
}

/*
 * Dump the parent names in an HTML page with all the links
 */

void generateHtmlParentsNames(FILE *html, rpmSubdirPtr tree) {
    int i, j, k;
    rpmSubdirPtr parent;

    if (tree->parent == NULL) return;

    /*
     * get the tree depth.
     */
    i = 0;
    parent = tree->parent;
    while ((parent != NULL) && (parent->parent != NULL)) {
        i++;
	parent = parent->parent;
    }

    /*
     * Dump each level in the parent tree.
     */
    for (j = i; j >= 0;j--) {
	parent = tree;
        for (k = 0;k < j;k++) parent = parent->parent;
	if (parent->htmlpath[0] != '\0') {
	    fprintf(html, "<a href=\"%s/%s/%s\">",
		    rpm2html_url, parent->htmlpath,
		    localizedStrings[LANG_INDEX_HTML]);
	} else {
	    fprintf(html, "<a href=\"%s/%s\">", rpm2html_url,
		    localizedStrings[LANG_INDEX_HTML]);
	}
	fprintf(html, "%s</a>\n", parent->name);
	if (j != 0) fprintf(html, " / ");
    }
}

/*
 * Dump the whole index for a full complete config file.
 */
void dumpIndex(time_t start_time, int installed) {
    FILE *html;

    if (!rpm2html_dump_html) return;

    if (rpm2htmlVerbose > 1) {
    printf("Dumping %s/%s\n", rpm2html_dir, localizedStrings[LANG_INDEX_HTML]);
    }
    snprintf(buf, sizeof(buf), "%s/%s", rpm2html_dir, localizedStrings[LANG_INDEX_HTML]);

    html = fopen(buf, "w");
    if (html == NULL) {
        fprintf(stderr, "Couldn't save to file %s: %s\n",
	        buf, strerror(errno));
        return;
    }

    if (installed) {
	snprintf(buf, sizeof(buf), "%s %s", localizedStrings[LANG_WELCOME_INSTALL],
	        rpm2html_host);
	generateHtmlHeader(html, buf, NULL);
	generateLinks(html, installed);
	fprintf(html, "<h1 align=center>%s %s</h1>\n",
		localizedStrings[LANG_WELCOME_INSTALL], rpm2html_host);
    } else {
	snprintf(buf, sizeof(buf), "%s %s", localizedStrings[LANG_WELCOME_REPOSITORY],
	        rpm2html_host);
	generateHtmlHeader(html, buf, NULL);
	generateLinks(html, installed);
	fprintf(html, "<h1 align=center>%s %s</h1>\n",
		localizedStrings[LANG_WELCOME_REPOSITORY], rpm2html_host);
    }

    fprintf(html, "%s\n", localizedStrings[LANG_RPM2HTML_INTRO]);
#ifdef WITH_SQL
    snprintf(buf, sizeof(buf), "http://%s/linux/rpm2html", rpm2html_host);
    fprintf(html, localizedStrings[LANG_SEARCH_FORM], buf);
#else
    fprintf(html, localizedStrings[LANG_SEARCH_FORM],
	    "http://rpmfind.net/linux/rpm2html");
#endif
    if (rpm2html_install_files != 0) {
	fprintf(html, "<h3>");
	fprintf(html, localizedStrings[LANG_INSTALLED_STATS],
		rpm2html_install_files, rpm2html_install_size / 1024);
	fprintf(html, "</h3>\n");
    }
    if (rpm2html_files != 0) {
	fprintf(html, "<h3>");
	fprintf(html, localizedStrings[LANG_STATS],
		rpm2html_files, rpm2html_size / 1024);
	fprintf(html, "</h3>\n");
    }
    fprintf(html, "<ul>\n");
    fprintf(html, "<li>\n%s <a href=\"%s\">%s</a>\n",
            localizedStrings[LANG_LIST],
	    localizedStrings[LANG_GROUP_HTML],
	    localizedStrings[LANG_INDEX_GROUP]);
    fprintf(html, "<li>\n%s <a href=\"%s\">%s</a>\n",
            localizedStrings[LANG_LIST],
	    localizedStrings[LANG_BYDATE_HTML],
	    localizedStrings[LANG_INDEX_CREATION]);
    fprintf(html, "<li>\n%s <a href=\"%s\">%s</a>\n",
            localizedStrings[LANG_LIST],
	    localizedStrings[LANG_BYNAME_HTML],
	    localizedStrings[LANG_INDEX_NAME]);
    fprintf(html, "<li>\n%s <a href=\"%s\">%s</a>\n",
            localizedStrings[LANG_LIST],
	    localizedStrings[LANG_VENDOR_HTML],
	    localizedStrings[LANG_INDEX_VENDOR]);
    fprintf(html, "<li>\n%s <a href=\"%s\">%s</a>\n",
            localizedStrings[LANG_LIST],
	    localizedStrings[LANG_DISTRIB_HTML],
	    localizedStrings[LANG_INDEX_DISTRIB]);
    fprintf(html, "</ul>\n");

    generateHtmlTree(html, dirTree, 0, 0);

    fprintf(html, "<p>%s %d %s</p>\n",
            localizedStrings[LANG_GENERATION_TIME],
            (int) (time(NULL) - start_time),
            localizedStrings[LANG_SECONDS]);
    generateHtmlFooter(html);
    fclose(html);
}

/*
 * Dump the whole index for a full complete config file.
 */
void dumpTopIndex(rpmDirPtr *sqlDirList) {
    FILE *html;
    int i;

    if (!rpm2html_dump_html) return;

    if (rpm2htmlVerbose > 1) {
    printf("Dumping %s/%s\n", rpm2html_dir, localizedStrings[LANG_INDEX_HTML]);
    }
    snprintf(buf, sizeof(buf), "%s/%s", rpm2html_dir, localizedStrings[LANG_INDEX_HTML]);

    html = fopen(buf, "w");
    if (html == NULL) {
        fprintf(stderr, "Couldn't save to file %s: %s\n",
	        buf, strerror(errno));
        return;
    }

    snprintf(buf, sizeof(buf), "%s %s", localizedStrings[LANG_WELCOME_REPOSITORY],
	    rpm2html_host);
    generateHtmlHeader(html, buf, NULL);
    generateLinks(html, 0);
    fprintf(html, "<h1 align=center>%s %s</h1>\n",
	    localizedStrings[LANG_WELCOME_REPOSITORY], rpm2html_host);

    fprintf(html, "%s\n", localizedStrings[LANG_RPM2HTML_INTRO]);
#ifdef WITH_SQL
    snprintf(buf, sizeof(buf), "http://%s/linux/rpm2html", rpm2html_host);
    fprintf(html, localizedStrings[LANG_SEARCH_FORM], buf);
#else
    fprintf(html, localizedStrings[LANG_SEARCH_FORM],
	    "http://rpmfind.net/linux/rpm2html");
#endif
    if (rpm2html_install_files != 0) {
	fprintf(html, "<h3>");
	fprintf(html, localizedStrings[LANG_INSTALLED_STATS],
		rpm2html_install_files, rpm2html_install_size / 1024);
	fprintf(html, "</h3>\n");
    }
    if (rpm2html_files != 0) {
	fprintf(html, "<h3>");
	fprintf(html, localizedStrings[LANG_STATS],
		rpm2html_files, rpm2html_size / 1024);
	fprintf(html, "</h3>\n");
    }
    fprintf(html, "<ul>\n");
    fprintf(html, "<li>\n%s <a href=\"%s\">%s</a>\n",
            localizedStrings[LANG_LIST],
	    localizedStrings[LANG_GROUP_HTML],
	    localizedStrings[LANG_INDEX_GROUP]);
    fprintf(html, "<li>\n%s <a href=\"%s\">%s</a>\n",
            localizedStrings[LANG_LIST],
	    localizedStrings[LANG_BYDATE_HTML],
	    localizedStrings[LANG_INDEX_CREATION]);
    fprintf(html, "<li>\n%s <a href=\"%s\">%s</a>\n",
            localizedStrings[LANG_LIST],
	    localizedStrings[LANG_BYNAME_HTML],
	    localizedStrings[LANG_INDEX_NAME]);
    fprintf(html, "<li>\n%s <a href=\"%s\">%s</a>\n",
            localizedStrings[LANG_LIST],
	    localizedStrings[LANG_VENDOR_HTML],
	    localizedStrings[LANG_INDEX_VENDOR]);
    fprintf(html, "<li>\n%s <a href=\"%s\">%s</a>\n",
            localizedStrings[LANG_LIST],
	    localizedStrings[LANG_DISTRIB_HTML],
	    localizedStrings[LANG_INDEX_DISTRIB]);
    fprintf(html, "</ul>\n");

    if (sqlDirList != NULL) {
	for (i = 0; i < 500;i++)
	    if (sqlDirList[i] != NULL) {
		/*
		 * Check first that the distro is still active and that
		 * the index has been generated
		 */
		if (!checkDirectory(sqlDirList[i]->rpmdir))
		    continue;
		snprintf(buf, sizeof(buf), "%s/%s/index.html",
			 sqlDirList[i]->dir, sqlDirList[i]->subdir);
		if (!checkFile(buf))
		    continue;

		fprintf(html,
"<p><strong><a href=\"%s/%s/index.html\"><img src=\"/linux/RPM/dir.png\" alt=\"sub-directory\" border=0>%s</a></strong></p>\n",
                        sqlDirList[i]->url, sqlDirList[i]->subdir,
			sqlDirList[i]->name);

	    }
    }

    generateHtmlFooter(html);
    fclose(html);
}

/*
 * Dump an RPM block as an HTML file.
 */

void dumpRpmHtml(rpmDataPtr rpm, rpmSubdirPtr tree) {
    struct tm * tstruct;
    rpmDirPtr dir = rpm->dir;
    int installed = dir->installbase;
    FILE *html;
    int i;

    if (!rpm2html_dump_html) return;

    /*
     * create the directory on the fly if needed.
     */
    if ((dir->subdir != NULL) && (dir->subdir[0] != '\0')) {
	if ((rpm->subdir != NULL) && (rpm->subdir[0] != '\0'))
	    snprintf(buf, sizeof(buf), "%s/%s/%s", dir->dir, dir->subdir,
	            rpm->subdir);
	else
	    snprintf(buf, sizeof(buf), "%s/%s", dir->dir, dir->subdir);
    } else {
	if ((rpm->subdir != NULL) && (rpm->subdir[0] != '\0'))
	    snprintf(buf, sizeof(buf), "%s/%s", dir->dir, rpm->subdir);
	else
	    snprintf(buf, sizeof(buf), "%s", dir->dir);
    }

    createDirectory(buf);
    strcat(buf, "/");
    strcat(buf, rpmName(rpm));
    strcat(buf, ".html");

    /*
     * !!!!!!!!
     * if (checkDate(buf, rpm->extra->stamp)) return;
     */
    if (rpm2htmlVerbose > 1) {
        printf("Dumping %s\n", buf);
    }

    html = fopen(buf, "w");
    if (html == NULL) {
        fprintf(stderr, "Couldn't save to file %s: %s\n",
	        buf, strerror(errno));
        return;
    }
    snprintf(buf, sizeof(buf), "%s RPM", rpmName(rpm));
    generateHtmlHeader(html, buf, NULL);
    generateLinks(html, installed);
    if ((rpm->subdir != NULL) && (rpm->subdir[0] != '\0')) {
	if (dir->mirrors[0] != NULL)
	    fprintf(html, "<h1 align=center><a href=\"%s/%s/%s\">\n",
		dir->mirrors[0], rpm->subdir, rpm->filename);
	else
	    fprintf(html, "<h1 align=center><a href=\"%s/%s/%s\">\n",
		dir->ftp, rpm->subdir, rpm->filename);
    } else {
	if (dir->mirrors[0] != NULL)
	    fprintf(html, "<h1 align=center><a href=\"%s/%s\">\n",
		dir->mirrors[0], rpm->filename);
	else
	    fprintf(html, "<h1 align=center><a href=\"%s/%s\">\n",
		dir->ftp, rpm->filename);
    }
    if (rpm->arch) {
        if (!strcmp(rpm->arch, "src"))
	    fprintf(html, "%s-%s-%s Source RPM</a></h1>\n",
	            rpm->name, rpm->version, rpm->release);
        else
	    fprintf(html, "%s-%s-%s RPM for %s</a></h1>\n",
		rpm->name, rpm->version, rpm->release, rpm->arch);
    } else {
	fprintf(html, "%s-%s-%s RPM</a></h1>\n",
            rpm->name, rpm->version, rpm->release);
    }

    if ((tree != NULL) && (tree->parent != NULL)) {
	fprintf(html, "<h3 align=center>%s ", localizedStrings[LANG_FROM]);
	generateHtmlParentsNames(html, tree);
	fprintf(html, "</h3>\n");
    } else if (dir->name) {
        fprintf(html, "<h3 align=center>%s <a href=\"%s\">%s</a></h3>\n",
	      localizedStrings[LANG_FROM],
	      dir->ftp, dir->name);
    }

    fprintf(html, "<table align=center border=5 cellspacing=5 cellpadding=5 bgcolor=\"%s\">", dir->color);
    fprintf(html, "<tbody>\n");
    fprintf(html, "<tr><td>%s: %s</td>\n",
	    localizedStrings[LANG_NAME],
            rpm->name);
    fprintf(html, "<td>%s: <a href=\"%s/%s\">%s</a></td></tr>\n",
	    localizedStrings[LANG_DISTRIBUTION], rpm2html_url,
            distribName(rpm->distribution), convertHTML(rpm->distribution));
    fprintf(html, "<tr><td>%s: %s</td>\n",
	    localizedStrings[LANG_VERSION],
            rpm->version);
    fprintf(html, "<td>%s: <a href=\"%s/%s\">%s</a></td></tr>\n",
	    localizedStrings[LANG_VENDOR], rpm2html_url,
            vendorName(rpm->vendor), convertHTML(rpm->vendor));
    tstruct = localtime(&(rpm->date));
#ifdef HAVE_STRFTIME
    strftime(buf, sizeof(buf) - 1, "%c", tstruct);
#else
#error "no strftime, please check !"
#endif
    if (installed) {
	fprintf(html, "<tr><td>%s: %s</td>\n<td>%s: %s</td></tr>\n",
		localizedStrings[LANG_RELEASE],
		rpm->release,
		localizedStrings[LANG_INSTALL_DATE],
		buf);
    } else {
	fprintf(html, "<tr><td>%s: %s</td>\n<td>%s: %s</td></tr>\n",
		localizedStrings[LANG_RELEASE],
		rpm->release,
		localizedStrings[LANG_BUILD_DATE],
		buf);
    }
    if (rpm2html_url != NULL)
	fprintf(html, "<tr><td>%s: <a href=\"%s/%s\">%s</a></td>\n",
		localizedStrings[LANG_GROUP], rpm2html_url,
		groupName(rpm->group), convertHTML(rpm->group));
    else
	fprintf(html, "<tr><td>%s: <a href=\"%s\">%s</a></td>\n",
		localizedStrings[LANG_GROUP],
		groupName(rpm->group), convertHTML(rpm->group));
    fprintf(html, "<td>%s: %s</td></tr>\n",
	    localizedStrings[LANG_BUILD_HOST], rpm->extra->host);
    fprintf(html, "<tr><td>%s: %d</td>\n",
	    localizedStrings[LANG_SIZE],
            rpm->size);
    if (dir->ftpsrc) {
	fprintf(html, "<td>%s: <a href=\"%s/%s\">%s</a></td></tr>\n",
	        localizedStrings[LANG_RPM_SRC],
		dir->ftpsrc, rpm->extra->srcrpm, rpm->extra->srcrpm);
    } else {
	fprintf(html, "<td>%s: %s</td></tr>\n",
	        localizedStrings[LANG_RPM_SRC],
		rpm->extra->srcrpm);
    }
    if (rpm->extra->packager) {
        char *email = extractEMail(rpm->extra->packager);
	if (email == NULL)
	    fprintf(html, "<tr><td colspan=\"2\">%s: %s</td></tr>\n",
		    localizedStrings[LANG_PACKAGER],
		    convertHTML(rpm->extra->packager));
        else
	    fprintf(html, "<tr><td colspan=\"2\">%s: <a href=\"mailto:%s\">%s</a></td></tr>\n",
		    localizedStrings[LANG_PACKAGER],
		    email, convertHTML(rpm->extra->packager));
    }
    if (rpm->url)
	fprintf(html, "<tr><td colspan=\"2\">%s: <a href=\"%s\">%s</a></td></tr>\n",
		localizedStrings[LANG_URL],
		rpm->url, rpm->url);
    fprintf(html, "<tr><td colspan=\"2\">%s: %s</td></tr>\n",
	    localizedStrings[LANG_SUMMARY],
            convertHTML(rpm->summary));
    fprintf(html, "</tbody>\n</table>\n");
    fprintf(html, "<pre>%s\n</pre>\n", convertHTML(rpm->extra->description));
    if ((rpm->extra->nb_resources + rpm->extra->nb_requires < 2) &&
        (!strstr(rpm->name, "lib")) &&
	(!strcmp(rpm->arch, "src"))) {
       if (rpm2htmlVerbose > 1)
	   fprintf(stderr, "Resource lists problem : %s\n", rpmName(rpm));
       fprintf(html, "<h2 align=center style=\"color : #ff0000\">%s</h2>\n",
               localizedStrings[LANG_WARNING_RESOURCES]);
       fprintf(html, "<h2 align=center><a href=\"%s/%s\">%s</a></h2>\n",
               rpm2html_url, resourceName(rpm->name),
               localizedStrings[LANG_CHOOSE_ANOTHER]);
    }
    if (rpm->extra->nb_resources > 0) {
       fprintf(html, "<h3>%s</h3>\n",
	    localizedStrings[LANG_PROVIDE]);
       fprintf(html, "<ul>\n");
       for (i = 0;i < rpm->extra->nb_resources;i++) {
#ifdef WITH_SQL
           if (rpm2html_search != NULL)
	       fprintf(html, "<li><a href=\"%s?query=%s\">%s</a>\n",
		   rpm2html_search,
		   escapeName(rpm->extra->resources[i]->name),
		   rpm->extra->resources[i]->name);
	   else
#endif
	   if (rpm2html_url != NULL)
	       fprintf(html, "<li><a href=\"%s/%s\">%s</a>\n",
		   rpm2html_url,
		   resourceName(rpm->extra->resources[i]->name),
		   rpm->extra->resources[i]->name);
	   else
	       fprintf(html, "<li><a href=\"%s\">%s</a>\n",
		       resourceName(rpm->extra->resources[i]->name),
		       rpm->extra->resources[i]->name);
       }
       fprintf(html, "</ul>\n");
    }
    if (rpm->extra->nb_requires > 0) {
       fprintf(html, "<h3>%s</h3>\n",
	    localizedStrings[LANG_REQUIRE]);
       fprintf(html, "<ul>\n");
       for (i = 0;i < rpm->extra->nb_requires;i++) {
#ifdef WITH_SQL
           if (rpm2html_search != NULL) {
	       if ((rpm->extra->requires[i]->flag != RPM2HTML_REQ_NONE) &&
		   (rpm->extra->requires[i]->version != NULL)) {
		   fprintf(html, "<li><a href=\"%s?query=%s\">%s</a> %s %s\n",
			   rpm2html_search,
			   escapeName(rpm->extra->requires[i]->name), 
			   rpm->extra->requires[i]->name,
			   html_flags[rpm->extra->requires[i]->flag],
			   rpm->extra->requires[i]->version);
	       } else {
		   fprintf(html, "<li><a href=\"%s?query=%s\">%s</a>\n",
			   rpm2html_search,
			   escapeName(rpm->extra->requires[i]->name), 
			   rpm->extra->requires[i]->name);
	       }
	   } else 
#endif
	   if (rpm2html_url != NULL) {
	       if ((rpm->extra->requires[i]->flag != RPM2HTML_REQ_NONE) &&
		   (rpm->extra->requires[i]->version != NULL)) {
		   fprintf(html, "<li><a href=\"%s/%s\">%s</a> %s %s\n",
			   rpm2html_url,
			   resourceName(rpm->extra->requires[i]->name), 
			   rpm->extra->requires[i]->name,
			   html_flags[rpm->extra->requires[i]->flag],
			   rpm->extra->requires[i]->version);
	       } else {
		   fprintf(html, "<li><a href=\"%s/%s\">%s</a>\n",
			   rpm2html_url,
			   resourceName(rpm->extra->requires[i]->name), 
			   rpm->extra->requires[i]->name);
	       }
	   } else {
	       if ((rpm->extra->requires[i]->flag != RPM2HTML_REQ_NONE) &&
		   (rpm->extra->requires[i]->version != NULL)) {
		   fprintf(html, "<li><a href=\"%s\">%s</a> %s %s\n",
			   resourceName(rpm->extra->requires[i]->name),
			   rpm->extra->requires[i]->name,
			   html_flags[rpm->extra->requires[i]->flag],
			   rpm->extra->requires[i]->version);
	       } else {
		   fprintf(html, "<li><a href=\"%s\">%s</a>\n",
			   resourceName(rpm->extra->requires[i]->name),
			   rpm->extra->requires[i]->name);
	       }
	   }
       }
       fprintf(html, "</ul>\n");
    }
    if (rpm->extra->copyright) {
       fprintf(html, "<h3>%s</h3>\n",
	    localizedStrings[LANG_COPYRIGHT]);
	fprintf(html, "<pre>%s\n</pre>\n", convertHTML(rpm->extra->copyright));
    }
    if (rpm->extra->sigs != NULL) {
       fprintf(html, "<h3>%s</h3>\n",
	    localizedStrings[LANG_SIGNATURES]);
	for(i = 0; i < rpm->extra->nb_sigs; i++)
	    fprintf(html, "<pre>%s\n</pre>\n", convertHTML(convertSIG(rpm->extra->sigs[i])));
    }
    if (rpm->extra->changelog) {
        fprintf(html, "<h3>%s</h3>\n",
	    localizedStrings[LANG_CHANGELOG]);
	fprintf(html, "<pre>%s\n</pre>\n", convertHTML(rpm->extra->changelog));
    }
    fprintf(html, "<h3>%s</h3>\n",
	localizedStrings[LANG_FILES]);
    if (rpm->extra->filelist == NULL) 
	fprintf(html, "<bold>%s</bold>\n",
	     localizedStrings[LANG_NO_FILES]);
    else
	fprintf(html, "<pre>%s\n</pre>\n", rpm->extra->filelist);
    
    generateHtmlFooter(html);
    fclose(html);
}

#ifdef WITH_SQL
/*
 * Dump a resource HTML file redirecting to the search engine.
 */

void dumpRessRedirHtml(const char *name) {
    FILE *html;

    if (!rpm2html_dump_html) return;

    /*
     * Guess why ??? index.html gets ovewritten !
     */
    if (!strcmp(name, "index"))
	return;

    snprintf(buf, sizeof(buf), "%s/%s", rpm2html_dir, resourceName(name));

    if (rpm2htmlVerbose > 1) {
	printf("Dumping %s/%s\n", rpm2html_dir, resourceName(name));
    }

    html = fopen(buf, "w");
    if (html == NULL) {
        fprintf(stderr, "Couldn't save to file %s: %s\n",
	        buf, strerror(errno));
        return;
    }
    snprintf(buf, sizeof(buf), "%s %s",
	 localizedStrings[LANG_RPM_RESOURCE], name);
    fprintf(html, "<!DOCTYPE HTML PUBLIC \"-//W3C//DTD HTML 4.0 Transitional//EN\">\n");
    fprintf(html, "<html>\n<head>\n<title>%s</title>\n", buf);
    fprintf(html, "<meta name=\"GENERATOR\" content=\"%s %s\">\n",
            rpm2html_rpm2html_name, rpm2html_rpm2html_ver);
    if (rpm2html_search != NULL)
	fprintf(html,
	    "<META HTTP-EQUIV=\"refresh\" CONTENT=\"0; URL=%s?query=%s\">\n", 
	    rpm2html_search, escapeName(name));
    else
	fprintf(html,
	    "<META HTTP-EQUIV=\"refresh\" CONTENT=\"0; URL=http://rpmfind.net/linux/rpm2html/search.php?query=%s\">\n", 
	        escapeName(name));
    fprintf(html, "</head>\n<body bgcolor=\"#ffffff\" text=\"#000000\">\n");
    generateLinks(html, 0);
    
    fprintf(html, "<h1 align=center>%s %s</h1>\n",
	 localizedStrings[LANG_RPM_RESOURCE], name);
    fprintf(html, "<p>This static page has been replaced\n");
    if (rpm2html_search != NULL)
	fprintf(html, "<a href=\"%s?query=%s\">",
		rpm2html_search, escapeName(name));
    else
	fprintf(html,
	  "<a href=\"http://rpmfind.net/linux/rpm2html/search.php?query=%s\">",
	        escapeName(name));
    fprintf(html, "by a dynamic one</a></p>\n");
    fprintf(html, "<p>Thank you for updating your bookmarks</p>\n");
    generateHtmlFooter(html);
    fclose(html);
}
#endif

/*
 * Dump a resource block as an HTML file.
 */

void dumpRessHtml(rpmRessPtr ress, int installed) {
    rpmDataPtr rpm;
    FILE *html;
    int i;
    const char *name = NULL;

    if (!rpm2html_dump_html) return;
#ifdef WITH_SQL
    if (rpm2html_search != NULL) {
	dumpRessRedirHtml(ress->name);
	return;
    }
#endif

    /*
     * Guess why ??? index.html gets ovewritten !
     */
    if (!strcmp(ress->name, "index"))
	return;

    rpmRessSort(ress);

    snprintf(buf, sizeof(buf), "%s/%s", rpm2html_dir,
           resourceName(ress->name));
    /*
     * !!!!!
     * if (checkDate(buf, ress->stamp)) return;
     */
    if (rpm2htmlVerbose > 1) {
    printf("Dumping %s/%s\n", rpm2html_dir,
           resourceName(ress->name));
    }

    html = fopen(buf, "w");
    if (html == NULL) {
        fprintf(stderr, "Couldn't save to file %s: %s\n",
	        buf, strerror(errno));
        return;
    }
    snprintf(buf, sizeof(buf), "%s %s",
	 localizedStrings[LANG_RPM_RESOURCE], ress->name);
    generateHtmlHeader(html, buf, NULL);
    generateLinks(html, installed);
    generateColorIndicator(html);
    fprintf(html, "<h1 align=center>%s %s</h1>\n",
	 localizedStrings[LANG_RPM_RESOURCE], ress->name);
    fprintf(html, "<h3>%s</h3>\n",
	 localizedStrings[LANG_PROVIDED_BY]);
    fprintf(html, "<table><tbody>\n");
    for (i = 0;i < ress->nb_provider;i++) {
        rpm = ress->provider[i];
	if ((name != NULL) && (strcmp(rpm->name, name)))
	    fprintf(html, "</tbody></table>\n<br>\n<table><tbody>\n");
	name = rpm->name;
	generateHtmlRpmRow(html, rpm, 1);
    }
    fprintf(html, "</tbody></table>\n");
    
    generateHtmlFooter(html);
    fclose(html);
}

/*
 * Dump all RPM blocks in the HTML files.
 */
void dumpAllRessHtml(int installed) {
    rpmRessPtr cur;

    if (!rpm2html_dump_html) return;

    if (installed) cur = ressInstalledList;
    else cur = ressList;

    while (cur != NULL) {
        dumpRessHtml(cur, installed);
	cur = cur->next;
    }
}

/*
 * Dump the Groups of all HTML files.
 * One expect that the RPM files have been sorted by group.
 */
void dumpDirRpmByGroups(rpmDataPtr list, rpmDirPtr dir,
                   char *subdir, int installed) {
    FILE *Groups;
    FILE *currentGroup = NULL;
    rpmDataPtr cur, prev = NULL;
    int count = 0;
    int pcount = 0;

    if (!rpm2html_dump_html) return;
    if ((dir != NULL) && (!dir->html)) return;

    cur = list;
    
    fullPathName(buf, sizeof(buf), dir, subdir, localizedStrings[LANG_GROUP_HTML]);
    if (rpm2htmlVerbose > 1) {
        printf("Dumping %s\n", buf);
    }

    Groups = fopen(buf, "w");
    if (Groups == NULL) {
        fprintf(stderr, "Couldn't save to file %s: %s\n",
	        buf, strerror(errno));
        return;
    }
    generateHtmlHeader(Groups, localizedStrings[LANG_SORTED_BY_GROUP], NULL);
    generateLinks(Groups, installed);
    fprintf(Groups, "<h1 align=center>%s</h1>\n",
        localizedStrings[LANG_SORTED_BY_GROUP]);
    fprintf(Groups, "<ul>\n");

    while (cur != NULL) {
        if ((cur->group != NULL) && (strlen(cur->group) > 0)) {
	    if ((prev == NULL) || (strcasecmp(prev->group, cur->group))) {
		if (pcount != 0)
		    fprintf(Groups, " (%d)</a></li>\n", pcount);
		pcount = 0;
	        if (currentGroup != NULL) {
		    /* one need to close the current group list */
		    fprintf(currentGroup,"</tbody></table>\n");
		    generateHtmlFooter(currentGroup);
		    fclose(currentGroup);
		}

		/* Add the current group in the Group list */
		fprintf(Groups, "<li><a href=\"%s\">%s",
		        groupName(cur->group), convertHTML(cur->group));

		/* open the new HTML group file */
                fullPathName(buf, sizeof(buf), dir, subdir, groupName(cur->group));
		if (rpm2htmlVerbose > 1) {
		    printf("Dumping %s\n", buf);
		}

		currentGroup = fopen(buf, "w");
		if (currentGroup == NULL) {
		    fprintf(stderr, "Couldn't save to file %s: %s\n",
			    buf, strerror(errno));
		    return;
		}
                snprintf(buf, sizeof(buf), "%s %s", localizedStrings[LANG_OF_GROUP],
		        convertHTML(cur->group));
                generateHtmlHeader(currentGroup, buf, NULL);
		generateLinks(currentGroup, installed);
		fprintf(currentGroup,
		  "<h1 align=center>%s %s</h1>\n",
		     localizedStrings[LANG_OF_GROUP], convertHTML(cur->group));
		generateColorIndicator(currentGroup);
		fprintf(currentGroup,"<table><tbody>\n");
		count = 0;
	    }
	    generateHtmlRpmArchRow(currentGroup, cur);
	    count++;
	    pcount++;
	    if ((count % MAX_TABLE_LENGHT) == 0)
		fprintf(currentGroup, "</tbody></table>\n<table><tbody>\n");
	}
	prev = cur;
	if ((dir == NULL) && (subdir == NULL))
	    cur = cur->nextSoft;
	else
	    cur = cur->next;
    }

    /*
     * finish the last group line.
     */
    if (pcount != 0)
	fprintf(Groups, " (%d)</a></li>\n", pcount);

    if (currentGroup != NULL) {
	/* one need to close the current group list */
	fprintf(currentGroup,"</tbody></table>\n");
	generateHtmlFooter(currentGroup);
	fclose(currentGroup);
    }
    fprintf(Groups, "</ul>\n");
    generateHtmlFooter(Groups);
    fclose(Groups);
}

/*
 * Dump the full set of  RPM by Group HTML file.
 * One expect that the RPM files have been sorted by groups.
 */
void dumpRpmByGroups(rpmDataPtr list, int installed) {
    if (!rpm2html_dump_html) return;

    dumpDirRpmByGroups(list, NULL, NULL, installed);
}

/*
 * Dump the Distribs HTML files.
 * One expect that the RPM files have been sorted by distribution.
 */
void dumpDirRpmByDistribs(rpmDataPtr list, rpmDirPtr dir,
                      char *subdir, int installed) {
    FILE *Distribs;
    FILE *currentDistrib = NULL;
    rpmDataPtr cur, prev = NULL;
    int count = 0;
    int pcount = 0;

    if (!rpm2html_dump_html) return;
    if ((dir != NULL) && (!dir->html)) return;

    cur = list;

    fullPathName(buf, sizeof(buf), dir, subdir, localizedStrings[LANG_DISTRIB_HTML]);
    if (rpm2htmlVerbose > 1) {
        printf("Dumping %s\n", buf);
    }

    Distribs = fopen(buf, "w");
    if (Distribs == NULL) {
        fprintf(stderr, "Couldn't save to file %s: %s\n",
	        buf, strerror(errno));
        return;
    }
    generateHtmlHeader(Distribs,
        localizedStrings[LANG_SORTED_BY_DISTRIB], NULL);
    generateLinks(Distribs, installed);
    fprintf(Distribs, "<h1 align=center>%s</h1>\n",
        localizedStrings[LANG_SORTED_BY_DISTRIB]);
    fprintf(Distribs, "<ul>\n");

    while (cur != NULL) {
        if ((cur->distribution != NULL) && (strlen(cur->distribution) > 0)) {
	    if ((prev == NULL) || (strcasecmp(prev->distribution, cur->distribution))) {
		if (pcount != 0)
		    fprintf(Distribs, " (%d)</a></li>\n", pcount);
		pcount = 0;

	        if (currentDistrib != NULL) {
		    /* one need to close the current distribution list */
		    fprintf(currentDistrib,"</tbody></table>\n");
		    generateHtmlFooter(currentDistrib);
		    fclose(currentDistrib);
		}

		/* Add the current distribution in the Distrib list */
		fprintf(Distribs, "<li><a href=\"%s\">%s",
		        distribName(cur->distribution),
			convertHTML(cur->distribution));

		/* open the new HTML distribution file */
                fullPathName(buf, sizeof(buf), dir, subdir, distribName(cur->distribution));
		if (rpm2htmlVerbose > 1) {
		    printf("Dumping %s\n", buf);
		}

		currentDistrib = fopen(buf, "w");
		if (currentDistrib == NULL) {
		    fprintf(stderr, "Couldn't save to file %s: %s\n",
			    buf, strerror(errno));
		    return;
		}
                snprintf(buf, sizeof(buf), "%s %s",
		    localizedStrings[LANG_OF_DISTRIB],
		    convertHTML(cur->distribution));
                generateHtmlHeader(currentDistrib, buf, NULL);
		generateLinks(currentDistrib, installed);
		fprintf(currentDistrib,
		  "<h1 align=center>%s %s</h1>\n",
		  localizedStrings[LANG_OF_DISTRIB],
		  convertHTML(cur->distribution));
		if (!installed) generateColorIndicator(currentDistrib);
		fprintf(currentDistrib,"<table><tbody>\n");
		count = 0;
	    }
	    generateHtmlRpmRow(currentDistrib, cur, 1);
	    count++;
	    pcount++;
	    if ((count % MAX_TABLE_LENGHT) == 0)
		fprintf(currentDistrib, "</tbody></table>\n<table><tbody>\n");
	}
	prev = cur;
	cur = cur->next;
    }
    /*
     * finish the last group line.
     */
    if (pcount != 0)
	fprintf(Distribs, " (%d)</a></li>\n", pcount);

    if (currentDistrib != NULL) {
	/* one need to close the current distribution list */
	fprintf(currentDistrib,"</tbody></table>\n");
	generateHtmlFooter(currentDistrib);
	fclose(currentDistrib);
    }
    fprintf(Distribs, "</ul>\n");
    generateHtmlFooter(Distribs);
    fclose(Distribs);
}

/*
 * Dump the full set of  RPM by Distribution HTML file.
 * One expect that the RPM files have been sorted by distributions.
 */
void dumpRpmByDistribs(rpmDataPtr list, int installed) {
    if (!rpm2html_dump_html) return;

    dumpDirRpmByDistribs(list, NULL, NULL, installed);
}

/*
 * Dump the Vendors HTML files.
 * One expect that the RPM files have been sorted by vendors.
 */
void dumpDirRpmByVendors(rpmDataPtr list, rpmDirPtr dir,
                      char *subdir, int installed) {
    FILE *Vendors;
    FILE *currentVendor = NULL;
    rpmDataPtr cur, prev = NULL;
    int count = 0;
    int pcount = 0;

    if (!rpm2html_dump_html) return;

    cur = list;

    fullPathName(buf, sizeof(buf), dir, subdir, localizedStrings[LANG_VENDOR_HTML]);
    if (rpm2htmlVerbose > 1) {
        printf("Dumping %s\n", buf);
    }

    Vendors = fopen(buf, "w");
    if (Vendors == NULL) {
        fprintf(stderr, "Couldn't save to file %s: %s\n",
	        buf, strerror(errno));
        return;
    }
    generateHtmlHeader(Vendors, localizedStrings[LANG_SORTED_BY_VENDOR], NULL);
    generateLinks(Vendors, installed);
    fprintf(Vendors, "<h1 align=center>%s</h1>\n",
        localizedStrings[LANG_SORTED_BY_VENDOR]);
    fprintf(Vendors, "<ul>\n");

    while (cur != NULL) {
        if ((cur->vendor != NULL) && (strlen(cur->vendor) > 0)) {
	    if ((prev == NULL) || (strcasecmp(prev->vendor, cur->vendor))) {
		if (pcount != 0)
		    fprintf(Vendors, " (%d)</a></li>\n", pcount);
		pcount = 0;

	        if (currentVendor != NULL) {
		    /* one need to close the current vendor list */
		    fprintf(currentVendor,"</tbody></table>\n");
		    generateHtmlFooter(currentVendor);
		    fclose(currentVendor);
		}

		/* Add the current vendor in the Vendor list */
		fprintf(Vendors, "<li><a href=\"%s\">%s",
		        vendorName(cur->vendor), convertHTML(cur->vendor));

		/* open the new HTML vendor file */
                fullPathName(buf, sizeof(buf), dir, subdir, vendorName(cur->vendor));
		if (rpm2htmlVerbose > 1) {
		    printf("Dumping %s\n", buf);
		}

		currentVendor = fopen(buf, "w");
		if (currentVendor == NULL) {
		    fprintf(stderr, "Couldn't save to file %s: %s\n",
			    buf, strerror(errno));
		    return;
		}
                snprintf(buf, sizeof(buf), "%s %s", localizedStrings[LANG_OF_VENDOR],
		    convertHTML(cur->vendor));
                generateHtmlHeader(currentVendor, buf, NULL);
		generateLinks(currentVendor, installed);
		fprintf(currentVendor,
		  "<h1 align=center>%s %s</h1>\n",
		  localizedStrings[LANG_OF_VENDOR], convertHTML(cur->vendor));
		if (!installed) generateColorIndicator(currentVendor);
		fprintf(currentVendor,"<table><tbody>\n");
		count = 0;
	    }
	    generateHtmlRpmRow(currentVendor, cur, 1);
	    count++;
	    pcount++;
	    if ((count % MAX_TABLE_LENGHT) == 0)
		fprintf(currentVendor, "</tbody></table>\n<table><tbody>\n");
	}
	prev = cur;
	cur = cur->next;
    }

    /*
     * finish the last group line.
     */
    if (pcount != 0)
	fprintf(Vendors, " (%d)</a></li>\n", pcount);

    if (currentVendor != NULL) {
	/* one need to close the current vendor list */
	fprintf(currentVendor,"</tbody></table>\n");
	generateHtmlFooter(currentVendor);
	fclose(currentVendor);
    }
    fprintf(Vendors, "</ul>\n");
    generateHtmlFooter(Vendors);
    fclose(Vendors);
}

/*
 * Dump the full set of  RPM by Vendor HTML file.
 * One expect that the RPM files have been sorted by vendors.
 */
void dumpRpmByVendors(rpmDataPtr list, int installed) {
    if (!rpm2html_dump_html) return;

    dumpDirRpmByVendors(list, NULL, NULL, installed);
}

/*
 * Dump the RPM by Date HTML file.
 * One expect that the RPM files have been sorted by date.
 */
void dumpDirRpmByDate(rpmDataPtr list, rpmDirPtr dir,
                      char *subdir, int installed) {
    FILE *ByDate;
    FILE *RSS;
    rpmDataPtr cur;
    int count = 0;
    int rdfcount = 0;
    int page = 1;

    if (!rpm2html_dump_html) return;
    if ((dir != NULL) && (!dir->html)) return;

    cur = list;

    fullPathName(buf, sizeof(buf), dir, subdir, localizedStrings[LANG_BYDATE_HTML]);
    if (rpm2htmlVerbose > 1) {
	printf("Dumping %s\n", buf);
    }

    ByDate = fopen(buf, "w");
    if (ByDate == NULL) {
        fprintf(stderr, "Couldn't save to file %s: %s\n",
	        buf, strerror(errno));
        return;
    }
    fullPathName(buf, sizeof(buf), dir, subdir, "rdf");
    if (rpm2htmlVerbose > 1) {
	printf("Dumping RSS channel %s\n", buf);
    }

    RSS = fopen(buf, "w");
    if (RSS == NULL) {
        fprintf(stderr, "Couldn't save to file %s: %s\n",
	        buf, strerror(errno));
	fclose(ByDate);
        return;
    }
    if (installed) {
	generateHtmlHeader(ByDate, localizedStrings[LANG_SORTED_BY_IDATE], NULL);
	generateLinks(ByDate, installed);
	fprintf(ByDate, "<h1 align=center>%s</h1>\n",
	    localizedStrings[LANG_SORTED_BY_IDATE]);
    } else {
	generateHtmlHeader(ByDate, localizedStrings[LANG_SORTED_BY_CDATE], NULL);
	generateLinks(ByDate, installed);
	fprintf(ByDate, "<h1 align=center>%s</h1>\n",
	    localizedStrings[LANG_SORTED_BY_CDATE]);
    }
    if (RSS) {
	fprintf(RSS, "<?xml version=\"1.0\"?>\n");
	fprintf(RSS,
    "<!DOCTYPE rss PUBLIC \"-//Netscape Communications//DTD RSS 0.91//EN\"\n");
	fprintf(RSS,
    "		 \"http://my.netscape.com/publish/formats/rss-0.91.dtd\">\n"); 
	fprintf(RSS, "<rss version=\"0.91\">\n");
	fprintf(RSS, "  <channel>\n");
	if (dir != NULL) {
	    fprintf(RSS, "    <title>Rpmfind %s index on %s</title>\n",
		    dir->name, rpm2html_host);
	    fullURL(buf, sizeof(buf), dir, subdir,
		    localizedStrings[LANG_INDEX_HTML]);
	    fprintf(RSS, "    <link>%s</link>\n", buf);
	    fprintf(RSS, "    <description>Rpmfind %s News on %s</description>\n",
		    rpm2html_host, dir->name);
	} else {
	    fprintf(RSS, "    <title>Rpmfind index on %s</title>\n",
		    rpm2html_host);
	    fullURL(buf, sizeof(buf), dir, subdir,
		    localizedStrings[LANG_INDEX_HTML]);
	    fprintf(RSS, "    <link>%s</link>\n", buf);
	    fprintf(RSS, "    <description>Rpmfind News on %s</description>\n",
		    rpm2html_host);
	}
	fprintf(ByDate, localizedStrings[LANG_RDF_CHANNEL], "rdf");
    }

    /*
     * Skip all the RPMs with date in the futur :-(
     */
    while ((cur != NULL) && ((cur->date < 0) || (currentTime < cur->date))) {
	if (rpm2htmlVerbose > 1) {
	    fprintf(stderr, "dropping %s, date %d > current time %d\n",
		    rpmSoftwareName(cur), (int) cur->date, (int) currentTime);
	}
	cur = cur->next;
    }

    /*
     * First dump RPMs less than 3 days old.
     */
    if (!installed) generateColorIndicator(ByDate);
    if (installed)
	fprintf(ByDate, "<h2>%s</h2>\n", localizedStrings[LANG_I_LESS_3D_OLD]);
    else
	fprintf(ByDate, "<h2>%s</h2>\n", localizedStrings[LANG_LESS_3D_OLD]);
    fprintf(ByDate, "<table><tbody>\n");
    while ((cur != NULL) && ((currentTime - cur->date) < (3 * 24 * 60 * 60))) {
	generateHtmlRpmRow(ByDate, cur, 0);
	if ((RSS != NULL) && (rdfcount++ < 20)) {
	    generateRSSItem(RSS, cur);
	}
	cur = cur->next;
	count++;
	if ((count % MAX_TABLE_LENGHT) == 0)
	    fprintf(ByDate, "</tbody></table>\n<table><tbody>\n");
	if (count > MAX_PAGE_LENGHT) {
	    count = 0;
	    fprintf(ByDate, "</tbody></table>\n");
	    fprintf(ByDate, "<a href=\"%d%s\">...</a>\n",
	            page, localizedStrings[LANG_BYDATE_HTML]);
	    generateHtmlFooter(ByDate);
	    fclose(ByDate);
	    fullPathNameNr(buf, sizeof(buf), dir, subdir, localizedStrings[LANG_BYDATE_HTML], page++);
	    if (rpm2htmlVerbose > 1) {
		printf("Dumping %s\n", buf);
	    }
	    ByDate = fopen(buf, "w");
	    if (ByDate == NULL) {
		fprintf(stderr, "Couldn't save to file %s: %s\n",
			buf, strerror(errno));
		if (RSS) {
		    fprintf(RSS, "  </channel>\n");
		    fprintf(RSS, "</rss>\n");
		    fclose(RSS);
		}
		return;
	    }
	    if (installed) {
		generateHtmlHeader(ByDate,
		          localizedStrings[LANG_SORTED_BY_IDATE], NULL);
		generateLinks(ByDate, installed);
		fprintf(ByDate, "<h1 align=center>%s</h1>\n",
		    localizedStrings[LANG_SORTED_BY_IDATE]);
	    } else {
		generateHtmlHeader(ByDate,
		          localizedStrings[LANG_SORTED_BY_CDATE], NULL);
		generateLinks(ByDate, installed);
		fprintf(ByDate, "<h1 align=center>%s</h1>\n",
		    localizedStrings[LANG_SORTED_BY_CDATE]);
	    }
	    if (installed)
		fprintf(ByDate, "<h2>%s</h2>\n", localizedStrings[LANG_I_LESS_3D_OLD]);
	    else
		fprintf(ByDate, "<h2>%s</h2>\n", localizedStrings[LANG_LESS_3D_OLD]);
	    fprintf(ByDate, "<table><tbody>\n");
	}
    }
    fprintf(ByDate, "</tbody></table>\n");

    /*
     * Then dump RPMs less than one week old.
     */
    if (installed)
	fprintf(ByDate, "<h2>%s</h2>\n", localizedStrings[LANG_I_LESS_1W_OLD]);
    else
	fprintf(ByDate, "<h2>%s</h2>\n", localizedStrings[LANG_LESS_1W_OLD]);
    fprintf(ByDate, "<table><tbody>\n");
    while ((cur != NULL) && ((currentTime - cur->date) < (7 * 24 * 60 * 60))) {
	generateHtmlRpmRow(ByDate, cur, 0);
	if ((RSS != NULL) && (rdfcount++ < 20)) {
	    generateRSSItem(RSS, cur);
	}
	cur = cur->next;
	count++;
	if ((count % MAX_TABLE_LENGHT) == 0)
	    fprintf(ByDate, "</tbody></table>\n<table><tbody>\n");
	if (count > MAX_PAGE_LENGHT) {
	    count = 0;
	    fprintf(ByDate, "</tbody></table>\n");
	    fprintf(ByDate, "<a href=\"%d%s\">...</a>\n",
	            page, localizedStrings[LANG_BYDATE_HTML]);
	    generateHtmlFooter(ByDate);
	    fclose(ByDate);
	    fullPathNameNr(buf, sizeof(buf), dir, subdir,
	                   localizedStrings[LANG_BYDATE_HTML], page++);
	    if (rpm2htmlVerbose > 1) {
		printf("Dumping %s\n", buf);
	    }
	    ByDate = fopen(buf, "w");
	    if (ByDate == NULL) {
		fprintf(stderr, "Couldn't save to file %s: %s\n",
			buf, strerror(errno));
		if (RSS) {
		    fprintf(RSS, "  </channel>\n");
		    fprintf(RSS, "</rss>\n");
		    fclose(RSS);
		}
		return;
	    }
	    if (installed) {
		generateHtmlHeader(ByDate,
		          localizedStrings[LANG_SORTED_BY_IDATE], NULL);
		generateLinks(ByDate, installed);
		fprintf(ByDate, "<h1 align=center>%s</h1>\n",
		    localizedStrings[LANG_SORTED_BY_IDATE]);
	    } else {
		generateHtmlHeader(ByDate,
		          localizedStrings[LANG_SORTED_BY_CDATE], NULL);
		generateLinks(ByDate, installed);
		fprintf(ByDate, "<h1 align=center>%s</h1>\n",
		    localizedStrings[LANG_SORTED_BY_CDATE]);
	    }
	    if (installed)
		fprintf(ByDate, "<h2>%s</h2>\n", localizedStrings[LANG_I_LESS_1W_OLD]);
	    else
		fprintf(ByDate, "<h2>%s</h2>\n", localizedStrings[LANG_LESS_1W_OLD]);
	    fprintf(ByDate, "<table><tbody>\n");
	}
    }
    fprintf(ByDate, "</tbody></table>\n");

    /*
     * Then dump RPMs less than two weeks old.
     */
    if (installed)
	fprintf(ByDate, "<h2>%s</h2>\n", localizedStrings[LANG_I_LESS_2W_OLD]);
    else
	fprintf(ByDate, "<h2>%s</h2>\n", localizedStrings[LANG_LESS_2W_OLD]);
    fprintf(ByDate, "<table><tbody>\n");
    while ((cur != NULL) && ((currentTime - cur->date) < (14 * 24 * 60 * 60))) {
	generateHtmlRpmRow(ByDate, cur, 0);
	if ((RSS != NULL) && (rdfcount++ < 20)) {
	    generateRSSItem(RSS, cur);
	}
	cur = cur->next;
	count++;
	if ((count % MAX_TABLE_LENGHT) == 0)
	    fprintf(ByDate, "</tbody></table>\n<table><tbody>\n");
	if (count > MAX_PAGE_LENGHT) {
	    count = 0;
	    fprintf(ByDate, "</tbody></table>\n");
	    fprintf(ByDate, "<a href=\"%d%s\">...</a>\n",
	            page, localizedStrings[LANG_BYDATE_HTML]);
	    generateHtmlFooter(ByDate);
	    fclose(ByDate);
	    fullPathNameNr(buf, sizeof(buf), dir, subdir,
	                   localizedStrings[LANG_BYDATE_HTML], page++);
	    if (rpm2htmlVerbose > 1) {
		printf("Dumping %s\n", buf);
	    }
	    ByDate = fopen(buf, "w");
	    if (ByDate == NULL) {
		fprintf(stderr, "Couldn't save to file %s: %s\n",
			buf, strerror(errno));
		if (RSS) {
		    fprintf(RSS, "  </channel>\n");
		    fprintf(RSS, "</rss>\n");
		    fclose(RSS);
		}
		return;
	    }
	    if (installed) {
		generateHtmlHeader(ByDate,
		          localizedStrings[LANG_SORTED_BY_IDATE], NULL);
		generateLinks(ByDate, installed);
		fprintf(ByDate, "<h1 align=center>%s</h1>\n",
		    localizedStrings[LANG_SORTED_BY_IDATE]);
	    } else {
		generateHtmlHeader(ByDate,
		          localizedStrings[LANG_SORTED_BY_CDATE], NULL);
		generateLinks(ByDate, installed);
		fprintf(ByDate, "<h1 align=center>%s</h1>\n",
		    localizedStrings[LANG_SORTED_BY_CDATE]);
	    }
	    if (installed)
		fprintf(ByDate, "<h2>%s</h2>\n", localizedStrings[LANG_I_LESS_2W_OLD]);
	    else
		fprintf(ByDate, "<h2>%s</h2>\n", localizedStrings[LANG_LESS_2W_OLD]);
	    fprintf(ByDate, "<table><tbody>\n");
	}
    }
    fprintf(ByDate, "</tbody></table>\n");

    /*
     * Then dump RPMs less than one month old.
     */
    if (installed)
	fprintf(ByDate, "<h2>%s</h2>\n", localizedStrings[LANG_I_LESS_1M_OLD]);
    else
	fprintf(ByDate, "<h2>%s</h2>\n", localizedStrings[LANG_LESS_1M_OLD]);
    fprintf(ByDate, "<table><tbody>\n");
    while ((cur != NULL) && ((currentTime - cur->date) < (30 * 24 * 60 * 60))) {
	generateHtmlRpmRow(ByDate, cur, 0);
	if ((RSS != NULL) && (rdfcount++ < 20)) {
	    generateRSSItem(RSS, cur);
	}
	cur = cur->next;
	count++;
	if ((count % MAX_TABLE_LENGHT) == 0)
	    fprintf(ByDate, "</tbody></table>\n<table><tbody>\n");
	if (count > MAX_PAGE_LENGHT) {
	    count = 0;
	    fprintf(ByDate, "</tbody></table>\n");
	    fprintf(ByDate, "<a href=\"%d%s\">...</a>\n",
	            page, localizedStrings[LANG_BYDATE_HTML]);
	    generateHtmlFooter(ByDate);
	    fclose(ByDate);
	    fullPathNameNr(buf, sizeof(buf), dir, subdir,
	                   localizedStrings[LANG_BYDATE_HTML], page++);
	    if (rpm2htmlVerbose > 1) {
		printf("Dumping %s\n", buf);
	    }
	    ByDate = fopen(buf, "w");
	    if (ByDate == NULL) {
		fprintf(stderr, "Couldn't save to file %s: %s\n",
			buf, strerror(errno));
		if (RSS) {
		    fprintf(RSS, "  </channel>\n");
		    fprintf(RSS, "</rss>\n");
		    fclose(RSS);
		}
		return;
	    }
	    if (installed) {
		generateHtmlHeader(ByDate,
		          localizedStrings[LANG_SORTED_BY_IDATE], NULL);
		generateLinks(ByDate, installed);
		fprintf(ByDate, "<h1 align=center>%s</h1>\n",
		    localizedStrings[LANG_SORTED_BY_IDATE]);
	    } else {
		generateHtmlHeader(ByDate,
		          localizedStrings[LANG_SORTED_BY_CDATE], NULL);
		generateLinks(ByDate, installed);
		fprintf(ByDate, "<h1 align=center>%s</h1>\n",
		    localizedStrings[LANG_SORTED_BY_CDATE]);
	    }
	    if (installed)
		fprintf(ByDate, "<h2>%s</h2>\n", localizedStrings[LANG_I_LESS_1M_OLD]);
	    else
		fprintf(ByDate, "<h2>%s</h2>\n", localizedStrings[LANG_LESS_1M_OLD]);
	    fprintf(ByDate, "<table><tbody>\n");
	}
    }
    fprintf(ByDate, "</tbody></table>\n");

    /*
     * Then dump RPMs more than one month old.
     */
    if (installed)
	fprintf(ByDate, "<h2>%s</h2>\n", localizedStrings[LANG_I_MORE_1M_OLD]);
    else
	fprintf(ByDate, "<h2>%s</h2>\n", localizedStrings[LANG_MORE_1M_OLD]);
    fprintf(ByDate, "<table><tbody>\n");
    while (cur != NULL) {
	generateHtmlRpmRow(ByDate, cur, 0);
	if ((RSS != NULL) && (rdfcount++ < 20)) {
	    generateRSSItem(RSS, cur);
	}
	cur = cur->next;
	count++;
	if ((count % MAX_TABLE_LENGHT) == 0)
	    fprintf(ByDate, "</tbody></table>\n<table><tbody>\n");
	if (count > MAX_PAGE_LENGHT) {
	    count = 0;
	    fprintf(ByDate, "</tbody></table>\n");
	    fprintf(ByDate, "<a href=\"%d%s\">...</a>\n",
	            page, localizedStrings[LANG_BYDATE_HTML]);
	    generateHtmlFooter(ByDate);
	    fclose(ByDate);
	    fullPathNameNr(buf, sizeof(buf), dir, subdir,
	                   localizedStrings[LANG_BYDATE_HTML], page++);
	    if (rpm2htmlVerbose > 1) {
		printf("Dumping %s\n", buf);
	    }
	    ByDate = fopen(buf, "w");
	    if (ByDate == NULL) {
		fprintf(stderr, "Couldn't save to file %s: %s\n",
			buf, strerror(errno));
		if (RSS) {
		    fprintf(RSS, "  </channel>\n");
		    fprintf(RSS, "</rss>\n");
		    fclose(RSS);
		}
		return;
	    }
	    if (installed) {
		generateHtmlHeader(ByDate,
		          localizedStrings[LANG_SORTED_BY_IDATE], NULL);
		generateLinks(ByDate, installed);
		fprintf(ByDate, "<h1 align=center>%s</h1>\n",
		    localizedStrings[LANG_SORTED_BY_IDATE]);
	    } else {
		generateHtmlHeader(ByDate,
		          localizedStrings[LANG_SORTED_BY_CDATE], NULL);
		generateLinks(ByDate, installed);
		fprintf(ByDate, "<h1 align=center>%s</h1>\n",
		    localizedStrings[LANG_SORTED_BY_CDATE]);
	    }
	    if (installed)
		fprintf(ByDate, "<h2>%s</h2>\n", localizedStrings[LANG_I_MORE_1M_OLD]);
	    else
		fprintf(ByDate, "<h2>%s</h2>\n", localizedStrings[LANG_MORE_1M_OLD]);
	    fprintf(ByDate, "<table><tbody>\n");
	}
    }
    fprintf(ByDate, "</tbody></table>\n");

    generateHtmlFooter(ByDate);
    fclose(ByDate);
    if (RSS) {
	fprintf(RSS, "  </channel>\n");
	fprintf(RSS, "</rss>\n");
	fclose(RSS);
    }
}

/*
 * Dump the full set of  RPM by Date HTML file.
 * One expect that the RPM files have been sorted by date.
 */
void dumpRpmByDate(rpmDataPtr list, int installed) {
    if (!rpm2html_dump_html) return;

    dumpDirRpmByDate(list, NULL, NULL, installed);
}

/*
 * Dump the RPM in a flat list sorted by Name.
 * One expect that the RPM files have been sorted by name.
 */
void dumpDirRpmByNameFlat(rpmDataPtr list, rpmDirPtr dir,
                      char *subdir, int installed) {
    FILE *ByName;
    rpmDataPtr cur;

    if (!rpm2html_dump_html) return;

    cur = list;

    fullPathName(buf, sizeof(buf), dir, subdir, localizedStrings[LANG_BYNAME_HTML]);
    if (rpm2htmlVerbose > 1) {
        printf("Dumping %s\n", buf);
    }

    ByName = fopen(buf, "w");
    if (ByName == NULL) {
        fprintf(stderr, "Couldn't save to file %s: %s\n",
	        buf, strerror(errno));
        return;
    }
    generateHtmlHeader(ByName, localizedStrings[LANG_SORTED_BY_NAME], NULL);
    generateLinks(ByName, installed);
    fprintf(ByName, "<h1 align=center>%s</h1>\n",
        localizedStrings[LANG_SORTED_BY_NAME]);
    if (!installed) generateColorIndicator(ByName);
    fprintf(ByName, "<table><tbody>\n");

    while (cur != NULL) {
	generateHtmlRpmArchRow(ByName, cur);
	if ((dir == NULL) && (subdir == NULL))
	    cur = cur->nextSoft;
	else
	    cur = cur->next;
    }

    fprintf(ByName, "</tbody></table>\n");
    generateHtmlFooter(ByName);
    fclose(ByName);
}

/*
 * Dump the RPM in a subtree by Name HTML file.
 * One expect that the RPM files have been sorted by name.
 */
void dumpDirRpmByName(rpmDataPtr list, rpmDirPtr dir,
                      char *subdir, int installed) {
    FILE *ByName;
    FILE *CurrName = NULL;
    rpmDataPtr cur;
    int i = 0;
    char last_letter = '\0', letter;

    if (!rpm2html_dump_html) return;

    if ((dir == NULL) && (subdir == NULL))
	for (cur = list; cur != NULL; cur = cur->nextSoft) i++;
    else
	for (cur = list; cur != NULL; cur = cur->next) i++;
    if (i < MAX_NAME_LIST_LENGHT) {
        dumpDirRpmByNameFlat(list, dir, subdir, installed);
        return;
    }
    i = 0;
    cur = list;

    fullPathName(buf, sizeof(buf), dir, subdir, localizedStrings[LANG_BYNAME_HTML]);
    if (rpm2htmlVerbose > 1) {
        printf("Dumping %s\n", buf);
    }

    ByName = fopen(buf, "w");
    if (ByName == NULL) {
        fprintf(stderr, "Couldn't save to file %s: %s\n",
	        buf, strerror(errno));
        return;
    }
    generateHtmlHeader(ByName, localizedStrings[LANG_SORTED_BY_NAME], NULL);
    generateLinks(ByName, installed);
    fprintf(ByName, "<h1 align=center>%s</h1>\n",
        localizedStrings[LANG_SORTED_BY_NAME]);

    while (cur != NULL) {
	letter = toupper(cur->name[0]);
	if (letter != last_letter) {
	    if (CurrName != NULL) {
		/*
		 * Finish the previous file.
		 */
		fprintf(CurrName, "</tbody></table>\n");
		generateHtmlFooter(CurrName);
		fclose(CurrName);
	    }
	    if (i != 0)
		fprintf(ByName, "<p><a href=\"%c%s\">%d %s %c</a></p>\n",
		        last_letter, localizedStrings[LANG_BYNAME_HTML],
			i, localizedStrings[LANG_BEGINNING_LETTER],
			last_letter);
	    fullPathNameLr(buf, sizeof(buf), dir, subdir,
	                   localizedStrings[LANG_BYNAME_HTML], letter);
	    if (rpm2htmlVerbose > 1) {
		printf("Dumping %s\n", buf);
	    }

	    CurrName = fopen(buf, "w");
	    if (CurrName == NULL) {
		fprintf(stderr, "Couldn't save to file %s: %s\n",
			buf, strerror(errno));
		return;
	    }
	    snprintf(buf, sizeof(buf), "%s %c", localizedStrings[LANG_BEGINNING_LETTER],
	            letter);
	    generateHtmlHeader(CurrName, buf, NULL);
	    generateLinks(CurrName, installed);
	    fprintf(CurrName, "<h1 align=center>%s</h1>\n", buf);
	    if (!installed) generateColorIndicator(CurrName);
	    fprintf(CurrName, "<table><tbody>\n");
	    i = 0;
	}
	i++;
	if ((i % MAX_TABLE_LENGHT) == 0)
	    fprintf(CurrName, "</tbody></table>\n<table><tbody>\n");
	last_letter = letter;
	generateHtmlRpmArchRow(CurrName, cur);
	if ((dir == NULL) && (subdir == NULL))
	    cur = cur->nextSoft;
	else
	    cur = cur->next;
    }

    if (i != 0)
	fprintf(ByName, "<p><a href=\"%c%s\">%d %s %c</a></p>\n",
		last_letter, localizedStrings[LANG_BYNAME_HTML],
		i, localizedStrings[LANG_BEGINNING_LETTER],
		last_letter);
    if (CurrName != NULL) {
	/*
	 * Finish the previous file.
	 */
	fprintf(CurrName, "</tbody></table>\n");
	generateHtmlFooter(CurrName);
	fclose(CurrName);
    }
    generateHtmlFooter(ByName);
    fclose(ByName);
}

/*
 * Dump the full set of  RPM by Name HTML file.
 * One expect that the RPM files have been sorted by name.
 */
void dumpRpmByName(rpmDataPtr list, int installed) {
    if (!rpm2html_dump_html) return;

    dumpDirRpmByName(list, NULL, NULL, installed);
}

/*
 * Dump the index for a directory and its subdirectories.
 */
rpmDataPtr dumpDirIndex(rpmDirPtr dir, rpmSubdirPtr tree, rpmDataPtr list) {
    FILE *html;
    int i;

    if (!rpm2html_dump_html) return(list);
    if ((dir != NULL) && (!dir->html)) return(list);

    if (tree->htmlpath[0] != '\0')
	snprintf(buf, sizeof(buf), "%s/%s/%s", rpm2html_dir, tree->htmlpath,
		localizedStrings[LANG_INDEX_HTML]);
    else
	snprintf(buf, sizeof(buf), "%s/%s", rpm2html_dir,
		localizedStrings[LANG_INDEX_HTML]);

    if (rpm2htmlVerbose > 1) {
	printf("Dumping %s\n", buf);
    }

    html = fopen(buf, "w");
    if (html == NULL) {
        fprintf(stderr, "Couldn't save to file %s: %s\n",
	        buf, strerror(errno));
        return(list);
    }
    if (tree->htmlpath[0] != '\0')
        snprintf(buf, sizeof(buf), "%s : %s", dir->name, tree->htmlpath);
    else
        snprintf(buf, sizeof(buf), "%s", dir->name);
    generateHtmlHeader(html, buf, NULL);
    generateLinks(html, 0);
    if (((tree->rpmpath == NULL) || (tree->rpmpath[0] == '\0')) &&
        (dir->build_tree))
        snprintf(buf, sizeof(buf), "<a href=\"Root%s\">%s</a>",
	        localizedStrings[LANG_TREE_HTML],
		localizedStrings[LANG_INDEX_HTML]);
    fprintf(html, "<h1 align=center>%s</h1>\n", buf);

    fprintf(html, "<ul>\n");
    fprintf(html, "<li>\n%s <a href=\"%s\">%s</a>\n",
            localizedStrings[LANG_LIST],
	    localizedStrings[LANG_GROUP_HTML],
	    localizedStrings[LANG_INDEX_GROUP]);
    fprintf(html, "<li>\n%s <a href=\"%s\">%s</a>\n",
            localizedStrings[LANG_LIST],
	    localizedStrings[LANG_BYDATE_HTML],
	    localizedStrings[LANG_INDEX_CREATION]);
    fprintf(html, "<li>\n%s <a href=\"%s\">%s</a>\n",
            localizedStrings[LANG_LIST],
	    localizedStrings[LANG_BYNAME_HTML],
	    localizedStrings[LANG_INDEX_NAME]);
    fprintf(html, "<li>\n%s <a href=\"%s\">%s</a>\n",
            localizedStrings[LANG_LIST],
	    localizedStrings[LANG_VENDOR_HTML],
	    localizedStrings[LANG_INDEX_VENDOR]);
    fprintf(html, "<li>\n%s <a href=\"%s\">%s</a>\n",
            localizedStrings[LANG_LIST],
	    localizedStrings[LANG_DISTRIB_HTML],
	    localizedStrings[LANG_INDEX_DISTRIB]);
    fprintf(html, "</ul>\n\n");
    if (((tree->rpmpath == NULL) || (tree->rpmpath[0] == '\0')) &&
        (dir->build_tree)) {
	fprintf(html, "<h3>%s</h3>\n",
	    localizedStrings[LANG_BROWSE_TREE]);
	if (tree->color)
	    fprintf(html, "<blockquote style=\"background : %s\">\n",
	            tree->color);
	else
	    fprintf(html, "<blockquote>\n");
	fprintf(html, "<p><strong><a href=\"%s\">",
	        localizedStrings[LANG_TREE_HTML]);
	fprintf(html,
	        "<img src=\"%s/dir.png\" alt=\"Directory\" border=0>",
	            rpm2html_url);
	fprintf(html, " ..</a></strong></p></blockquote>\n");
    }

    generateHtmlTree(html, tree, 0, 1);
    
    fprintf(html, "<table bgcolor=\"%s\"><tbody>\n<tr>\n",
	    dir->color);
    if (dir->name == NULL)
	fprintf(html, "<td bgcolor=\"%s\" colspan=2><a href=\"%s\">%s</a></td>\n",
		dir->color, dir->ftp, dir->ftp);
    else
	fprintf(html, "<td bgcolor=\"%s\" colspan=2><a href=\"%s\">%s</a></td>\n",
		dir->color, dir->ftp, dir->name);
    fprintf(html, "</tr>\n");
    if ((dir->ftpsrc != NULL) && (strcmp(dir->ftp, dir->ftpsrc)))
	fprintf(html, 
	    "<tr><td bgcolor=\"%s\">%s:</td><td bgcolor=\"%s\"><a href=\"%s\">%s</a></td></tr>\n",
	    dir->color, localizedStrings[LANG_SOURCES_REPOSITORY],
	    dir->color, dir->ftpsrc, dir->ftpsrc);
    if (dir->nb_mirrors > 0)
	fprintf(html,
	    "<tr><td bgcolor=\"%s\">%s:</td><td bgcolor=\"%s\"><a href=\"%s\">%s</a></td></tr>\n",
	    dir->color, localizedStrings[LANG_LOCAL_MIRROR],
	    dir->color, dir->mirrors[0], dir->mirrors[0]);
    if (dir->nb_mirrors > 1) {
	fprintf(html, "<tr><td bgcolor=\"%s\" colspan=2>%s</td></tr>\n",
		dir->color, localizedStrings[LANG_MIRRORS]);
	for (i = 1;i < dir->nb_mirrors;i++) {
	    fprintf(html,
		"<tr><td bgcolor=\"%s\" colspan=2><a href=\"%s\">%s</a></td></tr>\n",
		dir->color, dir->mirrors[i], dir->mirrors[i]);
	}
    }
    fprintf(html, "</tbody></table>\n");

    generateHtmlFooter(html);
    fclose(html);

    rpmDistribSort(&list, 0);
    dumpDirRpmByDistribs(list, dir, tree->htmlpath, 0);
    rpmGroupSort(&list, 0);
    dumpDirRpmByGroups(list, dir, tree->htmlpath, 0);
    rpmVendorSort(&list, 0);
    dumpDirRpmByVendors(list, dir, tree->htmlpath, 0);
    rpmDateSort(&list, 0);
    dumpDirRpmByDate(list, dir, tree->htmlpath, 0);
    rpmNameSort(&list, 0);
    dumpDirRpmByName(list, dir, tree->htmlpath, 0);
    return(list);
}

/*
 * Dump a real tree subdir title with links to parents and whole tree.
 */

void rpmDumpHtmlRealTreeTitle(FILE *subdir, rpmDirPtr dir, char *buffer) {
    char buf[3000];
    char tmp[1000];
    char path[1000];
    char loc[200];
    int i = 0, j = 0, k = 0; /* indexes for buffer, loc and path resp. */
    int l = 0; /* index for directory depth */

    /* link for the root */
    snprintf(buf, sizeof(buf), "<a href=\"%s\">/</a>",
	     localizedStrings[LANG_TREE_HTML]);
    while (buffer[i] != '\0') {
        if (buffer[i] == '_') {
	    loc[j] = '\0';
	    path[k++] = '_';
	    path[k] = '\0';
	    if (l++)
	    	snprintf(tmp, sizeof(tmp), " <a href=\"%s%s\">/%s</a>", path,
	            	localizedStrings[LANG_TREE_HTML], loc);
	    else
	    	snprintf(tmp, sizeof(tmp), " <a href=\"%s%s\">%s</a>", path,
	            	localizedStrings[LANG_TREE_HTML], loc);
            strcat(buf, tmp);
	    j = 0;
	} else {
	    loc[j++] = buffer[i];
	    path[k++] = buffer[i];
	}
        i++;
    }

    fprintf(subdir, "<h1 align=center><a href=\"Root%s\">%s</a> : %s</h1>\n",
            localizedStrings[LANG_TREE_HTML], dir->name, buf);
}

/*
 * Dump a real tree subdir.
 */

void rpmDumpHtmlRealTree(FILE *html, rpmDirPtr dir, rpmRealDirPtr tree,
                         char *buffer, int index) {
    FILE *subdir;
    char buf[1000];
    int has_subdir = 0;
    int has_file = 0;
    int i;

    if (!rpm2html_dump_html) return;
    if ((dir != NULL) && (!dir->html)) return;

    if (dir->subdir)
	snprintf(buf, sizeof(buf), "%s/%s/%s%s", rpm2html_dir,
	        dir->subdir, buffer,
		localizedStrings[LANG_TREE_HTML]);
    else
	snprintf(buf, sizeof(buf), "%s/%s%s", rpm2html_dir, buffer,
	        localizedStrings[LANG_TREE_HTML]);

    if (rpm2htmlVerbose > 1) {
	printf("Dumping %s\n", buf);
    }

    subdir = fopen(buf, "w");
    if (subdir == NULL) {
        fprintf(stderr, "Couldn't save to file %s: %s\n",
	        buf, strerror(errno));
        return;
    }


    generateHtmlHeader(subdir, buf, NULL);
    generateLinks(subdir, 1);
    rpmDumpHtmlRealTreeTitle(subdir, dir, buffer);

    for (i = 0;i < tree->nb_elem;i++) {
        if (tree->flags[i] & RPM_ELEM_DIRECTORY) {
	    if (!has_subdir)
		fprintf(html, "<ul>\n");
	    has_subdir++;
	    fprintf(html, "<li> <a href=\"%s%s_%s\">%s</a>\n",
	            buffer, tree->names[i],
		    localizedStrings[LANG_TREE_HTML], tree->names[i]);
	    snprintf(&buffer[index], 500, "%s_", tree->names[i]);
	    rpmDumpHtmlRealTree(html, dir, tree->elems[i],
	                        buffer, strlen(buffer));
	    buffer[index] = '\0';
	    if (!has_file)
		fprintf(subdir, "<ul>\n");
	    has_file++;
	    fprintf(subdir, "<li><a href=\"%s%s_%s\">",
	            buffer, tree->names[i],
		    localizedStrings[LANG_TREE_HTML]);
	    fprintf(subdir,
	        "<img src=\"%s/dir.png\" alt=\"subdir\" border=0>",
	            rpm2html_url);
	    fprintf(subdir, " %s</a>\n", tree->names[i]);

	} else {
	    if (!has_file)
		fprintf(subdir, "<ul>\n");
	    has_file++;
	    fprintf(subdir, "<li>");
	    generateHtmlRpmAnchor(subdir, tree->elems[i]);
	    fprintf(subdir, "%s</a>\n", tree->names[i]);
	} 
    }
    if (has_subdir)
	fprintf(html, "</ul>\n");
    if (has_file)
	fprintf(subdir, "</ul>\n");
    generateHtmlFooter(subdir);
    fclose(subdir);
}

/*
 * Dump a real tree root.
 */

void rpmDumpHtmlRealRoot(rpmDirPtr dir) {
    FILE *html;
    char buf[1000];
    rpmRealDirPtr tree;

    if (!rpm2html_dump_html) return;
    if ((dir != NULL) && (!dir->html)) return;

    if (dir == NULL) return;
    tree = dir->root;
    if (tree == NULL) return;
    if (dir->subdir)
	snprintf(buf, sizeof(buf), "%s/%s/Root%s", rpm2html_dir,
	        dir->subdir, localizedStrings[LANG_TREE_HTML]);
    else
	snprintf(buf, sizeof(buf), "%s/Root%s", rpm2html_dir,
	        localizedStrings[LANG_TREE_HTML]);

    if (rpm2htmlVerbose > 1) {
	printf("Dumping %s\n", buf);
    }

    html = fopen(buf, "w");
    if (html == NULL) {
        fprintf(stderr, "Couldn't save to file %s: %s\n",
	        buf, strerror(errno));
        return;
    }

    snprintf(buf, sizeof(buf), "%s", dir->name);

    generateHtmlHeader(html, buf, NULL);
    generateLinks(html, 1);
    fprintf(html, "<h1 align=center>%s</h1>\n", buf);

    buf[0] = 0;
    rpmDumpHtmlRealTree(html, dir, tree, buf, 0);
    
    generateHtmlFooter(html);
    fclose(html);
}

