/*

 Copyright (C) Gordon Matzigkeit <gord@fig.org>, Loic Dachary
 <loic@gnu.org>, 2001

 This program is free software; you can redistribute it and/or
 modify it under the terms of the GNU General Public License
 as published by the Free Software Foundation; either version 2
 of the License, or (at your option) any later version.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this program; if not, write to the Free Software
 Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

*/

/*
  sf_sync_www.c - Allow unprivileged users to do a web update on www.gnu.org

  In CVSROOT/loginfo file
  ALL     (date; cat; (sleep 2; /subversions/sourceforge/bin/sf_sync_www %{s} ) &\ ) >> /var/log/sf_sync_www 2>&1

  For test purpose:
  FAKE=yes ./sf_sync_www ' bla bla'
  FAKE=yes ./sf_sync_www 'foo/bar bla bla'
  FAKE=yes ./sf_sync_www 'f'\''oo bla bla'
  FAKE=yes ./sf_sync_www 'f'\''oo - New directory'
  FAKE=yes ./sf_sync_www 'bla - Imported sources'

  The idea is that it runs a cvs update -l (to prevent recursion) in the 
  directory where the commit was done. Since the command will be called
  once for each directory where a commit did some action there is no
  need for recursion. In the case of an import command this does not
  hold and a recursion must always be done since there is only one
  call to the script for a whole imported tree (this happens when the
  argument contains the Imported source string).

  The %{s} argument is a single argument that lists the directory and all
  the files involved. As a special case if the directory was added the file
  list is replaced by '- New directory'. This is lame since adding the files
  -, New and directory will produce the same effect, but it's unlikely. The
  same applies when a whole source tree is imported using cvs import in
  which case the file list is replaced by '- Imported sources'.

  There are three cases to take in account (topdir is the absolute path
  of the directory in which the CVS tree was extracted, subdirectory is
  the directory given in argument):
  - commit that modify the top level directory files
    cd topdir ; cvs update -l 
  - commit that adds a new directory or that import a whole source tree
    cd topdir ; cvs update 'subdirectory'
  - commit that modify files in a subdirectory
    cd topdir/subdirectory ; cvs update -l

  In order to prevent security compromision the directory name is quoted.

  Originaly by
  Gordon Matzigkeit <gord@fig.org>, 2000-11-28

  Update CVS_COMMANDS to reduce noise
  Loic Dachary <loic@gnu.org>, 2001-02-26

  Modify to allow generic call from loginfo file in an efficient way
  Loic Dachary <loic@gnu.org>, 2001-03-10
*/

#define REPOSITORY "/webcvs"
#define WWW_HOSTNAME "gnudist.gnu.org"
#define WWW_DIRECTORY "/home/www/html"
#define RSH_PROGRAM "/usr/bin/ssh"
#define CVS_COMMAND "CVS_RSH=ssh cvs -q -z3 update"

#include <ctype.h>
#include <string.h>
#include <stdio.h>

#include <error.h>
#include <errno.h>

#include <sys/stat.h>
#include <unistd.h>
#include <stdlib.h>


/*
 * Return a malloc'ed copy of the string argument with all '
 * substituted by '\'' and a trailing and leading '.
 * For instance : foo -> 'foo'
 *                fib'ou -> 'fib'\''ou'
 */
char* quote(char* string)
{
  int i;
  int length = strlen(string);
  int count = 0;
  char* out;
  for(i = 0; i < length; i++)
    if(string[i] == '\'')
      count++;
  /* +1 for null at end, (count * 3) for '\'', +2 for leading ' and trailing ' */
  out = (char*)malloc(length + 1 + (count * 3) + 2);
  if(count > 0) {
    char* from_p = string;
    char* to_p = out;
    *to_p++ = '\'';
    while(*from_p) {
      if(*from_p == '\'') {
	strcpy(to_p, "'\\''");
	to_p += 4;
	from_p++;
      } else {
	*to_p++ = *from_p++;
      }
    }
    *to_p++ = '\'';
    *to_p = '\0';
  } else {
    sprintf(out, "'%s'", string);
  }
  return out;
}

int
main (int argc, char **argv)
{
  char* fake = getenv("FAKE");
  char* program;
  int i, cvslen;
  char* unquoted_directory = 0;
  char* directory = 0;
  int directory_length = 0;
  int files = 0;

  i = geteuid ();
  if (setreuid (i, i))
    error (1, errno, "Cannot really assume effective userid");

  if (argc != 2)
    error (1, 0, "You must specify exactly one argument");

  {
    char* tmp = argv[1];

    /*
     * First comes the relative directory name or empty string if current.
     * directory = 0 if no subdirectory
     * directory = relative path if subdirectory
     */
    directory = tmp;
    if((tmp = strchr(tmp, ' '))) {
      *tmp = '\0';
      if(tmp - directory <= 0) {
	directory = 0;
      } else {
	unquoted_directory = directory;
	directory = quote(directory);
	directory_length = strlen(directory);
      }
      tmp++;
    } else {
      error(1, 0, "Failed to find directory in first argument");
    }

    if(!strcmp(tmp, "- New directory") || !strcmp(tmp, "- Imported sources")) {
      /*
       * If this is a new directory creation or an import command,
       * there are no files to collect in the argument list.
       */
    } else {
      /*
       * There are files to collect in the argument list.
       */
      files = 1;
    }
  }

  /* Check that the local directory exists. */
  if(!fake && unquoted_directory) {
    int unquoted_directory_length = strlen(unquoted_directory);
    i = strlen (REPOSITORY);
    {
      struct stat stbuf;
      char localdir[i + 1 + unquoted_directory_length + 1];

      strcpy (localdir, REPOSITORY);
      if (localdir[i - 1] != '/')
	localdir[i ++] = '/';
      strcpy (localdir + i, unquoted_directory);

      /* Check to see that the local unquoted_directory actually exists. */
      if (stat (localdir, &stbuf) ||
	  (! S_ISDIR (stbuf.st_mode) && (errno = ENOTDIR)))
	error (1, errno, "Cannot find `%s'", localdir);
    }
  }

  /* Now build up the command. */
  i = strlen (WWW_DIRECTORY);
  cvslen = strlen (CVS_COMMAND);
  program = fake ? "/bin/echo" : RSH_PROGRAM;
  {
    char cmd[i + directory_length + cvslen + 128];
    char *new_argv[] = { NULL, WWW_HOSTNAME, cmd, NULL };
    char *new_envp[] = { NULL };

    if(directory && !files) {
      sprintf(cmd, "cd %s && ( %s -d %s )",
	      WWW_DIRECTORY,
	      CVS_COMMAND,
	      directory);
    } else if(!directory && files) {
      sprintf(cmd, "cd %s && ( %s -l )",
	      WWW_DIRECTORY,
	      CVS_COMMAND);
    } else if(directory && files) {
      sprintf(cmd, "cd %s/%s && ( %s -l )",
	      WWW_DIRECTORY,
	      directory,
	      CVS_COMMAND);
    } else {
      error (1, 0, "Unexpected null directory and files");
    }

    /* Pick out the name of the program. */
    new_argv[0] = strrchr (program, '/');
    if (new_argv[0])
      new_argv[0] ++;
    else
      new_argv[0] = program;

    /* Execute the rsh command. */
    execve (program, new_argv, new_envp);
  }

  error (1, errno, "Cannot exec `%s'", RSH_PROGRAM);
  /* NOTREACHED */
  return 1;
}
