#!/usr/bin/perl
# This file is part of the Savane project
# <http://gna.org/projects/savane/>
#
# $Id: sv_sync_www.pl,v 1.3 2004/01/31 00:56:17 yeupou Exp $
# -w

# Copyright (C) 2003 Vincent Caron <v.caron@zerodeux.net>
#
# Based on sv_sync_www.c
# Copyright (C) Gordon Matzigkeit <gord@fig.org>, Loic Dachary
# <loic@gnu.org>, 2001
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.


#  sf_sync_www.pl - Allow unprivileged users to do a web update on www.[non]gnu.org

#  In CVSROOT/loginfo file
#  ALL     ( /subversions/sourceforge/bin/sv_sync_www %s & ) >> /var/log/sf_sync_www 2>&1

#  For test purpose:
#  FAKE=yes ./sf_sync_www ' bla bla'
#  FAKE=yes ./sf_sync_www 'foo/bar bla bla'
#  FAKE=yes ./sf_sync_www 'f'\''oo bla bla'
#  FAKE=yes ./sf_sync_www 'f'\''oo - New directory'
#  FAKE=yes ./sf_sync_www 'bla - Imported sources'

#  The idea is that it runs a cvs update -l (to prevent recursion) in the 
#  directory where the commit was done. Since the command will be called
#  once for each directory where a commit did some action there is no
#  need for recursion. In the case of an import command this does not
#  hold and a recursion must always be done since there is only one
#  call to the script for a whole imported tree (this happens when the
#  argument contains the Imported source string).
#
#  The %{s} argument is a single argument that lists the directory and all
#  the files involved. As a special case if the directory was added the file
#  list is replaced by '- New directory'. This is lame since adding the files
#  -, New and directory will produce the same effect, but it's unlikely. The
#  same applies when a whole source tree is imported using cvs import in
#  which case the file list is replaced by '- Imported sources'.
#
#  There are three cases to take in account (topdir is the absolute path
#  of the directory in which the CVS tree was extracted, subdirectory is
#  the directory given in argument):
#  - commit that modify the top level directory files
#    cd topdir ; cvs update -l 
#  - commit that adds a new directory or that import a whole source tree
#    cd topdir ; cvs update 'subdirectory'
#  - commit that modify files in a subdirectory
#    cd topdir/subdirectory ; cvs update -l
#
#  In order to prevent security compromision the directory name is quoted.
#
#  Originaly by
#  Gordon Matzigkeit <gord@fig.org>, 2000-11-28
#
#  Update CVS_COMMANDS to reduce noise
#  Loic Dachary <loic@gnu.org>, 2001-02-26
#
#  Modify to allow generic call from loginfo file in an efficient way
#  Loic Dachary <loic@gnu.org>, 2001-03-10
#
#  Conversion from Perl to C
#  Vincent Caron <v.caron@zerodeux.net>, 2003-08-12

use strict;
use POSIX qw(strftime);


my $webhost  = 'gnudist.gnu.org';
my $webroot  = '/home/www/html';
my $cvs_up   = 'CVS_RSH=ssh cvs -q -z3 update';
my $fake     = defined $ENV{FAKE};


# Return a copy of the string argument with all '
# substituted by '\'' and a trailing and leading '.
# For instance : foo    -> 'foo'
#                fib'ou -> 'fib'\''ou'
sub quote {
    local $_ = shift;
    s/'/'\\''/g;
    "'$_'";
}

# Execute a command given an input event.
#
# Input: the '%s' argument as substitued by CVSROOT/loginfo.
# Output: command status (0 is success).
#
sub do_event {
    my $event = shift;
    my $path = '';
    my @files = split / /, $event;

    if ($event =~ / - New directory$/ or
        $event =~ / - Imported sources$/) {
        $path = $files[0];
        @files = ();
    } else {
        $path = shift @files;
    }

    my $command = '';
    if (@files > 0) {
        $path = "/$path" if $path ne '';
        $command = "cd ". quote("$webroot$path"). " && ( $cvs_up -l )";
    } else {
        if ($path eq '') {
            print STDERR "error: unexpected event (no folder, no files).\n";
            return 1;
        }
        $command = "cd $webroot && ( $cvs_up -d ". quote($path). " )";
    }

    $command = "ssh $webhost \"$command\"";
    return (print("$command\n"), 0) if $fake;

    # Log event
    my $date = strftime "%Y-%m-%d %H:%M:%S", localtime(time());
    print "$date - $event\n";

    # Launch command
    if (not open(COM, "$command 2>&1 |")) {
        print "$date - error: $!\n";
        return 1;
    }

    # Log command output (stdout + stderr)
    while (<COM>) {
        print "$date - $_";
    }
    return close(COM) ? 0 : 1;
}


# main

$< = $>;  # set real to effective uid

if (@ARGV != 1) {
    print STDERR "usage:\n  sv_sync_www.pl 'files'\n";
    exit 1;
}

exit(do_event $ARGV[0]);
