/* 
 * main.c
 * the main control function for smtpmail 0.3
 * (c) 2003 Christian Toepp <c.toepp@gmx.de>
 */

/*
 * This file is part of smtpmail.
 * 
 * smtpmail is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Foobar is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with Foobar; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 */


#include <stdio.h>
#include <stdlib.h>
#include <pwd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <string.h>

#include "types.h"

int QUITE = 0;

extern char *create_mail (mail_opt * opts);
extern int send_mail (mail_opt * opts);
/*
 * prints the usage of the app and a little help
 */
static int
usage (const char *call)
{
  fprintf (stderr, "smtpmail version %s\n", VERSION);
  fprintf (stderr, "a tool to send your mail over an remote server\n");
  fprintf (stderr, "(c) 2003 Christian Toepp <c.toepp@gmx.de>\n");
  fprintf (stderr, "Licensed by the GPL, see COPYING for details\n\n");
  fprintf (stderr, "Usage: %s -t recipient@host -[hfspucbrv] [file]\n", call);
  fprintf (stderr, "where the options are:\n");
  fprintf (stderr, " -h smtpserver\n");
  fprintf (stderr, "    if not given, localhost will be used\n");
  fprintf (stderr, " -f from, your address\n");
  fprintf (stderr, "    if not given, it will be $login@$hostname\n");
  fprintf (stderr, " -s subject of the mail\n");
  fprintf (stderr, " -p password to athenticate by the server\n");
  fprintf (stderr, " -u username to autenticate by the server\n");
  fprintf (stderr, "    if not given and password is set, your\n");
  fprintf (stderr, "    address will be used\n");
  fprintf (stderr, " -t to, the recipient of the mail. can be a\n");
  fprintf (stderr, "    coma separated list (REQUIRED)\n");
  fprintf (stderr, " -c cc, send a carbon copy to this address\n");
  fprintf (stderr, "    can be a coma separated list\n");
  fprintf (stderr, " -b bcc, send a blind carbon copy to this address\n");
  fprintf (stderr, "    can be a coma separated list\n");
  fprintf (stderr, " -r add this reply-to address to the mail\n");
  fprintf (stderr, " -a add this file as attachment to the mail\n");
  fprintf (stderr, "    can be a coma separated list\n");
  fprintf (stderr, " -q quite mode (for scripting)\n");
  fprintf (stderr, " -v verbose output\n\n");
  fprintf (stderr, "file can be a existing textfile for the mail body\n");
  fprintf (stderr, "or - to read from STDIN\n\n");

  return 1;
}

/*
 * split a coma separated list into a array of tokens
 * and returns the count of tokens
 */

static unsigned int
split_opt (const char *option, char ***ret_opt)
{
  char buff[255];
  unsigned int ret = 0;
  int old = 0, i;
  for (i = 0; i < (int) strlen (option); i++)
    {
      if (option[i] == ',')
	{			/* end of the old string */
	  buff[i - old] = '\0';
	  old = i + 1;
	  ret++;
	  if((*ret_opt = realloc (*ret_opt, ret * sizeof (char *))) == NULL){
	    fprintf(stderr, "split_opt failed\n");
	    _exit(1);
	  }
	  (*ret_opt)[ret - 1] = (char *) malloc (strlen (buff));
	  sprintf ((*ret_opt)[ret - 1], "%s", buff);
	}
      else
	{
	  buff[i - old] = option[i];
	}
    }
  buff[i - old] = '\0';
  ret++;
  if((*ret_opt = realloc (*ret_opt, ret * sizeof (char *))) == NULL){
    fprintf(stderr, "split_opt failed\n");
    _exit(1);
  }
  (*ret_opt)[ret - 1] = (char *) malloc (strlen (buff));
  sprintf ((*ret_opt)[ret - 1], "%s", buff);
  return ret;
}

/*
 * creates a valid mailaddress from the users login
 * and the hostname
 */
static char *
get_sender (void)
{
  char *buff;
  char host[255];
  struct passwd *pass;
  buff = (char *) malloc (256);
  pass = getpwuid (getuid ());
  gethostname (host, 254);
  sprintf (buff, "%s@%s", pass->pw_name, host);
  return buff;
}

static int
test_files(unsigned int attc, char *attv[])
{
  int i;
  struct stat st;
  for(i = 0; i < attc; i++)
    {
      if(stat(attv[i], &st))
	{
	  fprintf (stderr, "cannot find the file %s\n", attv[i]);
	  return 1;
	}
    }
  return 0;
}

/*
 * extracts the options from the parameters
 */
static int
get_options (struct mail_opt *opts, int argc, char *argv[])
{
  int c;
  size_t f;
  int ret = 0;

  while ((c = getopt (argc, argv, "h:f:s:p:u:t:c:b:r:a:vq")) != -1)
    {
      switch (c)
	{
	case 'h':		/* mailserver */
	  f = strlen (optarg) + 1;
	  if ((opts->host = (char *) realloc (opts->host, f)) == NULL)
	    {
	      fprintf (stderr, "getopt(-f) failed\n");
	      return -1;
	    }
	  snprintf (opts->host, f, "%s", optarg);
	  break;
	case 'f':		/* from-address */
	  f = strlen (optarg) + 1;
	  if ((opts->sender = (char *) realloc (opts->sender, f)) == NULL)
	    {
	      fprintf (stderr, "getopt(-f) failed\n");
	      return -1;
	    }
	  snprintf (opts->sender, f, "%s", optarg);
	  break;
	case 's':		/* subject */
	  f = strlen (optarg) + 1;
	  if ((opts->subj = (char *) realloc (opts->subj, f)) == NULL)
	    {
	      fprintf (stderr, "getopt(-s) failed\n");
	      return -1;
	    }
	  snprintf (opts->subj, f, "%s", optarg);
	  break;
	case 'p':		/* server passord */
	  f = strlen (optarg) + 1;
	  if ((opts->pass = (char *) realloc (opts->pass, f)) == NULL)
	    {
	      fprintf (stderr, "getopt(-p) failed\n");
	      return -1;
	    }
	  snprintf (opts->pass, f, "%s", optarg);
	  break;
	case 'u':		/* server login */
	  f = strlen (optarg) + 1;
	  if ((opts->user = (char *) realloc (opts->user, f)) == NULL)
	    {
	      fprintf (stderr, "getopt(-u) failed\n");
	      return -1;
	    }
	  snprintf (opts->user, f, "%s", optarg);
	  break;
	case 't':		/* recipient */
	  opts->rcpt_count = split_opt (optarg, &(opts->rcpt_addr));
	  if (opts->rcpt_count > 0)
	    ret++;
	  break;
	case 'c':		/* carbon copy */
	  opts->cc_count = split_opt (optarg, &(opts->cc_addr));
	  break;
	case 'b':		/* blind carbon copy */
	  opts->bcc_count = split_opt (optarg, &(opts->bcc_addr));
	  break;
	case 'r':		/* reply-to */
	  f = strlen (optarg) + 1;
	  if ((opts->reply = (char *) realloc (opts->reply, f)) == NULL)
	    {
	      fprintf (stderr, "getopt(-r) failed\n");
	      return -1;
	    }
	  snprintf (opts->reply, f, "%s", optarg);
	  break;
	case 'a':
	  opts->att_count = split_opt (optarg, &(opts->att_names));
	  if(test_files(opts->att_count, opts->att_names))
	    return -1;
	  break;
	case 'v':		/* verbose */
	  opts->verbose++;
	  break;
	case 'q':
	  QUITE = 1;
	  break;
	}
    }
  if (opts->sender == NULL)
    opts->sender = get_sender ();
  if (opts->pass != NULL)
    {
      if (opts->user == NULL)
	opts->user = opts->sender;
    }
  if (opts->host == NULL)
    {
      opts->host = (char *) realloc (opts->host, strlen ("localhost"));
      snprintf (opts->host, strlen ("localhost"), "localhost");
    }
  if (ret == 1)
    return 0;
  return -1;
}

/*
 * read the mailbody
 */
static int
read_body (char *filename, char **body)
{
  struct stat st;
  FILE *fp;
  int mc, mi;

  if (filename == NULL || strcmp (filename, "-") == 0)
    {				/* read from STDIN */
      if(QUITE == 0)
	fprintf (stdout,
		 "please type your message text, ending with EOF (most Ctrl-D):\n");
      mi = 0;
      while ((mc = getchar ()) != EOF)
	{
	  *body = (char *) realloc (*body, mi + 1);
	  (*body)[mi] = mc;
	  mi++;
	}
      (*body)[mi - 1] = '\0';
    }
  else
    {
      if (stat (filename, &st) < 0)
	{
	  fprintf (stderr, "cannot find/access the file %s\n", filename);
	  return -1;
	}
      *body = (char *) realloc (*body, st.st_size);
      if (*body == NULL)
	{
	  fprintf (stderr, "cannot allocate memory for file %s\n", filename);
	  return -1;
	}
      fp = fopen (filename, "r");
      if (fp == NULL)
	{
	  fprintf (stderr, "cannot open file %s\n", filename);
	  return -1;
	}
      mi = 0;
      while ((mc = fgetc (fp)) != EOF)
	{
	  (*body)[mi] = mc;
	  mi++;
	}
      (*body)[mi] = '\0';
    }
  return 0;
}

/*
 * fill the struct mail_opt with default values
 */
static int
init_opts (struct mail_opt *opts)
{
  opts->host = NULL;
  opts->sender = NULL;
  opts->user = NULL;
  opts->pass = NULL;
  opts->rcpt_count = 0;
  opts->rcpt_addr = NULL;
  opts->cc_count = 0;
  opts->cc_addr = NULL;
  opts->bcc_count = 0;
  opts->bcc_addr = NULL;
  opts->msg = NULL;
  opts->reply = NULL;
  opts->subj = NULL;
  opts->verbose = 0;
  opts->att_count = 0;
  opts->att_names = NULL;
  return 0;
}

/*
 * releases the allocated memory of the mail options
 */
static int
free_opts (struct mail_opt *opts)
{
  unsigned int i;
  free (opts->host);
  free (opts->sender);
  free (opts->user);
  free (opts->pass);
  for (i = 0; i < opts->rcpt_count; i++)
    {
      free (opts->rcpt_addr[i]);
    }
  for (i = 0; i < opts->cc_count; i++)
    {
      free (opts->cc_addr[i]);
    }
  for (i = 0; i < opts->bcc_count; i++)
    {
      free (opts->bcc_addr[i]);
    }
  free (opts->msg);
  free (opts->reply);
  free (opts->subj);
  return 0;
}

int
main (int argc, char *argv[])
{
  struct mail_opt opts;

  init_opts (&opts);
  if (get_options (&opts, argc - 1, argv))
    return usage (argv[0]);
  if (read_body (argv[argc - 1], &opts.msg))
    return usage (argv[0]);
  opts.msg = create_mail (&opts);
  send_mail (&opts);
  free_opts (&opts);
  return 0;
}
