# GNU Solfege - ear training for GNOME
# Copyright (C) 2000, 2001, 2002, 2003, 2004  Tom Cato Amundsen
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

try:
    app_running == True
except:
    import sys
    sys.path.append("src")
    sys.path.append(".")
    import i18n
    i18n.setup_srcdir()


import gtk
import string, re, math
import mpd
import const
import random
import types
import soundcard, cfg

def play_tone(midi_int):
    soundcard.play_note(cfg.get_int('config/preferred_instrument'),
                        4, 0, midi_int,
                        cfg.get_int('config/preferred_instrument_velocity'))

def int_to_intervalname(i, shortname=None, updown=None):
    if shortname:
        n = const.short_interval_name[abs(i)]
    else:
        n = const.int_interval[abs(i)]
    if updown:
        if i > 0:
            n = "%s %s" % (n, _("up"))
        elif i < 0:
            n = "%s %s" % (n, _("down"))
    return n


def filter_intervals_within_range(lowest, highest, irange):
    assert type(lowest) in types.StringTypes
    assert type(highest) in types.StringTypes
    lowest = mpd.notename_to_int(lowest)
    highest = mpd.notename_to_int(highest)
    i = highest - lowest
    return filter(lambda x, i=i: abs(x) <= i, irange)

def random_interval(tonika, lowest, highest, irange):
    """
    Return an int representing the interval.
    Return None if it is not possible to create an interval.
    """
    if isinstance(tonika, mpd.MusicalPitch):
        tonika = tonika.semitone_pitch()
    assert type(tonika) == types.IntType
    assert type(lowest) == type(highest)
    if type(lowest) in types.StringTypes:
        lowest = mpd.notename_to_int(lowest)
    if type(highest) in types.StringTypes:
        highest = mpd.notename_to_int(highest)
    assert type(lowest) == type(highest) == types.IntType
    assert lowest <= highest
    assert type(irange) == types.ListType
    v = []
    for i in irange:
        if lowest <= tonika + i <= highest:
            v.append(i)
    if not v:
        return None
    return random.choice(v)


def random_tonika_and_interval(lowest, highest, irange):
    """ Return a tuple (tonika, interval) of types (MusicalPitch, int).
    Return (None, None) if it is not possible to make an interval
    because of a conflict between the lowest/highest and the available
    intervals

    lowest
    highest  an integer or a string representing a notename
    irange   list of integers representing intervals. 1 is minor
             second up, -2 is major second down
    """
    assert type(lowest) == type(highest)
    if type(lowest) in types.StringTypes:
        lowest = mpd.notename_to_int(lowest)
    if type(highest) == types.StringTypes:
        highest = mpd.notename_to_int(highest)
    assert type(lowest) == type(highest) == types.IntType
    assert lowest <= highest
    assert type(irange) == types.ListType
    # first we find out what is the largest interval we can have
    i = highest - lowest
    # then filter irange to only use intervals <= i
    v = filter(lambda x, i=i: abs(x) <= i, irange)
    if not v:
        return None, None
    interval = random.choice(v)
    # then, using that interval, make a list of possible tonikas
    tl = []
    for t in range(lowest, highest + 1):
       if lowest <= t + interval <= highest:
           tl.append(t)
    tonika = mpd.MusicalPitch.new_from_int(random.choice(tl))
    return tonika, interval


def adjust_low_high_to_irange(lowest, highest, irange):
    """
    lowest
    highest  string representing a notename
    irange   list of integers representing intervals. 1 is minor
             second up, -2 is major second down
    return tuple with two integers
    """
    assert type(lowest) in types.StringTypes
    assert type(highest) in types.StringTypes
    L = mpd.notename_to_int(lowest)
    H = mpd.notename_to_int(highest)
    # find the largest interval that can be asked for
    r = max(abs(max(irange)), abs(min(irange)))
    # adjust the lowest and highest note, if the intervals are larger than
    # the range we should try to stay within
    if H - L < r:
        H = H + (r - (H - L)) / 2
        L = L - (r - (H - L))
    return L, H


def parse_url(s):
    # protocol, action, fn, collection, lessonfile, config
    r = re.compile("""((?P<protocol>\w+):)? # protocol
           ((?P<action>\w[\w-]*))? #practise|config|statistics))? # 
           (/(?P<fn>[\w\-\.]+))? # exercise or html file name
           ((?P<anchor>\#\w+))? # anchor
           (/(?P<collection>\w+))? # collection
           (/(?P<lessonfile>[\.\w-]+))? # lesson file
           (\?(?P<config>.*))? """, re.VERBOSE|re.DOTALL)
    m = r.match(s)
    if not m.group('protocol'):
        if '#' in s:
            return [None, None] + s.split('#', 1) + [None, None, {}]
        return None, None, s, None, None, None, {}
    D = {}
    if m.group('config'):
        V = string.split(m.group('config'), ";")
        for e in V:
            n, v = string.split(e, "=")
            n = string.strip(n)
            v = string.strip(v)
            #if v[0] in string.letters:
            D[string.strip(n)] = v#eval("\""+v+"\"")
            #else:
            #    D[string.strip(n)] = eval(v)
    return m.group('protocol'), m.group('action'), m.group('fn'), \
           m.group('anchor'), m.group('collection'), m.group('lessonfile'), D

def get_modifier(s):
    m = (('<ctrl>', gtk.gdk.CONTROL_MASK),
         ('<shift>', gtk.gdk.SHIFT_MASK),
         ('<alt>', 8))#gtk.gdk.MOD1_MASK))
    for mod, mask in m:
        if s.startswith(mod):
            return mask, s[len(mod):]
    return None, s

def parse_key_string(string):
    if not string:
        return None, None
    mod = 0
    m, s = get_modifier(string)
    while m:
        mod = mod + m
        m, s = get_modifier(s)
    if len(s) == 1:
        return mod, ord(s)
    else:
        return mod, gtk.keysyms.__dict__[s]

def freq_to_notename_cent(freq):
    e = 440.0
    if e > freq:
        while e > freq:
            e = e / 2
    else:
        while e < freq/2:
            e = e * 2
    d = freq / e
    v = 12 * math.log(d) / math.log(2)
    i = int(v)
    cent = (v-i) * 100
    n = ('a', 'ais', 'b', 'c', 'cis', 'd', 'dis', 'e', 'f', 'fis', 'g', 'gis')
    if cent > 50:
        return n[(i+1) % 12], cent-100
    return n[int(v)], (v-int(v)) * 100

def exercise_name_to_module_name(name):
    # it is just plain luck that this works...
    return string.replace(name, '-', '')

def compare_version_strings(A, B):
    """
    Works with version strings like 1, 1.0, 1.1.3, 1.4.3.2
    Returns:
        -1 if A < B
         0 if A == B
         1 if A > B
    """
    if A == B == "":
        return 0
    elif A == "":
        return -1
    elif B == "":
        return 1
    av = map(lambda s: int(s), string.split(A, "."))
    bv = map(lambda s: int(s), string.split(B, "."))
    x = 0
    while len(av) > x < len(bv):
        if av[x] > bv[x]:
            return 1
        elif av[x] < bv[x]:
            return -1
        x = x + 1
    if len(av) > len(bv):
        return 1
    elif len(av) < len(bv):
        return -1
    return 0

if __name__ == '__main__':
        print parse_url("main.html#123")
        print parse_url("main.html")
        print parse_url("exercise:/main.html#23")
        print parse_url("solfege:practise/id-by-name/solfege/fifth-small-pure-0.99")
