;; Standard library provided to SpcialK applications

;; Copyright (C) 2004  Sylvain Beucler
;; Copyright (C) 2004  Julien Charles
;; Copyright (C) 2004  Pierre Chtel
;; Copyright (C) 2004  Cyril Rodas

;; This file is part of SpcialK.

;; SpcialK is free software; you can redistribute it and/or modify it
;; under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either version 2 of the License, or
;; (at your option) any later version.

;; SpcialK is distributed in the hope that it will be useful, but
;; WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
;; General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with SpcialK; if not, write to the Free Software Foundation,
;; Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

(module lib
  mzscheme
  (require "private/display-tree.ss")
  (require "private/display-vector.ss")
  (require "private/fast-k-translator.ss")
  (require "private/l10n.ss")
  (require (lib "pretty.ss")
	   (lib "string.ss")
	   (lib "trace.ss")
	   (lib "match.ss")
	   (lib "etc.ss")
	   (lib "process.ss")
	   (lib "file.ss")
	   (lib "class.ss")
	   (lib "mred.ss" "mred"))

  ;; SpcialK standard lib - check the user documentation for details
  (provide new-tree-viewer new-tab-viewer refresh-viewer
	   display-viewer load-k nasty-print trace untrace match
	   localized-message tree2vcg tree2dot vcg xvcg springgraph
	   make-tab tab-len list-len)

  ;; Arrays wrapper
  (define tab-len vector-length)
  (define list-len length)
  (define (make-tab . L)
    (if (null? L) 0
	(build-vector (car L) (lambda (x) (apply make-tab (cdr L))))))


  ;; MrEd data visualization
  (define (refresh-viewer v arg)
    (v 'refresh-with arg))
  (define (display-viewer v)
    (v 'display))
  
  (define readproc (lambda (file) 
                     (let ((line (read-line file))) 
                       (if (eof-object? line ) 
                           '()
                           (cons line (readproc file))))))
  
  ; Translate a SpcialK file and evaluate it so its definitions can
  ; be used in the current source file - that can be Scheme or
  ; SpcialK code
  (define (load-k file)
    (let ((src (open-input-file file)))
      (for-each (lambda (line)
                  (eval line)) (fast-k->scheme src src #t))))
  
  ; Display values using SpcialK's data representation
  ; This is not a pretty print...
  (define (nasty-print value port)
    (let ([value (if (is-a? value image-snip%)
		     value
		     (nasty-string value))])
      (fprintf port "~a~n" value)))

  (define (nasty-string value)
    (define (print-list value)
      (cond ((null? value) "nil")
	    ((pair? value) (if (pair? (car value))
			       (string-append "(" (print-list (car value)) "):" (print-list (cdr value)))
			       (string-append (print-list (car value)) ":" (print-list (cdr value)))))
	    (else (nasty-string value))))
    (define (print-vect i value)
      (cond ((= i 0)
	     (begin 
	       (string-append "{" 
			      (nasty-string (vector-ref value i))
			      (print-vect (add1 i) value ))))
	    ((= i (vector-length value)) "}")
	    (else
	     (begin (string-append ", " (nasty-string (vector-ref value i))
				   (print-vect (add1 i) value ))))))
    (cond 
     ((vector? value) (print-vect 0 value ))
     ((pair? value) (print-list value))
     ((not value) "false")
     ((boolean? value) "true")
     (else (expr->string value))))


  ;;-------------------------------------
  ;; Bindings with external data viewers
  ;;-------------------------------------

  ;; Generic function to parse an n-ary tree an output node and edge descriptions
  (define (tree->anything out T header footer get-node get-edge)
    (define index 1)
    (define (print-node-and-increment label horder)
      (display (get-node index label horder) out)
      (set! index (+ index 1)))
    (define (internal T horder)
      (let ([cur-index index]
	    [convert (lambda (L) ; support both r:g:d:nil and r:g:d tree layout
		       (let ((cur-cdr (cdr L)))
			 (if (and (not
				   (null? cur-cdr)) (not (pair? cur-cdr)))
			     (list cur-cdr)
			     cur-cdr)))])
	(cond ((pair? T) (print-node-and-increment (car T) horder)
	       (do ((order 1 (+ order 1))
		    (childs (convert T) (convert childs)))
		   ((null? childs)) ; end-of-loop test
		 (display (get-edge cur-index index) out)
		 (internal (car childs) order)))
	      (else (print-node-and-increment T horder)))))
    (display header out)
    (internal T 1)
    (display footer out))
  
  ;; Converts an n-ary tree to a .vcg description, used by [x]vcg
  (define (tree2vcg T . L)
    (let ([out (if (null? L) (current-output-port) (car L))]
	  [header "graph: {\nlayoutalgorithm: tree\n"]
	  [footer "}\n"]
	  [get-node (lambda (index label horder)
		      (format "node: { title:\"~a\" label:\"~a\" horizontal_order:~a}~n"
			      index (regexp-replace* "\"" (nasty-string label) "\\\\\"") horder))]
	  [get-edge (lambda (from to)
		      (format "edge: { sourcename:\"~a\" targetname:\"~a\" }~n" from to))])
      (tree->anything out T header footer get-node get-edge)))

  ;; Converts an n-ary tree to a .dot description, used by springgraph
  (define (tree2dot T . L)
    (let ([out (if (null? L) (current-output-port) (car L))]
	  [header "digraph \"\" {\n"]
	  [footer "}\n"]
	  [get-node (lambda (index label horder)
		      (format "~a [label=\"~a\"]~n" index (nasty-string label)))]
	  [get-edge (lambda (from to)
		      (format "~a -> ~a~n" from to))])
      (tree->anything out T header footer get-node get-edge)))

  ;; Launches 'name' with arg-list as arguments, feed it using f-stdin
  ;; and filter the result using f-stdout. Plus error handing.
  (define (process-apply f-stdin f-stdout name arg-list)
    (define (eat-string in)
      (let ([out (open-output-string)])
	(do ((char (read-char in) (read-char in)))
	    ((eof-object? char) (get-output-string out))
	  (display char out))))
    (let ([path (find-executable-path name #f)])
      (if (not path)
	  (begin
	    (display (localized-message 'process-cannot-find name))
	    #f)
	  (let* ([proc-info (apply process* path `(,@arg-list))]
		 [stdout (car proc-info)]
		 [stdin (cadr proc-info)]
		 [stderr (cadddr proc-info)]
		 [actor (car (cddddr proc-info))])
	    (f-stdin stdin)
	    (close-output-port stdin)
	    (do ()
		((not (eq? (actor 'status) 'running))
		 (if (eq? (actor 'status) 'done-error)
		     (begin
		       (display (localized-message 'process-exit-error name (eat-string stderr)))
		       #f)
		     (begin0
		      (f-stdout stdout)
		      ;(display (eat-string stdout))
		      ;(display (eat-string stderr))
		      (close-input-port stdout)
		      (close-input-port stderr))))
	      (sleep .2))))))
  

  ;; x -> void
  (define (nada x) (void))

  ;; Converts T to .vcg and launches xvcg (Visualization of Compiler
  ;; Graphs for X) on that data
  (define (xvcg T)
    (process-apply (lambda (out) (tree2vcg T out)) nada "xvcg" '("-silent" "-")))

  ;; Deletes a files with exception handling
  (define (delete-file-safe file)
    (with-handlers ([exn:i/o:filesystem? nada])
		   (delete-file file)))

  ;; Creates a displayable image from data input-port 'in'
  (define (get-image in)
    (let ([file (make-temporary-file)])
      (with-output-to-file file
	(lambda ()
	  (do ((char (read-char in) (read-char in))) ((eof-object? char)) (display char))) 'truncate)
      (begin0
       (make-object image-snip% file)
       (delete-file-safe file))))

  ;; Converts T to .dot and then to PNG using springgraph
  (define (vcg T)
    (let ([file (make-temporary-file)])
      ; delete the temporary file, since xvcg does not overwrite files
      (with-handlers ([exn:i/o:filesystem? (lambda (e) (void))])
					(delete-file file))
      (process-apply (lambda (out) (tree2vcg T out)) nada "xvcg"
		     `("-silent" "-ppmoutput" ,file "-"))
      (begin0
       (process-apply nada get-image "pnmtopng" `(,file))
       (delete-file-safe file))))

  ;; Converts T to .dot and then to PNG using springgraph
  (define (springgraph T . L)
    (process-apply (lambda (out) (tree2dot T out)) get-image "springgraph" L))

  ; Exemples:
  ;(let ([tree '(1 2 (3 5) 4)])
  ;  (tree2dot tree)
  ;  (tree2vcg tree)
  ;  (thread (lambda () (xvcg tree)))
  ;  (springgraph tree "-s" ".33"))
)
  
