;; Tree displaying library

;; Copyright (C) 2004  Sylvain Beucler
;; Copyright (C) 2004  Julien Charles
;; Copyright (C) 2004  Pierre Chtel
;; Copyright (C) 2004  Cyril Rodas

;; This file is part of SpcialK.

;; SpcialK is free software; you can redistribute it and/or modify it
;; under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either version 2 of the License, or
;; (at your option) any later version.

;; SpcialK is distributed in the hope that it will be useful, but
;; WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
;; General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with SpcialK; if not, write to the Free Software Foundation,
;; Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

(module display-tree
  mzscheme
  (provide new-tree-viewer)
  (require (lib "mred.ss" "mred")(lib "class.ss" "mzlib")
	   (lib "string.ss" "mzlib") "l10n.ss")
  
(define (new-tree-viewer tree . options)
  (let* ((required-width 400)
         (required-height 400)
         ($frame (instantiate frame% () (label (localized-message 'display-vector-title))))
         (face-bitmap (instantiate bitmap% ((* required-width 2) (* required-height 2))))
         (bm-dc (instantiate bitmap-dc% (face-bitmap)))
         (brushes `(,(instantiate brush% ("BLUE" 'solid))
                     ,(instantiate brush% ("YELLOW" 'solid))
                     ,(instantiate brush% ("GREEN" 'solid))
                     ,(instantiate brush% ("RED" 'solid))
                     ,(instantiate brush% ("ORANGE" 'solid))))
         (associations '())
         (canvas (instantiate canvas% ()
                   (parent $frame)
                   (min-width required-width)
                   (min-height required-height)
                   (stretchable-width #t)
                   (stretchable-height #t)
                   (paint-callback
                    (lambda (c dc)
                      (send bm-dc clear) 
                      (if (null? associations)
                          (let ((tmp (if (null? options)
                                         (draw-tree tree brushes '() (/ required-width 2) 20 bm-dc)
                                         (draw-tree tree brushes '() (/ required-width 2) 20 bm-dc options))))
                            (set! brushes (car tmp))
                            (set! associations (cadr tmp)))
                          (if (null? options)
                              (draw-tree tree brushes associations (/ required-width 2) 20 bm-dc)
                              (draw-tree tree brushes associations (/ required-width 2) 20 bm-dc options)))
                      
                      (send dc draw-bitmap face-bitmap 0 0)
                      )))))
    
    (define (tree-viewer action . args)
      (case action
        ;change the tree that will be displayed (but no refresh)
        ((set!) (if (not (null? args))
                    (set! tree (car args))))
        ;show viewer's frame
        ((display) (send $frame show #t))
        ;refresh the viewer
        ((refresh) (send canvas on-paint))
        ;refresh the viewer with a new tree
        ((refresh-with) (if (not (null? args))
                           (begin (set! tree (car args))
                                  (send canvas on-paint))))
        ;associate new brushes to elements and refresh
        ((colors) (if (not (null? args))
                      (begin (set! brushes (car args))
                             (tree-viewer 'reset))))
        ;recalculate colors associations
        ((reset) (set! associations '())
                 (tree-viewer 'refresh))
        (else (error (localized-message 'display-unknown-action) action))))
    tree-viewer))

;  (draw-tree '("+" 1 ("*" ("/" 2 3) "a")) 320 20 dc '(node-spacing-y 40))
;  (draw-tree '("animal" "elephant" ("insect" "bee" "fly")) 320 20 dc))

; Default settings, overrideable through "options"
  (define node-spacing-x 10)
  (define node-spacing-y 20)
  (define node-padding 5)
  (define node-corner-radius 8)
  (define arc-curvature 0.75) ; from 0=concave, 0.5=straight, 1=convex
  
; Calculates the width and height of given (sub)tree
  (define (tree-size tree)
    (if (list? tree)
        (let ((w (- node-spacing-x)) (h 0))
          (for-each 
           (lambda (x)
             (let ((cs  (tree-size x)))
               (set! w (+ w (car cs) node-spacing-x))
               (set! h (max h (cadr cs)))
               ))
           (cdr tree))
          (list (max w (car (tree-size (car tree))))
                (+ h node-spacing-y (cadr (tree-size (car tree))))))
        (list (caddr (tree 'bounds))
              (cadddr (tree 'bounds)))))
  
(define (draw-tree tree brushes associations x y dc . options )  
  ; Creates a drawable boxed string object
  (define (make-text-box str)
    (define x 0)
    (define y 0)
    (define text str)
    (let-values (((width height bot-dist top-dist)
                  (send dc get-text-extent text)))
     (lambda (m)
                   (let ((txt-top (- y (/ height 2)))
                         (txt-left (- x (/ width 2))))
                     (define (bounds)
                       (list (- txt-left node-padding) (- txt-top node-padding)
                             (+ width (* node-padding 2)) (+ height (* node-padding 2))))
                     (cond ((eq? m 'draw)
                            (lambda (dc brushes associations)
                              (let ((used? (assoc text associations)))
                                (if used?
                                    (send dc set-brush (cadr used?))
                                    (if (null? brushes)
                                        (send dc set-brush (instantiate brush% ("WHITE" 'solid)))
                                        (begin (send dc set-brush (car brushes))
                                               (set! associations (cons (list text (car brushes)) associations))
                                               (set! brushes (cdr brushes)))))
                                (send dc draw-rounded-rectangle
                                      (car (bounds)) (cadr (bounds))
                                      (caddr (bounds)) (cadddr (bounds))
                                      node-corner-radius)
                                (send dc draw-text text txt-left txt-top))
                              (list brushes associations)
                              ))
                           
                           ((eq? m 'move!)
                            (lambda (new-x new-y)
                              (set! x new-x)
                              (set! y new-y)))
                           ((eq? m 'bounds) (bounds))
                           (else error (localized-message 'display-unknown-message) m))))))
        
  ; Draws a tree of boxed string objects
  (define (draw-text-box-tree tree x y dc brushes associations)
    (if (list? tree)
        (begin
          (if (list? (car tree))
              (error (localized-message 'display-tree-error-root)))
          (let ((size (tree-size tree))
                (root (car tree))
                (rx x) (ry y)
                (root-size (cddr ((car tree) 'bounds))))
            
            (let ((tmp (draw-text-box-tree root x y dc brushes associations)))
              (set! brushes (car tmp))
              (set! associations (cadr tmp)))
            
            (set! x (- x (/ (car size) 2)))
            (for-each
             (lambda (child)
               (let ((cs (tree-size child))
                     (cx (+ x (/ (car(tree-size child)) 2)))
                     (cy (+ y (cadr root-size) node-spacing-y))
                     (child-root-size
                      ((if (list? child) (car child) child) 'bounds)))
                 
                 (let ((tmp (draw-text-box-tree child cx cy dc brushes associations)))
                   (set! brushes (car tmp))
                   (set! associations (cadr tmp)))
                 
                 (let ((root-bott (+ ry (/ (cadr root-size) 2)))
                       (child-top (- cy (/ (cadddr child-root-size) 2))))
                   (send dc draw-spline
                         rx
                         root-bott
                         (+ rx (* (- cx rx) arc-curvature))
                         (+ root-bott (* (- child-top root-bott) (- 1 arc-curvature)))
                         cx
                         child-top))
                 (set! x (+ x (car (tree-size child)) node-spacing-x))))
             (cdr tree))
            (list brushes associations)))
        (begin 
          ((tree 'move!) x y)
          ((tree 'draw) dc brushes associations))))
  
  ; Convert given object to string - extend this!
  (define (to-string x)
    (cond ((string? x) x)
          ((char? x) (list->string (list x)))
          ((number? x) (number->string x))
          (else (expr->string x))))
  
  (define (make-text-box-tree tree)
    (map (lambda (x)
           (if (list? x)
               (make-text-box-tree x)
               (make-text-box (to-string x))))
         tree))
  
  ; Walk through the (possible) options
  (for-each
   (lambda (x)
     (if (and (list? x) (not (null? (cdr x))) (number? (cadr x)))
         (let ((name (car x)) (val (cadr x)))
           (cond ((eq? name 'node-spacing-x)
                  (set! node-spacing-x val))
                 ((eq? name 'node-spacing-y)
                  (set! node-spacing-y val))
                 ((eq? name 'node-padding)
                  (set! node-padding val))
                 ((eq? name 'node-corner-radius)
                  (set! node-corner-radius val))
                 ((eq? name 'arc-curvature)
                  (set! arc-curvature val))
                 (else (error (localized-message 'display-unknown-option) name))))
         (error (localized-message 'display-bad-option x))))
   options)
  
  ; Do the drawing
  (draw-text-box-tree (make-text-box-tree tree) x y dc brushes associations)))

;(define test (new-tree-viewer '("+" 1 ("*" ("/" "SpcialK is fun!" +) "SpcialK is fun!")) ;'arc-curvature 0.5))
;(test 'display)
