;; DrScheme Gui hackings

;; Copyright (C) 2004  Sylvain Beucler
;; Copyright (C) 2004  Julien Charles
;; Copyright (C) 2004  Pierre Chtel
;; Copyright (C) 2004  Cyril Rodas

;; This file is part of SpcialK.

;; SpcialK is free software; you can redistribute it and/or modify it
;; under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either version 2 of the License, or
;; (at your option) any later version.

;; SpcialK is distributed in the hope that it will be useful, but
;; WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
;; General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with SpcialK; if not, write to the Free Software Foundation,
;; Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

(module special-k-gui
  mzscheme
  (require (lib "lex.ss" "parser-tools")
           (lib "framework.ss" "framework")
           (lib "class.ss" "mzlib")
           (lib "pregexp.ss" "mzlib")
           (lib "mred.ss" "mred")
           "l10n.ss")
  (provide  mode-surrogate
            repl-submit 
            matches-language
            register-colors)
  
  
  
  
  ;; lexer for syntax coloring
  
  (define-empty-tokens op-tokens (SYMS COMMENT ERR white-space ID NUMBER BOOL SCHEMECODE STRING))
  
  (define-lex-abbrevs (letter (: (- #\a #\z) (- #\A #\Z)))
                      (alphanum (:(- #\a #\z) (- #\A #\Z) (- #\0 #\9) #\_ #\- #\?))
                      (ascii (- #\000 #\177))
                      (digit (- #\0 #\9)))
  (define (syn-val lex a b c d)
    (values lex a b (position-offset c) (position-offset d)))
  
  ;; the lexer used for syntax coloring
  (define get-special-k-tokens
    (lexer
     ((eof) (syn-val 'eof 'eof #f start-pos end-pos))
     ((@ #\" (* (^ #\")))
      (syn-val lexeme 'err #f start-pos end-pos))
     ((@ #\" (* (^ #\")) #\")
      (syn-val lexeme 'string #f start-pos end-pos))
     ((@ "/*" (~ (@ (&) "*/" (&))))
      (syn-val lexeme 'err #f start-pos end-pos))
     ((@ "/*" (~ (@ (&) "*/" (&))) "*/")
      (syn-val lexeme 'comment #f start-pos end-pos))
     ((: #\tab #\space #\newline )
      (syn-val lexeme 'white-space #f start-pos end-pos))
     ((: "(" ")" "{" "}" "[" "]")
      (syn-val lexeme 'syms #t start-pos end-pos))
     ((: "->" "=" ";" "." "," ":" "<" ">"  "!" "==" "<>" 
         "<=" ">=" "<-" "<->" "+" "-" "*" "/" "|"  "&")
      (syn-val lexeme 'syms #f start-pos end-pos))
     ((: "true" "false") (syn-val lexeme 'bool #f start-pos end-pos))
     ((: "nil") (syn-val lexeme 'nil #f start-pos end-pos))
     ((@ (+ letter) (* alphanum)) (syn-val lexeme 'id #f start-pos end-pos))
     ((: (+ digit) (@ (* digit) #\. (+ digit))) (syn-val lexeme 'number #f start-pos end-pos))
     ((@ "<%%" (* (: (@ #\% #\% (^ #\>))
                     (@ #\% (^ #\%))
                     (@ (^ #\%)))) "%%>") (syn-val lexeme 'schemecode #f start-pos end-pos))
     
     ((- #\000 #\377) (syn-val lexeme 'err #f start-pos end-pos))))
  
  
  ;; the tabbing utilities
  (define paren-pos 10)
  (define prompt-pos 0)
  (define comment-start 2)
  
  ;; if we're in a clause or statement at the specified pos.
  ;; text-canvas the canvas
  ;; start the specified pos
  (define (is-statement? text-canvas start)
    (define (regexp-state pos)
      (let* ((s1 (pregexp-replace* "[ \t\n]+"
                                   (send text-canvas get-text pos start) ""))
             (s2 (pregexp-replace* ;(: (+ (: " " "\t" "\n"))
                  ;   (@ "/*" (~ (@ (* (^)) "*/" (* (^)))) "*/"))
                  "/\\*((([^*][^/])*.\\*/)|(([^*][^/])*\\*/)|(.\\*/)|(\\*/))"
                  ;  "([ \t\n]+)|/\\*(([^*][^/])*.?)*\\*/"
                  s1 "")))
        ;(display s2) (newline)
        (zero? (string-length s2))))
    (let* ((s (find-not-in-comment "." text-canvas start))
           (e (find-not-in-comment ";" text-canvas start)))
      (cond ((and s e (> s e)) (if (regexp-state (add1 s))
                                   'dot #f))
            ((and s e (< s e)) (regexp-state (add1 e)))
            (s (if (regexp-state s) 'dot #f))
            (e (regexp-state e))
            (else (if (regexp-state 0)
                      'dot #f)))))
  
  ;; find in a canvas something ignoring the comments or the strings
  ;; str the something to search for
  ;; text-canvas the canvas where to search
  ;; the starting point
  ;; n.b.: it searches backwards, to undo the magick.
  (define (find-not-in-comment str text-canvas start)
    (let ((s (send text-canvas find-string str 'backward start 0 #t #t)))
      (if s
          (if (is-comment? text-canvas s)
              (find-not-in-comment str text-canvas (sub1 s))
              s)
          #f)))
  
  ;; tell whether or not 'end' is a position inside a comment
  ;; text-canvas the context canvas
  ;; end the position
  (define (is-comment? text-canvas end)
    (define (comment-end start end)
      (let ((c (send text-canvas find-string "*/" 'forward start end #t #t)))
        (if c (comment-begin (add1 c) end) #t)))
    (define (string-end start end)
      (let ((c1 (send text-canvas find-string "\"" 'forward start end #t #t))
            (c2 (send text-canvas find-string "\\\"" 'forward start end #t #t)))
        (cond 
          ((and c2 (= c1 (add1 c2))) (string-end (add1 c1) end))
          (c1 (comment-begin (add1 c1) end))
          (else #t))))
    (define (comment-begin start end)
      (let* ((com (send text-canvas find-string "/*" 'forward start end #t #t))
             (str (send text-canvas find-string "\"" 'forward start end #t #t)))
        (cond 
          ((or (and com str (< com str))
               (and com (not str)))
           ;; commentaire en premier: on en cherche la fin
           (comment-end (add1 com) end))
          ((or (and com str (> com str))
               (and str (not com)))
           ;; string en premier: on en cherche la fin
           (string-end (add1 str) end))
          (else #f))))
    (comment-begin prompt-pos end))
  
  ;; tell whether or not 'start' is a position inside some parenthesis form
  ;; text-canvas the context canvas
  ;; start the position
  (define (in-paren? text-canvas start)
    (let* ((s (find-not-in-comment "(" text-canvas start))
           (e (find-not-in-comment ")" text-canvas start)))
      (if s
          (set! paren-pos (let* ((line (send text-canvas position-line s #f))
                                 (line-start (send text-canvas line-start-position line #f)))
                            (- s line-start))))
      (or (and s e (> s e))
          (and s (not e)))))
  
  ;; get the blanks of the preceding line; if there aren't any  it returns ""
  ;; canvas the texte-canvas where to search
  ;; line the current line
  (define (get-blanks canvas line)
    (let* ((line-start (send canvas line-start-position line #f))
           (line-end (send canvas line-end-position line #f))
           (text (send canvas get-text line-start line-end)))
      (car (pregexp-match "^ *\t* *" text))))
  
  ;; tabify a line
  ;; text-canvas the canvas
  ;; line the line to tab
  ;; len the len to tab
  (define (tabify-line text-canvas line len)
    (let ((line-start (send text-canvas line-start-position line #f))
          (current-blanks (string-length (get-blanks text-canvas line))))
      (send text-canvas insert (make-string len #\ ) line-start (+ line-start current-blanks))))
  
  ;; do a simple tabbing
  ;; text-canvas the canvas
  ;; line the line to tab
  ;; diff the len to tab
  (define (tabify-simple text-canvas line diff)
    (let* ((old-blanks (if (not (zero? line)) (get-blanks text-canvas (- line 1)) "")))
      (tabify-line text-canvas line (+ (string-length old-blanks) diff))))
  
  ;; tabify a line in a canvas
  ;; orig the canvas
  ;; line the line
  (define (tabify orig line)
    (let* ((line-start (send orig line-start-position line #f)))
      (cond ((is-comment? orig line-start)
             (tabify-simple orig line 0))
            ((in-paren? orig line-start)
             (tabify-line orig line paren-pos))
            ((not (is-statement? orig line-start))
             (tabify-line orig line 3))
            (else (tabify-line orig line 0)))))
  
  ;; a fun to tabify a region in a canva
  ;; canvas being the canvas where the region is
  ;;  start the offset where to begin
  ;; end the offset where to end the tabbing
  (define (tabify-region canvas start end)
    (if (<= start end)
        (begin
          (tabify canvas start)
          (tabify-region canvas (add1 start) end))))
  
  ;; the specialk color mode text... useful for custom tabbing...
  (define special-k:text-mode% 
    (class color:text-mode% 
      (define/override (on-char orig s event)
        (case (send event get-key-code) 
          ((#\tab)(let* ((start-box (box 500))
                         (end-box (box 500))
                         (start (begin (send orig get-position start-box end-box)
                                       (unbox start-box)))
                         (end (unbox end-box)))
                    (tabify-region orig (send orig position-line start #f)
                                   (send orig position-line end #f))))
          ((#\return)
           (let* ((pos-box (box 500))
                  (pos (begin (send orig get-position pos-box #f)
                              (unbox pos-box))))
             (s event)
             (tabify orig (add1 (send orig position-line pos #f))))) 
          (else (s event))))
      (super-instantiate ())))
  
  
  ;; Les couleurs par dfaut
  (define color-prefs-table
    `((id ,(send the-color-database find-color "Blue") "id")
      (bool ,(send the-color-database find-color "Purple") "bool")
      (nil ,(send the-color-database find-color "LightSlateGray") "nil")
      (schemecode ,(send the-color-database find-color "Magenta") "schemecode")
      (string ,(send the-color-database find-color "forestgreen") "string")
      (number ,(send the-color-database find-color "Purple") "number")
      (comment ,(send the-color-database find-color "DarkOrange") "comment")
      (err ,(send the-color-database find-color "Red") "err")          
      (syms ,(make-object color% "black") "syms")))
  
  ;; short-sym->pref-name : symbol -> symbol
  ;; returns the preference name for the color prefs
  (define (short-sym->pref-name sym) (string->symbol (short-sym->style-name sym)))
  
  ;; short-sym->style-name : symbol->string
  ;; converts the short name (from the table above) into a name in the editor list
  ;; (they are added in by `color-prefs:register-color-pref', called below)
  (define (short-sym->style-name sym)
    (format "specialk:syntax-coloring:scheme:~a" sym))
  
  
  
  ;; extend-preferences-panel : vertical-panel -> void
  ;; adds in the configuration for the special-k colors to the prefs panel
  (define (extend-preferences-panel parent)
    (for-each
     (lambda (line)
       (let ([sym (car line)])
         (color-prefs:build-color-selection-panel 
          parent
          (short-sym->pref-name sym)
          (short-sym->style-name sym)
          (format "~a" sym))))
     color-prefs-table))
  
  
  ;; Le mode d'criture 'special-k' (pour le tabbing et le syntax coloring)
  (define mode-surrogate
    (new special-k:text-mode%
         (matches (list (list '|{| '|}|)
                        (list '|/*| '|*/|)
                        (list '|"| '|"|)
                        (list '|(| '|)|)
                        (list '|[| '|]|)))
         (get-token get-special-k-tokens)
         (token-sym->style short-sym->style-name)))
  
  
  ;; matches-language : (union #f (listof string)) -> boolean
  (define (matches-language l)
    (and l (pair? l) (pair? (cdr l)) (string=? (cadr l) (localized-message 'language-name))))
  
  
  
  ;; the fun to tell wether or not the window has to be in specialk-colored
  (define (repl-submit text prompt-position)
    (set! prompt-pos prompt-position)
    (let* ((line-start (send text last-position))
           (res (and (not (is-comment? text line-start))
                     (not (in-paren? text line-start))
                     (eq? 'dot (is-statement? text line-start)))))
      (set! prompt-pos 0)
      res))
  
  
  ;; the fun to register the color in drscheme
  (define (register-colors)
    (color-prefs:add-to-preferences-panel (localized-message 'language-name) extend-preferences-panel)
    (for-each (lambda (line)
                (let ([sym (car line)]
                      [color (cadr line)])
                  (color-prefs:register-color-pref (short-sym->pref-name sym)
                                                   (short-sym->style-name sym)
                                                   color)))
              color-prefs-table)))
