/*
 * $Id: st-cache-save.c,v 1.46 2004/03/27 01:23:05 jylefort Exp $
 *
 * Copyright (c) 2002, 2003, 2004 Jean-Yves Lefort
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of Jean-Yves Lefort nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND
 * CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 * ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include <string.h>
#include <unistd.h>
#include <errno.h>
#include <gtk/gtk.h>
#include "gettext.h"
#include "sg-dumper.h"
#include "st-handler.h"
#include "st-handlers.h"
#include "st-settings.h"
#include "st-stream-api.h"
#include "sg-util.h"
#include "st-category-store.h"
#include "st-stream-store.h"
#include "st-handler-field.h"

/*** type definitions ********************************************************/

typedef struct
{
  SGDumper	*dumper;
  GError	**err;
} STCacheSaveInfo;

/*** function declarations ***************************************************/

static gboolean st_cache_save_cleanup_images	(GError		 **err);
static gboolean st_cache_save_handler		(SGDumper	 *dumper,
						 STHandler	 *handler,
						 GError		 **err);
static gboolean st_cache_save_category		(SGDumper	 *dumper,
						 STCategoryBag	 *category_bag,
						 STCategoryBag	 *parent_bag,
						 GError		 **err);
static gboolean st_cache_save_category_cb	(STCategoryStore *store,
						 STCategoryBag	 *bag,
						 STCategoryBag	 *parent,
						 gpointer	 data);
static gboolean st_cache_save_streams		(SGDumper	 *dumper,
						 STStreamStore	 *streams,
						 GError		 **err);
static gboolean	st_cache_save_stream_cb		(STStreamStore	 *store,
						 STStreamBag	 *bag,
						 gpointer	 data);
static gboolean st_cache_save_stream		(SGDumper	 *dumper,
						 STStreamBag	 *stream_bag,
						 GError		 **err);
static gboolean st_cache_save_value		(SGDumper	 *dumper,
						 const GValue	 *value,
						 GError		 **err);

/*** implementation **********************************************************/

gboolean
st_cache_save (const char *filename, GError **err)
{
  SGDumper *dumper;
  GSList *l;

  g_return_val_if_fail(filename != NULL, FALSE);

  if (! st_cache_save_cleanup_images(err))
    return FALSE;

  dumper = sg_dumper_new(filename, err);
  if (! dumper)
    return FALSE;

  if (! sg_dumper_write(dumper, err,
			"#", _("streamtuner cache file, automatically generated"),
			"",
			NULL))
    goto error;

  SG_LIST_FOREACH(l, st_handlers_list)
    if (! st_cache_save_handler(dumper, l->data, err))
      goto error;

  return sg_dumper_free(dumper, err);

 error:
  sg_dumper_free(dumper, NULL);	/* ignore errors since ERR is already set */
  return FALSE;
}

static gboolean
st_cache_save_cleanup_images (GError **err)
{
  GDir *dir;
  const char *filename;
  gboolean status = TRUE;

  dir = g_dir_open(st_settings.images_dir, 0, err);
  if (! dir)
    return FALSE;

  while ((filename = g_dir_read_name(dir)))
    {
      if (filename[0] != '.')
	{
	  char *pathname;

	  pathname = g_build_filename(st_settings.images_dir, filename, NULL);
	  status = unlink(pathname) >= 0;
	  g_free(pathname);

	  if (! status)
	    {
	      g_set_error(err, 0, 0, g_strerror(errno));
	      break;
	    }
	}
    }
  g_dir_close(dir);

  return status;
}

static gboolean
st_cache_save_handler (SGDumper *dumper, STHandler *handler, GError **err)
{
  STCacheSaveInfo info = { dumper, err };
  STCategoryStore *categories;

  g_return_val_if_fail(dumper != NULL, FALSE);
  g_return_val_if_fail(ST_IS_HANDLER(handler), FALSE);

  if (! sg_dumper_write(dumper, err,
			"ks{", "handler", st_handler_get_name(handler),
			NULL))
    return FALSE;

  categories = st_handler_get_categories(handler);
  st_category_store_foreach(categories, st_cache_save_category_cb, &info);
  g_object_unref(categories);

  if (*info.err)
    return FALSE;
      
  return sg_dumper_write(dumper, err, "}", NULL);
}

static gboolean
st_cache_save_category (SGDumper *dumper,
			STCategoryBag *category_bag,
			STCategoryBag *parent_bag,
			GError **err)
{
  STStreamStore *streams;

  g_return_val_if_fail(dumper != NULL, FALSE);
  g_return_val_if_fail(ST_IS_CATEGORY_BAG(category_bag), FALSE);

  if (! sg_dumper_write(dumper, err,
			"ks{", "category", ST_CATEGORY(category_bag)->name,
			"ku", "flags", st_category_bag_get_flags(category_bag),
			NULL))
    return FALSE;

  if (parent_bag)
    {
      if (! sg_dumper_write(dumper, err,
			    "ks", "parent", ST_CATEGORY(parent_bag)->name,
			    NULL))
	return FALSE;
    }

  if (! ST_CATEGORY_BAG_IS_STOCK(category_bag))
    {
      if (! sg_dumper_write(dumper, err,
			    "ks", "label", ST_CATEGORY(category_bag)->label,
			    "ks", "url_postfix", ST_CATEGORY(category_bag)->url_postfix,
			    NULL))
	return FALSE;
    }

  streams = st_handler_get_streams(category_bag->handler, ST_CATEGORY(category_bag)->name);
  if (streams)
    {
      gboolean status;

      status = st_cache_save_streams(dumper, streams, err);
      g_object_unref(streams);

      if (! status)
	return FALSE;
    }

  return sg_dumper_write(dumper, err, "}", NULL);
}

static gboolean
st_cache_save_category_cb (STCategoryStore *store,
			   STCategoryBag *bag,
			   STCategoryBag *parent,
			   gpointer data)
{
  STCacheSaveInfo *info = data;

  return ! st_cache_save_category(info->dumper, bag, parent, info->err);
}

static gboolean
st_cache_save_streams (SGDumper *dumper,
		       STStreamStore *streams,
		       GError **err)
{
  STCacheSaveInfo info = { dumper, err };

  g_return_val_if_fail(dumper != NULL, FALSE);
  g_return_val_if_fail(ST_IS_STREAM_STORE(streams), FALSE);

  if (! sg_dumper_write(dumper, err, "k{", "streams", NULL))
    return FALSE;

  st_stream_store_foreach(streams, st_cache_save_stream_cb, &info);

  if (*info.err)
    return FALSE;

  return sg_dumper_write(dumper, err, "}", NULL);
}

static gboolean
st_cache_save_stream_cb (STStreamStore *store,
			 STStreamBag *bag,
			 gpointer data)
{
  STCacheSaveInfo *info = data;

  return ! st_cache_save_stream(info->dumper, bag, info->err);
}

static gboolean
st_cache_save_stream (SGDumper *dumper, STStreamBag *stream_bag, GError **err)
{
  GSList *l;
  
  g_return_val_if_fail(dumper != NULL, FALSE);
  g_return_val_if_fail(ST_IS_STREAM_BAG(stream_bag), FALSE);

  if (! sg_dumper_write(dumper, err,
			"ks{", "stream", ST_STREAM(stream_bag)->name,
			"k{", "fields",
			NULL))
    return FALSE;

  SG_LIST_FOREACH(l, st_handler_get_fields(stream_bag->handler))
    {
      STHandlerField *field = l->data;

      if (! ST_HANDLER_FIELD_IS_VOLATILE(field))
	{
	  GValue value = { 0, };
	  gboolean status;
      
	  st_stream_bag_get_field(stream_bag, field, &value);
	  status = st_cache_save_value(dumper, &value, err);
	  g_value_unset(&value);

	  if (! status)
	    return FALSE;
	}
    }
  
  return sg_dumper_write(dumper, err, "}", "}", NULL);
}

static gboolean
st_cache_save_value (SGDumper *dumper, const GValue *value, GError **err)
{
  g_return_val_if_fail(dumper != NULL, FALSE);
  g_return_val_if_fail(G_IS_VALUE(value), FALSE);

  if (G_VALUE_HOLDS(value, G_TYPE_VALUE_ARRAY))
    {
      int i;
      GValueArray *value_array = g_value_get_boxed(value);

      if (! sg_dumper_write(dumper, err, "k{", "array", NULL))
	return FALSE;

      if (value_array)
	for (i = 0; i < value_array->n_values; i++)
	  if (! st_cache_save_value(dumper, g_value_array_get_nth(value_array, i), err))
	    return FALSE;

      return sg_dumper_write(dumper, err, "}", NULL);
    }
  else if (G_VALUE_HOLDS(value, GDK_TYPE_PIXBUF))
    {
      GdkPixbuf *pixbuf;
      char *pathname = NULL;
      gboolean status = TRUE;

      pixbuf = g_value_get_object(value);
      if (pixbuf)
	{
	  static unsigned int image_id = 0;
	  char *filename;

	  filename = g_strdup_printf("%i.png",  image_id++);
	  pathname = g_build_filename(st_settings.images_dir, filename, NULL);
	  g_free(filename);

	  status = gdk_pixbuf_save(pixbuf, pathname, "png", err, NULL);
	}

      if (status)
	status = sg_dumper_write(dumper, err, "s", pathname, NULL);
      g_free(pathname);

      return status;
    }
  else
    {
      switch (G_VALUE_TYPE(value))
	{
	case G_TYPE_BOOLEAN:
	  return sg_dumper_write(dumper, err,
				 "b", g_value_get_boolean(value),
				 NULL);

	case G_TYPE_INT:
	  return sg_dumper_write(dumper, err,
				 "i", g_value_get_int(value),
				 NULL);

	case G_TYPE_STRING:
	  return sg_dumper_write(dumper, err,
				 "s", g_value_get_string(value),
				 NULL);
      
	default:
	  g_assert_not_reached();
	}
    }

  return FALSE;			/* never reached */
}
