/*
 * $Id: st-session.c,v 1.18 2004/03/26 19:02:43 jylefort Exp $
 *
 * Copyright (c) 2002, 2003, 2004 Jean-Yves Lefort
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of Jean-Yves Lefort nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND
 * CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 * ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include <errno.h>
#include <glib.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <gtk/gtk.h>
#include "gettext.h"
#include "sg-util.h"
#include "st-cache-load.h"
#include "st-cache-save.h"
#include "st-config-load.h"
#include "st-config-save.h"
#include "st-dialog-api.h"
#include "st-settings.h"
#include "st-preselections.h"

/*** function declarations ***************************************************/

static gboolean st_session_mkdir (const char *dirname);
static gboolean st_session_save_splash_enabled (const char *filename,
						GError **err);

/*** implementation **********************************************************/

void
st_session_load (void)
{
  GError *err = NULL;
  
  if (g_file_test(st_settings.accels_file, G_FILE_TEST_IS_REGULAR))
    gtk_accel_map_load(st_settings.accels_file);

  if (g_file_test(st_settings.config_file, G_FILE_TEST_IS_REGULAR))
    {
      if (! st_config_load(st_settings.config_file, &err))
	{
	  char *normalized;

	  normalized = st_dialog_normalize(err->message);

	  st_error_dialog(_("Unable to load configuration."), "%s", normalized);

	  g_free(normalized);
	  g_clear_error(&err);
	}
    }
  
  if (g_file_test(st_settings.cache_file, G_FILE_TEST_IS_REGULAR))
    {
      if (! st_cache_load(st_settings.cache_file, &err))
	{
	  char *normalized;

	  normalized = st_dialog_normalize(err->message);
	  
	  st_error_dialog(_("Unable to load cache."), "%s", normalized);

	  g_free(normalized);
	  g_clear_error(&err);
	}
    }

  st_preselections_add_stock();
}

void
st_session_save (void)
{
  GError *err = NULL;

  /* create directories */

  if (! st_session_mkdir(st_settings.private_dir))
    return;
  if (! st_session_mkdir(st_settings.images_dir))
    return;

  /* save accel map */

  gtk_accel_map_save(st_settings.accels_file);

  /* save config */

  if (! st_config_save(st_settings.config_file, &err))
    {
      char *normalized;
      
      normalized = st_dialog_normalize(err->message);
      
      st_error_dialog(_("Unable to save configuration."), "%s", normalized);
      
      g_free(normalized);
      g_clear_error(&err);
    }
  
  /* save cache */

  if (! st_cache_save(st_settings.cache_file, &err))
    {
      char *normalized;
      
      normalized = st_dialog_normalize(err->message);
      
      st_error_dialog(_("Unable to save cache."), "%s", normalized);
      
      g_free(normalized);
      g_clear_error(&err);
    }

  /* save status of splash */

  if (! st_session_save_splash_enabled(st_settings.splash_disabled_file, &err))
    {
      char *normalized;
      
      normalized = st_dialog_normalize(err->message);
      
      st_error_dialog(_("Unable to save splash screen setting."), "%s", normalized);
      
      g_free(normalized);
      g_clear_error(&err);
    }
}

static gboolean
st_session_mkdir (const char *dirname)
{
  gboolean status = TRUE;

  if (! g_file_test(dirname, G_FILE_TEST_IS_DIR))
    {
      if (mkdir(dirname, 0755) < 0)
	{
	  char *secondary;
	  char *normalized;

	  secondary = g_strdup_printf("Unable to create directory <i>%s</i>: %s.",
				      dirname, g_strerror(errno));
	  normalized = st_dialog_normalize(secondary);
	  g_free(secondary);

	  st_error_dialog(_("Unable to save session."), "%s", normalized);
	  g_free(normalized);

	  status = FALSE;
	}
    }

  return status;
}

static gboolean
st_session_save_splash_enabled (const char *filename, GError **err)
{
  g_return_val_if_fail(filename != NULL, FALSE);

  if (g_file_test(filename, G_FILE_TEST_EXISTS))
    {
      if (st_settings.splash_enabled)
	{
	  if (unlink(filename) < 0)
	    {
	      g_set_error(err, 0, 0, _("unable to unlink %s: %s"), filename, g_strerror(errno));
	      return FALSE;
	    }
	}
    }
  else
    {
      if (! st_settings.splash_enabled)
	{
	  GError *tmp_err = NULL;
	  GIOChannel *channel;
	  gboolean status;

	  channel = g_io_channel_new_file(filename, "w", &tmp_err);
	  if (! channel)
	    {
	      g_set_error(err, 0, 0, _("unable to create %s: %s"), filename, tmp_err->message);
	      g_error_free(tmp_err);
	      
	      return FALSE;
	    }

	  status = g_io_channel_shutdown(channel, TRUE, &tmp_err) == G_IO_STATUS_NORMAL;
	  g_io_channel_unref(channel);

	  if (! status)
	    {
	      g_set_error(err, 0, 0, _("unable to close %s: %s"), filename, tmp_err->message);
	      g_error_free(tmp_err);

	      return FALSE;
	    }
	}
    }

  return TRUE;
}
