/*
 * $Id: st-shell.c,v 1.170 2004/03/30 12:15:09 jylefort Exp $
 *
 * Copyright (c) 2004 Jean-Yves Lefort
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of Jean-Yves Lefort nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND
 * CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 * ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include "config.h"
#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>
#include "art/streamtuner.h"
#include "gettext.h"
#include "sg-util.h"
#include "sgtk-hig.h"
#include "sgtk-util.h"
#include "st-shell.h"
#include "st-browser-tab.h"
#include "st-settings.h"
#include "sgtk-message-dialog.h"
#include "st-dialog-api.h"
#include "st-action.h"
#include "st-about-dialog.h"
#include "st-stream-properties-dialog.h"
#include "st-preferences-dialog.h"
#include "st-stock.h"
#include "st-stream-menu-items.h"
#include "st-handlers.h"
#include "st-thread.h"
#include "st-link.h"
#include "st-statusbar.h"
#include "st-preselections.h"
#include "st-main.h"
#include "st-browser-tab-label.h"
#include "st-stream-store.h"
#include "st-stream-task.h"
#include "st-bookmarks.h"
#include "st-util.h"

/*** type definitions ********************************************************/

struct _STShellPrivate
{
  GtkWidget		*window;
  GtkAccelGroup		*accel_group;
  GtkWidget		*box;

  /* menubar */
  GtkItemFactory	*factory;
  STStreamMenuItems	*stream_items;
  GtkWidget		*stop_item;
  GtkWidget		*refresh_item;
  GtkWidget		*view_tab_icons_item;
  GtkWidget		*stream_columns_item;
  GtkWidget		*directories_menu;
  GtkWidget		*directory_homepage_item;
  GSList		*directory_items;

  /* toolbar */
  GtkWidget		*toolbar;
  GtkWidget		*tune_in_button;
  GtkWidget		*record_button;
  GtkWidget		*browse_button;
  GtkWidget		*stop_button;
  GtkWidget		*refresh_button;
  GtkWidget		*link;

  /* notebook */
  GtkWidget		*notebook;
  unsigned int		switch_page_hid;

  /* statusbox */
  GtkWidget		*statusbox;

  GSList		*tabs;

  GSList		*tasks;
  STThread		*tune_in_thread;

  /* dialogs */
  GtkWidget		*stream_properties;
  GtkWidget		*preferences;
  GtkWidget		*about;
};

/*** variable declarations ***************************************************/

static GObjectClass *parent_class = NULL;

/*** function declarations ***************************************************/

static void st_shell_class_init (STShellClass *class);
static void st_shell_init (STShell *shell);
static void st_shell_finalize (GObject *object);

static void st_shell_make_window (STShell *shell);
static void st_shell_make_menubar (STShell *shell);
static void st_shell_menubar_make_directory_items (STShell *shell);
static void st_shell_menubar_select_directory_item (STShell *shell);
static void st_shell_make_toolbar (STShell *shell);
static void st_shell_make_notebook (STShell *shell);
static void st_shell_make_statusbox (STShell *shell);

static void st_shell_menubar_init_view_item (STShell *shell,
					     const char *path,
					     gboolean *var);
static void st_shell_menubar_view_toggled_h (GtkCheckMenuItem *item,
					     gpointer user_data);
static void st_shell_menubar_init_toolbar_style_item (STShell *shell,
						      const char *path,
						      int style);
static void st_shell_menubar_toolbar_style_toggled_h (GtkCheckMenuItem *item,
						      gpointer user_data);
static void st_shell_menubar_init_toolbar_size_item (STShell *shell,
						     const char *path,
						     int size);
static void st_shell_menubar_toolbar_size_toggled_h (GtkCheckMenuItem *item,
						     gpointer user_data);

static void st_shell_menubar_directory_toggled_h (GtkCheckMenuItem *item,
						  gpointer user_data);
static gboolean st_shell_window_delete_event_h (GtkWidget *widget,
						GdkEvent *event,
						gpointer user_data);
static void st_shell_toolbar_button_clicked_h (GtkButton *button,
					       gpointer user_data);

static void st_shell_tab_label_drag_begin_h (GtkWidget *widget,
					     GdkDragContext *drag_context,
					     gpointer user_data);
static void st_shell_tab_label_drag_end_h (GtkWidget *widget,
					   GdkDragContext *drag_context,
					   gpointer user_data);
static int st_shell_handlers_compare (gconstpointer a,
				      gconstpointer b,
				      gpointer user_data);
static void st_shell_tab_label_drag_motion_h (GtkWidget *widget,
					      GdkDragContext *drag_context,
					      int x,
					      int y,
					      unsigned int _time,
					      gpointer user_data);

static void st_shell_switch_page_h (GtkNotebook *notebook,
				    GtkNotebookPage *page,
				    unsigned int page_num,
				    gpointer user_data);
static void st_shell_tab_selected (STShell *shell);

static void st_shell_update_visibility (STShell *shell);
static void st_shell_update_toolbar_style (STShell *shell);
static void st_shell_update_toolbar_size (STShell *shell);

static STBrowserTab *st_shell_get_selected_tab (STShell *shell);

static gboolean st_shell_can_stop (STShell *shell);
static void st_shell_stop (STShell *shell);

static gboolean st_shell_can_refresh (STShell *shell);
static void st_shell_refresh (STShell *shell);

static void st_shell_stream_properties_response_h (GtkDialog *dialog,
						   int response,
						   gpointer data);

static void st_shell_present_preferences (STShell *shell);
static void st_shell_present_about (STShell *shell);
static void st_shell_dialog_response_h (GtkDialog *dialog,
					int response,
					gpointer data);

static gboolean st_shell_can_visit_directory_homepage (STShell *shell);
static void st_shell_visit_directory_homepage (STShell *shell);

static void st_shell_stream_properties_update_stream (STShell *shell);

static void st_shell_select_previous_tab (STShell *shell);
static void st_shell_select_next_tab (STShell *shell);

static void st_shell_visit_homepage (STShell *shell, const char *url);

static void st_shell_help (STShell *shell);
static void st_shell_homepage (STShell *shell);

static STBrowserTab *st_shell_get_tab_with_handler (STShell *shell,
						    STHandler *handler);
static void st_shell_select_tab (STShell *shell,
				 STBrowserTab *tab,
				 gboolean force_update);

static void st_shell_launch_stream_task (STShell *shell, STStreamTask task);

static void st_shell_category_selection_changed_h (STCategoryView *view,
						   gpointer user_data);
static void st_shell_stream_selection_changed_h (STStreamView *view,
						 gpointer user_data);
static void st_shell_stream_activated_h (GtkTreeView *treeview,
					 GtkTreePath *arg1,
					 GtkTreeViewColumn *arg2,
					 gpointer user_data);
static void st_shell_tab_refreshing_changed_h (STBrowserTab *tab,
					       gboolean refreshing,
					       gpointer user_data);

/*** implementation **********************************************************/

GType
st_shell_get_type (void)
{
  static GType shell_type = 0;
  
  if (! shell_type)
    {
      static const GTypeInfo shell_info = {
	sizeof(STShellClass),
	NULL,
	NULL,
	(GClassInitFunc) st_shell_class_init,
	NULL,
	NULL,
	sizeof(STShell),
	0,
	(GInstanceInitFunc) st_shell_init,
      };
      
      shell_type = g_type_register_static(G_TYPE_OBJECT,
					  "STShell",
					  &shell_info,
					  0);
    }
  
  return shell_type;
}

static void
st_shell_class_init (STShellClass *class)
{
  GObjectClass *object_class = G_OBJECT_CLASS(class);

  parent_class = g_type_class_peek_parent(class);

  object_class->finalize = st_shell_finalize;
}

static void
st_shell_init (STShell *shell)
{
  STHandler *handler = NULL;

  shell->priv = g_new0(STShellPrivate, 1);

  st_shell_make_window(shell);
  st_shell_make_menubar(shell);
  st_shell_make_toolbar(shell);
  st_shell_make_notebook(shell);
  st_shell_make_statusbox(shell);

  st_shell_update_visibility(shell);
  st_shell_update_toolbar_style(shell);
  st_shell_update_toolbar_size(shell);

  /* select a tab */

  if (st_settings.selected_handler_name)
    handler = st_handlers_find_by_name(st_settings.selected_handler_name);
  if (! handler)
    handler = st_handlers_list->data; /* fallback */
  if (handler)
    st_shell_select_tab(shell, st_shell_get_tab_with_handler(shell, handler), TRUE);

  /* initialize the sensitivity */

  st_shell_update_sensitivity(shell);
}

static void
st_shell_finalize (GObject *object)
{
  STShell *shell = ST_SHELL(object);
  STBrowserTab *selected_tab;

  /* stop everything */
  
  if (st_shell_can_stop(shell))
    st_shell_stop(shell);

  /* store some settings and destroy the window */
  
  selected_tab = st_shell_get_selected_tab(shell);
  if (selected_tab)
    {
      g_free(st_settings.selected_handler_name);
      st_settings.selected_handler_name = g_strdup(st_handler_get_name(selected_tab->handler));
    }
  
  gtk_window_get_size(GTK_WINDOW(shell->priv->window),
		      &st_settings.main_window_width,
		      &st_settings.main_window_height);

  gtk_widget_destroy(shell->priv->window);

  /* cleanup */

  g_object_unref(shell->priv->accel_group);
  g_object_unref(shell->priv->factory);
  st_stream_menu_items_free(shell->priv->stream_items);
  g_slist_free(shell->priv->directory_items);
  g_slist_free(shell->priv->tabs);

  g_free(shell->priv);

  G_OBJECT_CLASS(parent_class)->finalize(object);
}

static void
st_shell_make_window (STShell *shell)
{
  GdkPixbuf *icon;

  shell->priv->window = gtk_window_new(GTK_WINDOW_TOPLEVEL);
  shell->priv->accel_group = gtk_accel_group_new();
  gtk_window_add_accel_group(GTK_WINDOW(shell->priv->window), shell->priv->accel_group);

  icon = gdk_pixbuf_new_from_inline(sizeof(art_streamtuner),
				    art_streamtuner,
				    FALSE,
				    NULL);
  gtk_window_set_icon(GTK_WINDOW(shell->priv->window), icon);
  g_object_unref(icon);

  shell->priv->box = gtk_vbox_new(FALSE, 0);
  gtk_container_add(GTK_CONTAINER(shell->priv->window), shell->priv->box);
  gtk_widget_show(shell->priv->box);

  gtk_window_set_default_size(GTK_WINDOW(shell->priv->window),
			      st_settings.main_window_width,
			      st_settings.main_window_height);

  g_signal_connect(G_OBJECT(shell->priv->window),
		   "delete-event",
		   G_CALLBACK(st_shell_window_delete_event_h),
		   NULL);
}

static gboolean
st_shell_window_delete_event_h (GtkWidget *widget,
				GdkEvent *event,
				gpointer user_data)
{
  st_main_quit();

  return TRUE;
}

static void
st_shell_make_menubar (STShell *shell)
{
  static GtkItemFactoryEntry entries[] = {
    {
      N_("/_Stream"),				NULL,
      NULL,					0,
      "<Branch>",				NULL
    },
    {
      "/Stream/separator1",			NULL,
      NULL,					0,
      "<Separator>",				NULL
    },
    {
      N_("/Stream/_Quit"),			NULL,
      st_main_quit,				0,
      "<StockItem>",				GTK_STOCK_QUIT
    },
    {
      N_("/_Edit"),				NULL,
      NULL,					0,
      "<Branch>",				NULL
    },
    {
      N_("/Edit/_Preferences"),			NULL,
      st_shell_present_preferences,		0,
      "<StockItem>",				GTK_STOCK_PREFERENCES
    },
    {
      N_("/_View"),				NULL,
      NULL,					0,
      "<Branch>",				NULL
    },
    {
      N_("/View/_Toolbar"),			NULL,
      NULL,					0,
      "<CheckItem>",				NULL
    },
    {
      N_("/View/Ta_bs"),			NULL,
      NULL,					0,
      "<CheckItem>",				NULL
    },
    {
      N_("/View/Tab Ic_ons"),			NULL,
      NULL,					0,
      "<CheckItem>",				NULL
    },
    {
      N_("/View/St_atusbar"),			NULL,
      NULL,					0,
      "<CheckItem>",				NULL
    },
    {
      "/View/separator1",			NULL,
      NULL,					0,
      "<Separator>",				NULL
    },
    {
      N_("/View/Toolbar Styl_e"),		NULL,
      NULL,					0,
      "<Branch>",				NULL
    },
    {
      N_("/View/Toolbar Style/_Desktop Default"), NULL,
      NULL,					0,
      "<RadioItem>",				NULL
    },
    {
      "/View/Toolbar Style/separator1",		NULL,
      NULL,					0,
      "<Separator>",				NULL
    },
    {
      N_("/View/Toolbar Style/I_cons Only"),	NULL,
      NULL,					0,
      N_("/View/Toolbar Style/Desktop Default"), NULL
    },
    {
      N_("/View/Toolbar Style/_Text Only"),	NULL,
      NULL,					0,
      N_("/View/Toolbar Style/Desktop Default"), NULL
    },
    {
      N_("/View/Toolbar Style/Text Belo_w Icons"), NULL,
      NULL,					0,
      N_("/View/Toolbar Style/Desktop Default"), NULL
    },
    {
      N_("/View/Toolbar Style/Text Be_side Icons"), NULL,
      NULL,					0,
      N_("/View/Toolbar Style/Desktop Default"), NULL
    },
    {
      N_("/View/Toolbar Si_ze"),		NULL,
      NULL,					0,
      "<Branch>",				NULL
    },
    {
      N_("/View/Toolbar Size/_Desktop Default"), NULL,
      NULL,					0,
      "<RadioItem>",				NULL
    },
    {
      "/View/Toolbar Size/separator1",		NULL,
      NULL,					0,
      "<Separator>",				NULL
    },
    {
      N_("/View/Toolbar Size/_Small"),		NULL,
      NULL,					0,
      N_("/View/Toolbar Size/Desktop Default"),	NULL
    },
    {
      N_("/View/Toolbar Size/_Large"),		NULL,
      NULL,					0,
      N_("/View/Toolbar Size/Desktop Default"),	NULL
    },
    {
      "/View/separator2",			NULL,
      NULL,					0,
      "<Separator>",				NULL
    },
    {
      N_("/View/Stream _Columns"),		NULL,
      NULL,					0,
      "<Branch>",				NULL
    },
    {
      "/View/separator3",			NULL,
      NULL,					0,
      "<Separator>",				NULL
    },
    {
      N_("/View/_Stop"),			"Escape",
      st_shell_stop,				0,
      "<StockItem>",				GTK_STOCK_STOP
    },
    {
      N_("/View/_Refresh"),			"<Control>R",
      st_shell_refresh,				0,
      "<StockItem>",				GTK_STOCK_REFRESH
    },
    {
      N_("/_Directories"),			NULL,
      NULL,					0,
      "<Branch>",				NULL
    },
    {
      N_("/Directories/_Previous"),		NULL,
      st_shell_select_previous_tab,		0,
      "<StockItem>",				GTK_STOCK_GO_BACK
    },
    {
      N_("/Directories/_Next"),			NULL,
      st_shell_select_next_tab,			0,
      "<StockItem>",				GTK_STOCK_GO_FORWARD
    },
    {
      "/Directories/separator1",		NULL,
      NULL,					0,
      "<Separator>",				NULL
    },
    {
      N_("/Directories/_Homepage of Selected Directory"), NULL,
      st_shell_visit_directory_homepage,	0,
      "<StockItem>",				GTK_STOCK_HOME
    },
    {
      N_("/_Help"),				NULL,
      NULL,					0,
      "<Branch>",				NULL
    },
    {
      N_("/Help/_Contents"),			"F1",
      st_shell_help,				0,
      "<StockItem>",				GTK_STOCK_HELP
    },
    {
      N_("/Help/_Homepage"),			NULL,
      st_shell_homepage,			0,
      "<StockItem>",				GTK_STOCK_HOME
    },
    {
      "/Help/separator1",			NULL,
      NULL,					0,
      "<Separator>",				NULL
    },
    {
      N_("/Help/_About"),			NULL,
      st_shell_present_about,			0,
      "<StockItem>",				GTK_STOCK_DIALOG_INFO
    }
  };
  GtkWidget *menu;
  char *accel_path;

  /* create factory */

  shell->priv->factory = gtk_item_factory_new(GTK_TYPE_MENU_BAR,
					      "<streamtuner-Browser>",
					      shell->priv->accel_group);

  gtk_item_factory_set_translate_func(shell->priv->factory,
				      sgtk_translate_func,
				      NULL,
				      NULL);
  gtk_item_factory_create_items(shell->priv->factory,
				G_N_ELEMENTS(entries),
				entries,
				shell);

  /* add stream items */

  menu = gtk_item_factory_get_widget(shell->priv->factory, N_("/Stream"));

  shell->priv->stream_items = st_stream_menu_items_new(shell, shell->priv->accel_group);
  st_stream_menu_items_prepend_to_shell(shell->priv->stream_items, GTK_MENU_SHELL(menu));

  accel_path = g_strdup_printf("<streamtuner-Browser>/Stream/%s", _("Tune in"));
  gtk_accel_map_add_entry(accel_path, GDK_T, GDK_CONTROL_MASK);
  g_free(accel_path);

  accel_path = g_strdup_printf("<streamtuner-Browser>/Stream/%s", _("Delete"));
  gtk_accel_map_add_entry(accel_path, GDK_Delete, 0);
  g_free(accel_path);

  /* remember the items we'll need later on */

  shell->priv->stop_item = gtk_item_factory_get_item(shell->priv->factory, N_("/View/Stop"));
  shell->priv->refresh_item = gtk_item_factory_get_item(shell->priv->factory, N_("/View/Refresh"));
  shell->priv->view_tab_icons_item = gtk_item_factory_get_item(shell->priv->factory, N_("/View/Tab Icons"));
  shell->priv->stream_columns_item = gtk_item_factory_get_item(shell->priv->factory, N_("/View/Stream Columns"));
  shell->priv->directories_menu = gtk_item_factory_get_widget(shell->priv->factory, N_("/Directories"));
  shell->priv->directory_homepage_item = gtk_item_factory_get_item(shell->priv->factory, N_("/Directories/Homepage of Selected Directory"));

  /* add directory items */

  st_shell_menubar_make_directory_items(shell);

  /* configure the view menu items */

  st_shell_menubar_init_view_item(shell, N_("/View/Toolbar"), &st_settings.view_toolbar);
  st_shell_menubar_init_view_item(shell, N_("/View/Tabs"), &st_settings.view_tabs);
  st_shell_menubar_init_view_item(shell, N_("/View/Tab Icons"), &st_settings.view_tab_icons);
  st_shell_menubar_init_view_item(shell, N_("/View/Statusbar"), &st_settings.view_statusbar);

  st_shell_menubar_init_toolbar_style_item(shell, N_("/View/Toolbar Style/Desktop Default"), ST_SHELL_TOOLBAR_STYLE_GTK);
  st_shell_menubar_init_toolbar_style_item(shell, N_("/View/Toolbar Style/Icons Only"), GTK_TOOLBAR_ICONS);
  st_shell_menubar_init_toolbar_style_item(shell, N_("/View/Toolbar Style/Text Only"), GTK_TOOLBAR_TEXT);
  st_shell_menubar_init_toolbar_style_item(shell, N_("/View/Toolbar Style/Text Below Icons"), GTK_TOOLBAR_BOTH);
  st_shell_menubar_init_toolbar_style_item(shell, N_("/View/Toolbar Style/Text Beside Icons"), GTK_TOOLBAR_BOTH_HORIZ);

  st_shell_menubar_init_toolbar_size_item(shell, N_("/View/Toolbar Size/Desktop Default"), ST_SHELL_TOOLBAR_SIZE_GTK);
  st_shell_menubar_init_toolbar_size_item(shell, N_("/View/Toolbar Size/Small"), GTK_ICON_SIZE_SMALL_TOOLBAR);
  st_shell_menubar_init_toolbar_size_item(shell, N_("/View/Toolbar Size/Large"), GTK_ICON_SIZE_LARGE_TOOLBAR);

  /* pack the menubar */
  
  menu = gtk_item_factory_get_widget(shell->priv->factory, "<streamtuner-Browser>");
  gtk_box_pack_start(GTK_BOX(shell->priv->box), menu, FALSE, FALSE, 0);
  gtk_widget_show(menu);
}

static void
st_shell_menubar_make_directory_items (STShell *shell)
{
  GtkWidget *separator;
  GSList *group = NULL;
  GSList *l;

  g_return_if_fail(ST_IS_SHELL(shell));

  /* destroy the old items */
  SG_LIST_FOREACH(l, shell->priv->directory_items)
    gtk_widget_destroy(l->data);

  g_slist_free(shell->priv->directory_items);
  shell->priv->directory_items = NULL;

  separator = gtk_separator_menu_item_new();
  gtk_widget_show(separator);
  gtk_menu_shell_append(GTK_MENU_SHELL(shell->priv->directories_menu), separator);
  shell->priv->directory_items = g_slist_append(shell->priv->directory_items, separator);

  SG_LIST_FOREACH(l, st_handlers_list)
    {
      STHandler *handler = l->data;
      GtkWidget *item;
      unsigned int hid;

      item = gtk_radio_menu_item_new_with_label(group, st_handler_get_label(handler));
      group = gtk_radio_menu_item_get_group(GTK_RADIO_MENU_ITEM(item));

      gtk_widget_show(item);
      gtk_menu_shell_append(GTK_MENU_SHELL(shell->priv->directories_menu), item);

      hid = g_signal_connect(G_OBJECT(item),
			     "toggled",
			     G_CALLBACK(st_shell_menubar_directory_toggled_h),
			     shell);

      g_object_set_data(G_OBJECT(item), "handler", handler);
      g_object_set_data(G_OBJECT(item), "hid", GINT_TO_POINTER(hid));

      shell->priv->directory_items = g_slist_append(shell->priv->directory_items, item);
    }
}

/*
 * Select the menu item corresponding to the currently selected tab.
 */
static void
st_shell_menubar_select_directory_item (STShell *shell)
{
  STBrowserTab *tab;
  GSList *l;
  
  g_return_if_fail(ST_IS_SHELL(shell));

  tab = st_shell_get_selected_tab(shell);

  SG_LIST_FOREACH(l, shell->priv->directory_items)
    {
      GtkWidget *item = l->data;
      STHandler *handler = g_object_get_data(G_OBJECT(item), "handler");
      unsigned int hid = GPOINTER_TO_INT(g_object_get_data(G_OBJECT(item), "hid"));
      
      if (handler == tab->handler)
	{
	  g_signal_handler_block(item, hid);
	  gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(item), TRUE);
	  g_signal_handler_unblock(item, hid);
	  
	  break;
	}
    }
}

static void
st_shell_menubar_init_view_item (STShell *shell,
				 const char *path,
				 gboolean *var)
{
  GtkWidget *item;

  g_return_if_fail(ST_IS_SHELL(shell));
  g_return_if_fail(path != NULL);
  g_return_if_fail(var != NULL);

  item = gtk_item_factory_get_item(shell->priv->factory, path);
  gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(item), *var);

  if (var == &st_settings.view_tabs)
    gtk_widget_set_sensitive(shell->priv->view_tab_icons_item, st_settings.view_tabs);

  g_object_set_data(G_OBJECT(item), "shell", shell);
  g_signal_connect(G_OBJECT(item),
		   "toggled",
		   G_CALLBACK(st_shell_menubar_view_toggled_h),
		   var);
}

static void
st_shell_menubar_view_toggled_h (GtkCheckMenuItem *item, gpointer user_data)
{
  gboolean *var = user_data;
  STShell *shell = g_object_get_data(G_OBJECT(item), "shell");

  *var = gtk_check_menu_item_get_active(item);
  st_shell_update_visibility(shell);

  if (var == &st_settings.view_tabs)
    gtk_widget_set_sensitive(shell->priv->view_tab_icons_item, st_settings.view_tabs);
}

static void
st_shell_menubar_init_toolbar_style_item (STShell *shell,
					  const char *path,
					  int style)
{
  GtkWidget *item;

  g_return_if_fail(ST_IS_SHELL(shell));
  g_return_if_fail(path != NULL);

  item = gtk_item_factory_get_item(shell->priv->factory, path);
  gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(item), st_settings.toolbar_style == style);

  g_object_set_data(G_OBJECT(item), "shell", shell);
  g_signal_connect(G_OBJECT(item),
		   "toggled",
		   G_CALLBACK(st_shell_menubar_toolbar_style_toggled_h),
		   GINT_TO_POINTER(style));
}

static void
st_shell_menubar_toolbar_style_toggled_h (GtkCheckMenuItem *item,
					  gpointer user_data)
{
  if (gtk_check_menu_item_get_active(item))
    {
      int style = GPOINTER_TO_INT(user_data);
      STShell *shell = g_object_get_data(G_OBJECT(item), "shell");

      st_settings.toolbar_style = style;
      st_shell_update_toolbar_style(shell);
    }
}

static void
st_shell_menubar_init_toolbar_size_item (STShell *shell,
					 const char *path,
					 int size)
{
  GtkWidget *item;

  g_return_if_fail(ST_IS_SHELL(shell));
  g_return_if_fail(path != NULL);

  item = gtk_item_factory_get_item(shell->priv->factory, path);
  gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(item), st_settings.toolbar_size == size);

  g_object_set_data(G_OBJECT(item), "shell", shell);
  g_signal_connect(G_OBJECT(item),
		   "toggled",
		   G_CALLBACK(st_shell_menubar_toolbar_size_toggled_h),
		   GINT_TO_POINTER(size));
}

static void
st_shell_menubar_toolbar_size_toggled_h (GtkCheckMenuItem *item,
					 gpointer user_data)
{
  if (gtk_check_menu_item_get_active(item))
    {
      int size = GPOINTER_TO_INT(user_data);
      STShell *shell = g_object_get_data(G_OBJECT(item), "shell");

      st_settings.toolbar_size = size;
      st_shell_update_toolbar_size(shell);
    }
}

static void
st_shell_menubar_directory_toggled_h (GtkCheckMenuItem *item,
				      gpointer user_data)
{
  if (gtk_check_menu_item_get_active(item))
    {
      STShell *shell = user_data;
      STHandler *handler = g_object_get_data(G_OBJECT(item), "handler");
      STBrowserTab *tab = st_shell_get_tab_with_handler(shell, handler);

      st_shell_select_tab(shell, tab, FALSE);
    }
}

static void
st_shell_make_toolbar (STShell *shell)
{
  shell->priv->toolbar = gtk_toolbar_new();

  shell->priv->tune_in_button = gtk_toolbar_insert_stock(GTK_TOOLBAR(shell->priv->toolbar),
							 ST_STOCK_TUNE_IN,
							 _("Listen to the selected stream"),
							 NULL,
							 (GtkSignalFunc) st_shell_toolbar_button_clicked_h,
							 shell,
							 -1);
  shell->priv->record_button = gtk_toolbar_insert_stock(GTK_TOOLBAR(shell->priv->toolbar),
							ST_STOCK_RECORD,
							_("Record the selected stream"),
							NULL,
							(GtkSignalFunc) st_shell_toolbar_button_clicked_h,
							shell,
							-1);
  shell->priv->browse_button = gtk_toolbar_insert_stock(GTK_TOOLBAR(shell->priv->toolbar),
							ST_STOCK_BROWSE,
							_("Browse the selected stream's web page"),
							NULL,
							(GtkSignalFunc) st_shell_toolbar_button_clicked_h,
							shell,
							-1);
  shell->priv->stop_button = gtk_toolbar_insert_stock(GTK_TOOLBAR(shell->priv->toolbar),
						      GTK_STOCK_STOP,
						      _("Stop the currently running tasks"),
						      NULL,
						      (GtkSignalFunc) st_shell_toolbar_button_clicked_h,
						      shell,
						      -1);
  shell->priv->refresh_button = gtk_toolbar_insert_stock(GTK_TOOLBAR(shell->priv->toolbar),
							 GTK_STOCK_REFRESH,
							 _("Refresh the selected category"),
							 NULL,
							 (GtkSignalFunc) st_shell_toolbar_button_clicked_h,
							 shell,
							 -1);

  gtk_toolbar_append_space(GTK_TOOLBAR(shell->priv->toolbar));

  shell->priv->link = st_link_new();
  gtk_widget_show(shell->priv->link);
  gtk_container_set_border_width(GTK_CONTAINER(shell->priv->link), SGTK_HIG_CONTROL_SPACING);

  gtk_toolbar_append_widget(GTK_TOOLBAR(shell->priv->toolbar), shell->priv->link, NULL, NULL);

  gtk_box_pack_start(GTK_BOX(shell->priv->box), shell->priv->toolbar, FALSE, FALSE, 0);
  gtk_widget_show(shell->priv->toolbar);
}

static void
st_shell_toolbar_button_clicked_h (GtkButton *button, gpointer user_data)
{
  STShell *shell = user_data;

  if ((GtkWidget *) button == shell->priv->tune_in_button)
    st_shell_tune_in(shell);
  else if ((GtkWidget *) button == shell->priv->record_button)
    st_shell_record(shell);
  else if ((GtkWidget *) button == shell->priv->browse_button)
    st_shell_browse(shell);
  else if ((GtkWidget *) button == shell->priv->stop_button)
    st_shell_stop(shell);
  else if ((GtkWidget *) button == shell->priv->refresh_button)
    st_shell_refresh(shell);
  else
    g_return_if_reached();
}

static void
st_shell_make_notebook (STShell *shell)
{
  GSList *l;
  GdkPixbuf *drag_pixbuf;

  shell->priv->notebook = gtk_notebook_new();
  gtk_notebook_popup_enable(GTK_NOTEBOOK(shell->priv->notebook));

  /*
   * The tab labels will be moved in realtime by
   * st_shell_tab_label_drag_motion_h(): using a drag icon would be
   * visually redundant, so we use a blank icon.
   */
  drag_pixbuf = gdk_pixbuf_new(GDK_COLORSPACE_RGB, TRUE, 8, 1, 1);
  gdk_pixbuf_fill(drag_pixbuf, 0); /* fill with transparent black */

  SG_LIST_FOREACH(l, st_handlers_list)
    {
      STHandler *handler = l->data;
      GtkWidget *tab;
      GtkWidget *tab_label;
      GtkWidget *menu_label;
      static GtkTargetEntry drag_targets[] = {
	{ "STBrowserTabLabel", 0, 0 }
      };
      STStreamMenuItems *stream_menu_items;
	  
      tab = st_browser_tab_new(handler);
      tab_label = st_browser_tab_get_label(ST_BROWSER_TAB(tab), TRUE);
      menu_label = st_browser_tab_get_label(ST_BROWSER_TAB(tab), FALSE);
      
      stream_menu_items = st_stream_menu_items_new(shell, shell->priv->accel_group);
      st_stream_view_set_menu_items(ST_BROWSER_TAB(tab)->stream_view, stream_menu_items);

      gtk_drag_source_set(tab_label,
			  GDK_BUTTON1_MASK,
			  drag_targets,
			  G_N_ELEMENTS(drag_targets),
			  GDK_ACTION_MOVE);
      gtk_drag_dest_set(tab_label,
			GTK_DEST_DEFAULT_MOTION | GTK_DEST_DEFAULT_DROP,
			drag_targets,
			G_N_ELEMENTS(drag_targets),
			GDK_ACTION_MOVE);
      gtk_drag_source_set_icon_pixbuf(tab_label, drag_pixbuf);

      g_signal_connect(G_OBJECT(tab_label),
		       "drag-begin",
		       G_CALLBACK(st_shell_tab_label_drag_begin_h),
		       shell);
      g_signal_connect(G_OBJECT(tab_label),
		       "drag-end",
		       G_CALLBACK(st_shell_tab_label_drag_end_h),
		       shell);
      g_signal_connect(G_OBJECT(tab_label),
		       "drag-motion",
		       G_CALLBACK(st_shell_tab_label_drag_motion_h),
		       shell);
			
      g_signal_connect(G_OBJECT(tab),
		       "refreshing-changed",
		       G_CALLBACK(st_shell_tab_refreshing_changed_h),
		       shell);

      if (ST_BROWSER_TAB(tab)->category_view)
	g_signal_connect(G_OBJECT(ST_BROWSER_TAB(tab)->category_view),
			 "selection-changed",
			 G_CALLBACK(st_shell_category_selection_changed_h),
			 shell);
      
      g_signal_connect(G_OBJECT(ST_BROWSER_TAB(tab)->stream_view),
		       "selection-changed",
		       G_CALLBACK(st_shell_stream_selection_changed_h),
		       shell);
      g_signal_connect(G_OBJECT(ST_BROWSER_TAB(tab)->stream_view),
		       "row-activated",
		       G_CALLBACK(st_shell_stream_activated_h),
		       shell);

      gtk_widget_show(tab);
      gtk_widget_show(tab_label);
      gtk_widget_show(menu_label);

      gtk_notebook_append_page_menu(GTK_NOTEBOOK(shell->priv->notebook),
				    tab,
				    tab_label,
				    menu_label);

      shell->priv->tabs = g_slist_append(shell->priv->tabs, tab);
    }

  g_object_unref(drag_pixbuf);

  shell->priv->switch_page_hid = g_signal_connect_after(G_OBJECT(shell->priv->notebook),
							"switch-page",
							G_CALLBACK(st_shell_switch_page_h),
							shell);

  gtk_box_pack_start(GTK_BOX(shell->priv->box), shell->priv->notebook, TRUE, TRUE, 0);
  gtk_widget_show(shell->priv->notebook);
}

static void
st_shell_tab_label_drag_begin_h (GtkWidget *widget,
				 GdkDragContext *drag_context,
				 gpointer user_data)
{
  if (! gdk_pointer_is_grabbed())
    {
      STShell *shell = user_data;
      GdkCursor *cursor = gdk_cursor_new(GDK_FLEUR);

      gdk_pointer_grab(shell->priv->notebook->window,
		       FALSE,
		       GDK_BUTTON1_MOTION_MASK | GDK_BUTTON_RELEASE_MASK,
		       NULL,
		       cursor,
		       drag_context->start_time);
      gdk_cursor_unref(cursor);
    }
}

static void
st_shell_tab_label_drag_end_h (GtkWidget *widget,
			       GdkDragContext *drag_context,
			       gpointer user_data)
{
  STShell *shell = user_data;

  /* a tab might have been moved, reorder the handlers list */
  st_handlers_list = g_slist_sort_with_data(st_handlers_list,
					    st_shell_handlers_compare,
					    shell);

  /* reorder the directories menu by recreating it */
  st_shell_menubar_make_directory_items(shell);
  st_shell_menubar_select_directory_item(shell);
}

static int
st_shell_handlers_compare (gconstpointer a,
			   gconstpointer b,
			   gpointer user_data)
{
  STShell *shell = user_data;
  STBrowserTab *tab1 = st_shell_get_tab_with_handler(shell, (STHandler *) a);
  STBrowserTab *tab2 = st_shell_get_tab_with_handler(shell, (STHandler *) b);
  int num1 = gtk_notebook_page_num(GTK_NOTEBOOK(shell->priv->notebook), GTK_WIDGET(tab1));
  int num2 = gtk_notebook_page_num(GTK_NOTEBOOK(shell->priv->notebook), GTK_WIDGET(tab2));

  return (num1 < num2) ? -1 : ((num1 == num2) ? 0 : 1);
}

static void
st_shell_tab_label_drag_motion_h (GtkWidget *widget,
				  GdkDragContext *drag_context,
				  int x,
				  int y,
				  unsigned int _time,
				  gpointer user_data)
{
  GtkWidget *source;

  source = gtk_drag_get_source_widget(drag_context);
  if (source)
    {
      STShell *shell = user_data;
      STBrowserTabLabel *source_tab_label = ST_BROWSER_TAB_LABEL(source);
      STBrowserTabLabel *dest_tab_label = ST_BROWSER_TAB_LABEL(widget);
      int num = gtk_notebook_page_num(GTK_NOTEBOOK(shell->priv->notebook), GTK_WIDGET(dest_tab_label->tab));

      g_return_if_fail(num != -1);
      gtk_notebook_reorder_child(GTK_NOTEBOOK(shell->priv->notebook),
				 GTK_WIDGET(source_tab_label->tab),
				 num);
    }
}

static void
st_shell_switch_page_h (GtkNotebook *notebook,
			GtkNotebookPage *page,
			unsigned int page_num,
			gpointer user_data)
{
  STShell *shell = user_data;
  
  st_shell_tab_selected(shell);
}

static void
st_shell_tab_selected (STShell *shell)
{
  STBrowserTab *tab;
  const char *description;
  char *markup;
  
  g_return_if_fail(ST_IS_SHELL(shell));

  tab = st_shell_get_selected_tab(shell);
  st_shell_update_title(shell);
  
  /* update toolbar */

  description = st_handler_get_description(tab->handler);
  markup = g_strdup_printf("<span size=\"x-large\">%s</span>", description ? description : st_handler_get_label(tab->handler));
  st_link_set_text(ST_LINK(shell->priv->link), markup);
  g_free(markup);

  st_link_set_uri(ST_LINK(shell->priv->link), st_handler_get_home(tab->handler));

  /* update menubar */

  st_shell_menubar_select_directory_item(shell);
  gtk_menu_item_set_submenu(GTK_MENU_ITEM(shell->priv->stream_columns_item), tab->stream_view->columns_menu);

  /* pack tab's statusbar */

  if (GTK_BIN(shell->priv->statusbox)->child)
    gtk_container_remove(GTK_CONTAINER(shell->priv->statusbox), GTK_BIN(shell->priv->statusbox)->child);
  gtk_container_add(GTK_CONTAINER(shell->priv->statusbox), tab->statusbar);

  /* work-around a bug in the Bluecurve theme engine */
  gtk_widget_realize(GTK_WIDGET(ST_STATUSBAR(tab->statusbar)->progress_bar));

  if (! gtk_tree_view_get_model(GTK_TREE_VIEW(tab->stream_view)) && ! tab->refreshing)
    st_browser_tab_update(tab);
  
  st_shell_update_sensitivity(shell);

  if (shell->priv->stream_properties)
    st_stream_properties_dialog_update_sensitivity(ST_STREAM_PROPERTIES_DIALOG(shell->priv->stream_properties));
}

static void
st_shell_make_statusbox (STShell *shell)
{
  shell->priv->statusbox = gtk_event_box_new();

  gtk_box_pack_start(GTK_BOX(shell->priv->box), shell->priv->statusbox, FALSE, FALSE, 0);
  gtk_widget_show(shell->priv->statusbox);
}

static void
st_shell_update_visibility (STShell *shell)
{
  GSList *l;

  g_object_set(shell->priv->toolbar, "visible", st_settings.view_toolbar, NULL);
  gtk_notebook_set_show_tabs(GTK_NOTEBOOK(shell->priv->notebook), st_settings.view_tabs);
  SG_LIST_FOREACH(l, shell->priv->tabs)
    st_browser_tab_set_label_icon_visible(l->data, st_settings.view_tab_icons);
  g_object_set(shell->priv->statusbox, "visible", st_settings.view_statusbar, NULL);
}

static void
st_shell_update_toolbar_style (STShell *shell)
{
  GtkToolbarStyle style = st_settings.toolbar_style;

  if (style == ST_SHELL_TOOLBAR_STYLE_GTK)
    {
      GtkSettings *settings;
      
      settings = gtk_widget_get_settings(GTK_WIDGET(shell->priv->toolbar));
      g_object_get(settings, "gtk-toolbar-style", &style, NULL);
    }

  gtk_toolbar_set_style(GTK_TOOLBAR(shell->priv->toolbar), style);
}

static void
st_shell_update_toolbar_size (STShell *shell)
{
  GtkIconSize size = st_settings.toolbar_size;

  if (size == ST_SHELL_TOOLBAR_SIZE_GTK)
    {
      GtkSettings *settings;
      
      settings = gtk_widget_get_settings(GTK_WIDGET(shell->priv->toolbar));
      g_object_get(settings, "gtk-toolbar-icon-size", &size, NULL);
    }

  gtk_toolbar_set_icon_size(GTK_TOOLBAR(shell->priv->toolbar), size);
}

STShell *
st_shell_new (void)
{
  return g_object_new(ST_TYPE_SHELL, NULL);
}

void
st_shell_show (STShell *shell)
{
  g_return_if_fail(ST_IS_SHELL(shell));

  gtk_widget_show(shell->priv->window);
}

void
st_shell_update_title (STShell *shell)
{
  STBrowserTab *selected_tab;
  GString *title;

  g_return_if_fail(ST_IS_SHELL(shell));

  selected_tab = st_shell_get_selected_tab(shell);
  if (selected_tab)
    {
      STCategoryBag *category_bag;

      title = g_string_new(st_handler_get_label(selected_tab->handler));

      category_bag = st_handler_get_selected_category(selected_tab->handler);
      g_return_if_fail(category_bag != NULL);

      if (ST_CATEGORY(category_bag)->label)
	{
	  g_string_append_printf(title, ", %s", ST_CATEGORY(category_bag)->label);
	  g_object_unref(category_bag);
	}
    }
  else
    title = g_string_new("streamtuner");
  
  gtk_window_set_title(GTK_WINDOW(shell->priv->window), title->str);
  g_string_free(title, TRUE);
}

void
st_shell_update_sensitivity (STShell *shell)
{
  GSList *l;

  g_return_if_fail(ST_IS_SHELL(shell));

  /* menubar */
  st_stream_menu_items_update_sensitivity(shell->priv->stream_items);
  gtk_widget_set_sensitive(shell->priv->stop_item, st_shell_can_stop(shell));
  gtk_widget_set_sensitive(shell->priv->refresh_item, st_shell_can_refresh(shell));
  gtk_widget_set_sensitive(shell->priv->directory_homepage_item, st_shell_can_visit_directory_homepage(shell));

  /* toolbar */
  gtk_widget_set_sensitive(shell->priv->tune_in_button, st_shell_can_tune_in(shell));
  gtk_widget_set_sensitive(shell->priv->record_button, st_shell_can_record(shell));
  gtk_widget_set_sensitive(shell->priv->browse_button, st_shell_can_browse(shell));
  gtk_widget_set_sensitive(shell->priv->stop_button, st_shell_can_stop(shell));
  gtk_widget_set_sensitive(shell->priv->refresh_button, st_shell_can_refresh(shell));

  /* tabs */
  SG_LIST_FOREACH(l, shell->priv->tabs)
    st_browser_tab_update_sensitivity(l->data);
}

static STBrowserTab *
st_shell_get_selected_tab (STShell *shell)
{
  int num;

  g_return_val_if_fail(ST_IS_SHELL(shell), NULL);

  num = gtk_notebook_get_current_page(GTK_NOTEBOOK(shell->priv->notebook));
  return num != -1
    ? (STBrowserTab *) gtk_notebook_get_nth_page(GTK_NOTEBOOK(shell->priv->notebook), num)
    : NULL;
}

static gboolean
st_shell_can_stop (STShell *shell)
{
  STBrowserTab *selected_tab;

  g_return_val_if_fail(ST_IS_SHELL(shell), FALSE);

  selected_tab = st_shell_get_selected_tab(shell);

  return (selected_tab && st_browser_tab_can_stop(selected_tab))
    || shell->priv->tasks != NULL;
}

static void
st_shell_stop (STShell *shell)
{
  GSList *tasks;
  GSList *l;

  g_return_if_fail(ST_IS_SHELL(shell));
  g_return_if_fail(st_shell_can_stop(shell));

  SG_LIST_FOREACH(l, shell->priv->tabs)
    if (st_browser_tab_can_stop(l->data))
      st_browser_tab_stop(l->data);

  /*
   * We use a copy, because shell->priv->tasks will be modified in
   * st_thread_abort().
   */
  tasks = g_slist_copy(shell->priv->tasks);
  SG_LIST_FOREACH(l, tasks)
    st_thread_abort(l->data);
  g_slist_free(tasks);
}

static gboolean
st_shell_can_refresh (STShell *shell)
{
  STBrowserTab *selected_tab;

  g_return_val_if_fail(ST_IS_SHELL(shell), FALSE);

  selected_tab = st_shell_get_selected_tab(shell);

  return selected_tab
    && ST_HANDLER_EVENT_HAS_REFRESH(selected_tab->handler);
}

static void
st_shell_refresh (STShell *shell)
{
  STBrowserTab *selected_tab;

  g_return_if_fail(ST_IS_SHELL(shell));
  g_return_if_fail(st_shell_can_refresh(shell));

  selected_tab = st_shell_get_selected_tab(shell);
  st_browser_tab_refresh(selected_tab);
}

static void
st_shell_launch_stream_task (STShell *shell, STStreamTask task)
{
  STBrowserTab *selected_tab;
  GSList *selected_streams;

  g_return_if_fail(ST_IS_SHELL(shell));

  selected_tab = st_shell_get_selected_tab(shell);
  selected_streams = st_stream_view_get_selected_streams(selected_tab->stream_view);

  /* selected_streams's ownership is taken by st_stream_task_launch() */
  st_stream_task_launch(shell, selected_tab->handler, task, selected_streams);
}

gboolean
st_shell_can_tune_in (STShell *shell)
{
  STBrowserTab *selected_tab;

  g_return_val_if_fail(ST_IS_SHELL(shell), FALSE);

  selected_tab = st_shell_get_selected_tab(shell);

  return selected_tab
    && ST_HANDLER_EVENT_HAS_TUNE_IN(selected_tab->handler)
    && st_stream_view_has_selected_streams(selected_tab->stream_view);
}

void
st_shell_tune_in (STShell *shell)
{
  g_return_if_fail(ST_IS_SHELL(shell));
  g_return_if_fail(st_shell_can_tune_in(shell));

  st_shell_launch_stream_task(shell, ST_STREAM_TASK_TUNE_IN);
}

gboolean
st_shell_can_record (STShell *shell)
{
  STBrowserTab *selected_tab;

  g_return_val_if_fail(ST_IS_SHELL(shell), FALSE);

  selected_tab = st_shell_get_selected_tab(shell);

  return selected_tab
    && st_handler_event_is_bound(selected_tab->handler, ST_HANDLER_EVENT_STREAM_RECORD)
    && st_stream_view_has_selected_streams(selected_tab->stream_view);
}

void
st_shell_record (STShell *shell)
{
  g_return_if_fail(ST_IS_SHELL(shell));
  g_return_if_fail(st_shell_can_record(shell));

  st_shell_launch_stream_task(shell, ST_STREAM_TASK_RECORD);
}

gboolean
st_shell_can_browse (STShell *shell)
{
  STBrowserTab *selected_tab;

  g_return_val_if_fail(ST_IS_SHELL(shell), FALSE);

  selected_tab = st_shell_get_selected_tab(shell);

  return selected_tab
    && st_handler_event_is_bound(selected_tab->handler, ST_HANDLER_EVENT_STREAM_BROWSE)
    && st_stream_view_has_selected_streams(selected_tab->stream_view);
}

void
st_shell_browse (STShell *shell)
{
  g_return_if_fail(ST_IS_SHELL(shell));
  g_return_if_fail(st_shell_can_browse(shell));

  st_shell_launch_stream_task(shell, ST_STREAM_TASK_BROWSE);
}

gboolean
st_shell_can_present_stream_properties (STShell *shell)
{
  STBrowserTab *selected_tab;

  g_return_val_if_fail(ST_IS_SHELL(shell), FALSE);

  selected_tab = st_shell_get_selected_tab(shell);

  return selected_tab
    && st_stream_view_has_selected_streams(selected_tab->stream_view);
}

void
st_shell_present_stream_properties (STShell *shell)
{
  g_return_if_fail(ST_IS_SHELL(shell));

  if (shell->priv->stream_properties)
    gtk_window_present(GTK_WINDOW(shell->priv->stream_properties));
  else
    {
      shell->priv->stream_properties = st_stream_properties_dialog_new(shell);
      g_object_add_weak_pointer(G_OBJECT(shell->priv->stream_properties), (gpointer *) &shell->priv->stream_properties);

      g_signal_connect(G_OBJECT(shell->priv->stream_properties),
		       "response",
		       G_CALLBACK(st_shell_stream_properties_response_h),
		       shell);
      
      st_shell_stream_properties_update_stream(shell);
      gtk_widget_show(shell->priv->stream_properties);
    }
}

static void
st_shell_stream_properties_response_h (GtkDialog *dialog,
				       int response,
				       gpointer data)
{
  STShell *shell = data;
  STBrowserTab *selected_tab = st_shell_get_selected_tab(shell);
  
  switch (response)
    {
    case SGTK_RESPONSE_PREVIOUS:
      st_stream_view_select_previous(selected_tab->stream_view);
      break;

    case SGTK_RESPONSE_NEXT:
      st_stream_view_select_next(selected_tab->stream_view);
      break;

    case GTK_RESPONSE_APPLY:
      st_stream_properties_dialog_apply(ST_STREAM_PROPERTIES_DIALOG(dialog));
      break;

    case GTK_RESPONSE_DELETE_EVENT:
    case GTK_RESPONSE_CANCEL:
      gtk_widget_hide(GTK_WIDGET(dialog));
      st_stream_properties_dialog_cancel(ST_STREAM_PROPERTIES_DIALOG(dialog));
      gtk_widget_destroy(GTK_WIDGET(dialog));
      break;
      
    case GTK_RESPONSE_OK:
      gtk_widget_hide(GTK_WIDGET(dialog));
      st_stream_properties_dialog_apply(ST_STREAM_PROPERTIES_DIALOG(dialog));
      gtk_widget_destroy(GTK_WIDGET(dialog));
      break;
    }
}

static void
st_shell_present_preferences (STShell *shell)
{
  g_return_if_fail(ST_IS_SHELL(shell));

  if (shell->priv->preferences)
    gtk_window_present(GTK_WINDOW(shell->priv->preferences));
  else
    {
      shell->priv->preferences = st_preferences_dialog_new(GTK_WINDOW(shell->priv->window));
      g_object_add_weak_pointer(G_OBJECT(shell->priv->preferences), (gpointer *) &shell->priv->preferences);

      gtk_widget_show(shell->priv->preferences);
    }
}

static void
st_shell_present_about (STShell *shell)
{
  g_return_if_fail(ST_IS_SHELL(shell));

  if (shell->priv->about)
    gtk_window_present(GTK_WINDOW(shell->priv->about));
  else
    {
      shell->priv->about = st_about_dialog_new(GTK_WINDOW(shell->priv->window));
      g_object_add_weak_pointer(G_OBJECT(shell->priv->about), (gpointer *) &shell->priv->about);

      g_signal_connect(G_OBJECT(shell->priv->about),
		       "response",
		       G_CALLBACK(st_shell_dialog_response_h),
		       NULL);
      
      gtk_widget_show(shell->priv->about);
    }
}

static void
st_shell_dialog_response_h (GtkDialog *dialog, int response, gpointer data)
{
  gtk_widget_destroy(GTK_WIDGET(dialog));
}

static gboolean
st_shell_can_visit_directory_homepage (STShell *shell)
{
  STBrowserTab *selected_tab;

  g_return_val_if_fail(ST_IS_SHELL(shell), FALSE);

  selected_tab = st_shell_get_selected_tab(shell);

  return selected_tab && st_handler_get_home(selected_tab->handler);
}

static void
st_shell_visit_directory_homepage (STShell *shell)
{
  STBrowserTab *selected_tab;

  g_return_if_fail(ST_IS_SHELL(shell));
  g_return_if_fail(st_shell_can_visit_directory_homepage(shell));

  selected_tab = st_shell_get_selected_tab(shell);
  st_shell_visit_homepage(shell, st_handler_get_home(selected_tab->handler));
}

gboolean
st_shell_can_add_bookmarks (STShell *shell)
{
  STBrowserTab *selected_tab;

  g_return_val_if_fail(ST_IS_SHELL(shell), FALSE);

  selected_tab = st_shell_get_selected_tab(shell);

  return selected_tab
    && selected_tab->handler != st_bookmarks_handler
    && st_stream_view_has_selected_streams(selected_tab->stream_view);
}

void
st_shell_add_bookmarks (STShell *shell)
{
  g_return_if_fail(ST_IS_SHELL(shell));
  g_return_if_fail(st_shell_can_add_bookmarks(shell));

  st_shell_launch_stream_task(shell, ST_STREAM_TASK_ADD_BOOKMARKS);
}

gboolean
st_shell_can_delete_streams (STShell *shell)
{
  STBrowserTab *selected_tab;

  g_return_val_if_fail(ST_IS_SHELL(shell), FALSE);

  selected_tab = st_shell_get_selected_tab(shell);

  return selected_tab
    && st_handler_event_is_bound(selected_tab->handler, ST_HANDLER_EVENT_STREAM_DELETE)
    && st_stream_view_has_selected_streams(selected_tab->stream_view);
}

void
st_shell_delete_streams (STShell *shell)
{
  STBrowserTab *selected_tab;
  GSList *selected_streams;
  GSList *l;
  gboolean confirmed = TRUE;

  g_return_if_fail(ST_IS_SHELL(shell));
  g_return_if_fail(st_shell_can_delete_streams(shell));

  selected_tab = st_shell_get_selected_tab(shell);
  selected_streams = st_stream_view_get_selected_streams(selected_tab->stream_view);

  if (ST_HANDLER_MUST_CONFIRM_DELETION(selected_tab->handler))
    {
      GtkWidget *dialog;

      dialog = sgtk_message_dialog_new(GTK_WINDOW(shell->priv->window),
				       GTK_STOCK_DIALOG_WARNING,
				       ngettext("Delete selected stream?",
						"Delete selected streams?",
						g_slist_length(selected_streams)),
				       _("The deletion will probably be irreversible."));

      gtk_dialog_add_button(GTK_DIALOG(dialog), GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL);
      gtk_dialog_add_button(GTK_DIALOG(dialog), GTK_STOCK_DELETE, GTK_RESPONSE_YES);

      confirmed = gtk_dialog_run(GTK_DIALOG(dialog)) == GTK_RESPONSE_YES;
      gtk_widget_destroy(dialog);
    }
  
  if (confirmed)
    SG_LIST_FOREACH(l, selected_streams)
      {
	GError *err = NULL;
	
	if (! st_stream_bag_delete(l->data, &err))
	  {
	    char *normalized;
	    
	    normalized = st_dialog_normalize(err->message);
	    g_error_free(err);
	    
	    st_error_dialog(_("Unable to delete stream."), "%s", normalized);
	    g_free(normalized);
	  }
      }
  
  sg_objects_free(selected_streams);
}

gboolean
st_shell_can_select_previous_stream (STShell *shell)
{
  STBrowserTab *selected_tab;

  g_return_val_if_fail(ST_IS_SHELL(shell), FALSE);

  selected_tab = st_shell_get_selected_tab(shell);

  return selected_tab
    && st_stream_view_can_select_previous(selected_tab->stream_view);
}

gboolean
st_shell_can_select_next_stream (STShell *shell)
{
  STBrowserTab *selected_tab;

  g_return_val_if_fail(ST_IS_SHELL(shell), FALSE);

  selected_tab = st_shell_get_selected_tab(shell);

  return selected_tab
    && st_stream_view_can_select_next(selected_tab->stream_view);
}

static void
st_shell_stream_properties_update_stream (STShell *shell)
{
  STBrowserTab *selected_tab;
  GSList *selected_streams;

  g_return_if_fail(ST_IS_SHELL(shell));
  g_return_if_fail(shell->priv->stream_properties != NULL);

  selected_tab = st_shell_get_selected_tab(shell);
  g_return_if_fail(selected_tab != NULL);

  selected_streams = st_stream_view_get_selected_streams(selected_tab->stream_view);
  if (selected_streams)
    {
      st_stream_properties_dialog_set_stream(ST_STREAM_PROPERTIES_DIALOG(shell->priv->stream_properties), selected_streams->data);
      sg_objects_free(selected_streams);
    }
}

static STBrowserTab *
st_shell_get_tab_with_handler (STShell *shell, STHandler *handler)
{
  GSList *l;

  g_return_val_if_fail(ST_IS_SHELL(shell), NULL);
  g_return_val_if_fail(ST_IS_HANDLER(handler), NULL);

  SG_LIST_FOREACH(l, shell->priv->tabs)
    {
      STBrowserTab *tab = l->data;

      if (tab->handler == handler)
	return tab;
    }

  return NULL;
}

static void
st_shell_select_tab (STShell *shell,
		     STBrowserTab *tab,
		     gboolean force_update)
{
  int num;

  g_return_if_fail(ST_IS_SHELL(shell));
  g_return_if_fail(ST_IS_BROWSER_TAB(tab));
  
  num = gtk_notebook_page_num(GTK_NOTEBOOK(shell->priv->notebook), GTK_WIDGET(tab));
  g_return_if_fail(num != -1);

  if (force_update)
    {
      g_signal_handler_block(shell->priv->notebook, shell->priv->switch_page_hid);
      gtk_notebook_set_page(GTK_NOTEBOOK(shell->priv->notebook), num);
      g_signal_handler_unblock(shell->priv->notebook, shell->priv->switch_page_hid);

      st_shell_tab_selected(shell);
    }
  else
    gtk_notebook_set_page(GTK_NOTEBOOK(shell->priv->notebook), num);
}

static void
st_shell_select_previous_tab (STShell *shell)
{
  int num;

  g_return_if_fail(ST_IS_SHELL(shell));

  num = gtk_notebook_get_current_page(GTK_NOTEBOOK(shell->priv->notebook));
  if (--num < 0)
    num = gtk_notebook_get_n_pages(GTK_NOTEBOOK(shell->priv->notebook)) - 1;

  gtk_notebook_set_current_page(GTK_NOTEBOOK(shell->priv->notebook), num);
}

static void
st_shell_select_next_tab (STShell *shell)
{
  int num;
  int npages;

  g_return_if_fail(ST_IS_SHELL(shell));

  num = gtk_notebook_get_current_page(GTK_NOTEBOOK(shell->priv->notebook));
  npages = gtk_notebook_get_n_pages(GTK_NOTEBOOK(shell->priv->notebook));

  if (++num == npages)
    num = 0;

  gtk_notebook_set_current_page(GTK_NOTEBOOK(shell->priv->notebook), num);
}

static void
st_shell_visit_homepage (STShell *shell, const char *url)
{
  GError *err = NULL;

  g_return_if_fail(ST_IS_SHELL(shell));
  g_return_if_fail(url != NULL);

  if (! st_action_run("view-web", url, &err))
    {
      char *normalized;

      normalized = st_dialog_normalize(err->message);

      st_error_dialog(_("Unable to visit the homepage."), "%s", normalized);

      g_free(normalized);
      g_error_free(err);
    }
}

static void
st_shell_help (STShell *shell)
{
  st_show_help(NULL);
}

static void
st_shell_homepage (STShell *shell)
{
  g_return_if_fail(ST_IS_SHELL(shell));

  st_shell_visit_homepage(shell, ST_HOME);
}

void
st_shell_add_stream_task (STShell *shell, STThread *thread, gboolean tune_in)
{
  g_return_if_fail(ST_IS_SHELL(shell));
  g_return_if_fail(thread != NULL);

  if (tune_in)
    {
      if (shell->priv->tune_in_thread)
	st_thread_abort(shell->priv->tune_in_thread);
      shell->priv->tune_in_thread = thread;
    }
  
  shell->priv->tasks = g_slist_append(shell->priv->tasks, thread);
  st_shell_update_sensitivity(shell);
}

void
st_shell_remove_stream_task (STShell *shell, STThread *thread)
{
  g_return_if_fail(ST_IS_SHELL(shell));
  g_return_if_fail(thread != NULL);

  shell->priv->tasks = g_slist_remove(shell->priv->tasks, thread);
  if (thread == shell->priv->tune_in_thread)
    shell->priv->tune_in_thread = NULL;

  st_shell_update_sensitivity(shell);
}

void
st_shell_new_preselection (STShell *shell)
{
  STBrowserTab *tab;
  STStreamBag *stream_bag;

  g_return_if_fail(ST_IS_SHELL(shell));

  /* switch to the preselections tab */

  tab = st_shell_get_tab_with_handler(shell, st_preselections_handler);
  st_shell_select_tab(shell, tab, FALSE);

  /* create a new preselection, select and present it */

  stream_bag = st_preselections_new();
  st_stream_view_select_stream(tab->stream_view, stream_bag);
  st_stream_view_present_stream(tab->stream_view, stream_bag);

  /* present the stream properties dialog and cleanup */

  st_shell_present_stream_properties(shell);
  g_object_unref(stream_bag);
}

GtkWindow *
st_shell_get_transient (STShell *shell)
{
  g_return_val_if_fail(ST_IS_SHELL(shell), NULL);

  return GTK_WINDOW(shell->priv->window);
}

static void
st_shell_category_selection_changed_h (STCategoryView *view,
				       gpointer user_data)
{
  STShell *shell = user_data;

  st_shell_update_title(shell);
  st_shell_update_sensitivity(shell);
}

static void
st_shell_stream_selection_changed_h (STStreamView *view, gpointer user_data)
{
  STShell *shell = user_data;

  st_shell_update_sensitivity(shell);
  if (shell->priv->stream_properties)
    {
      st_stream_properties_dialog_apply(ST_STREAM_PROPERTIES_DIALOG(shell->priv->stream_properties));

      st_stream_properties_dialog_update_sensitivity(ST_STREAM_PROPERTIES_DIALOG(shell->priv->stream_properties));
      st_shell_stream_properties_update_stream(shell);
    }
}

static void
st_shell_stream_activated_h (GtkTreeView *treeview,
			     GtkTreePath *arg1,
			     GtkTreeViewColumn *arg2,
			     gpointer user_data)
{
  STShell *shell = user_data;

  if (st_shell_can_tune_in(shell))
    st_shell_tune_in(shell);
  else if (st_shell_can_browse(shell))
    st_shell_browse(shell);
}

static void
st_shell_tab_refreshing_changed_h (STBrowserTab *tab,
				   gboolean refreshing,
				   gpointer user_data)
{
  STShell *shell = user_data;

  if (refreshing)
    /*
     * The selected_category label might have been changed by url_cb,
     * so we update the shell title.
     */
    st_shell_update_title(shell);

  st_shell_update_sensitivity(shell);
}
