/*
 * $Id: st-link.c,v 1.11.2.2 2004/05/11 15:40:41 jylefort Exp $
 *
 * Copyright (c) 2003, 2004 Jean-Yves Lefort
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of Jean-Yves Lefort nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND
 * CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 * ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include "config.h"
#include <gtk/gtk.h>
#include "gettext.h"
#include "sgtk-util.h"
#include "st-link.h"
#include "st-action.h"
#include "st-dialog-api.h"
#include "st-stock.h"

/*** type definitions ********************************************************/

struct _STLinkPrivate
{
  GtkWidget		*label;
  GtkItemFactory	*factory;

  char			*text;
  char			*uri;
  gboolean		in_link;
};

/*** variable declarations ***************************************************/

static GObjectClass *parent_class = NULL;

static GtkTooltips *tooltips = NULL;

/*** function declarations ***************************************************/

static void st_link_init (STLink *link);
static void st_link_class_init (STLinkClass *class);

static void st_link_finalize (GObject *object);

static void st_link_update_text (STLink *link);
static void st_link_update_cursor (STLink *link);

static gboolean st_link_enter_notify_event_h (STLink *link,
					      GdkEventCrossing *event,
					      gpointer user_data);
static gboolean st_link_leave_notify_event_h (STLink *link,
					      GdkEventCrossing *event,
					      gpointer user_data);
static void st_link_open_activate_h (GtkMenuItem *item,
				     gpointer user_data);
static void st_link_copy_address_activate_h (GtkMenuItem *item,
					     gpointer user_data);
static gboolean st_link_button_press_event_h (STLink *link,
					      GdkEventButton *event,
					      gpointer user_data);
static gboolean st_link_button_release_event_h (STLink *link,
						GdkEventButton *event,
						gpointer user_data);

static void st_link_open (STLink *link);
static void st_link_copy_address (STLink *link);

/*** implementation **********************************************************/

GType
st_link_get_type (void)
{
  static GType link_type = 0;
  
  if (! link_type)
    {
      static const GTypeInfo link_info = {
	sizeof(STLinkClass),
	NULL,
	NULL,
	(GClassInitFunc) st_link_class_init,
	NULL,
	NULL,
	sizeof(STLink),
	0,
	(GInstanceInitFunc) st_link_init,
      };
      
      link_type = g_type_register_static(GTK_TYPE_EVENT_BOX,
					 "STLink",
					 &link_info,
					 0);
    }

  return link_type;
}

static void
st_link_class_init (STLinkClass *class)
{
  GObjectClass *object_class = G_OBJECT_CLASS(class);

  parent_class = g_type_class_peek_parent(class);

  object_class->finalize = st_link_finalize;

  tooltips = gtk_tooltips_new();
}

static void
st_link_init (STLink *link)
{
  GtkWidget *item;
  static GtkItemFactoryEntry entries[] = {
    {
      N_("/_Open Link"),			NULL,
      NULL,					0,
      "<StockItem>",				ST_STOCK_OPEN_LINK
    },
    {
      N_("/_Copy Link Address"),		NULL,
      NULL,					0,
      "<StockItem>",				ST_STOCK_COPY_LINK_ADDRESS
    }
  };

  link->priv = g_new0(STLinkPrivate, 1);

  link->priv->factory = gtk_item_factory_new(GTK_TYPE_MENU, "<streamtuner-Link>", NULL);
  gtk_item_factory_set_translate_func(link->priv->factory, sgtk_translate_func, NULL, NULL);
  gtk_item_factory_create_items(link->priv->factory, G_N_ELEMENTS(entries), entries, NULL);

  item = gtk_item_factory_get_item(link->priv->factory, N_("/Open Link"));
  g_signal_connect(G_OBJECT(item), "activate",
		   G_CALLBACK(st_link_open_activate_h), link);

  item = gtk_item_factory_get_item(link->priv->factory, N_("/Copy Link Address"));
  g_signal_connect(G_OBJECT(item), "activate",
		   G_CALLBACK(st_link_copy_address_activate_h), link);

  link->priv->label = gtk_label_new(NULL);
  gtk_container_add(GTK_CONTAINER(link), link->priv->label);
  gtk_widget_show(link->priv->label);

  g_signal_connect(G_OBJECT(link), "enter-notify-event",
		   G_CALLBACK(st_link_enter_notify_event_h), NULL);
  g_signal_connect(G_OBJECT(link), "leave-notify-event",
		   G_CALLBACK(st_link_leave_notify_event_h), NULL);
  g_signal_connect(G_OBJECT(link), "button-press-event",
		   G_CALLBACK(st_link_button_press_event_h), NULL);
  g_signal_connect(G_OBJECT(link), "button-release-event",
		   G_CALLBACK(st_link_button_release_event_h), NULL);
}

static void
st_link_finalize (GObject *object)
{
  STLink *link = ST_LINK(object);

  g_object_unref(link->priv->factory);
  g_free(link->priv->text);
  g_free(link->priv->uri);
  g_free(link->priv);

  G_OBJECT_CLASS(parent_class)->finalize(object);
}

static void
st_link_update_text (STLink *link)
{
  g_return_if_fail(ST_IS_LINK(link));

  if (link->priv->text)
    {
      char *markup;
      
      if (link->priv->uri)
	markup = g_strdup_printf("<span underline=\"single\">%s</span>", link->priv->text);
      else
	markup = g_strdup(link->priv->text);
      
      gtk_label_set_markup(GTK_LABEL(link->priv->label), markup);
      g_free(markup);
    }
  else
    gtk_label_set_text(GTK_LABEL(link->priv->label), NULL);

  if (link->priv->uri)
    {
      char *tip;

      tip = g_strdup_printf(_("Visit %s"), link->priv->uri);
      gtk_tooltips_set_tip(tooltips, GTK_WIDGET(link), tip, NULL);
      g_free(tip);
    }
  else
    gtk_tooltips_set_tip(tooltips, GTK_WIDGET(link), NULL, NULL);
}

static void
st_link_update_cursor (STLink *link)
{
  g_return_if_fail(ST_IS_LINK(link));

  /* GtkWidget->window is only available if the widget is realized. */
  if (GTK_WIDGET_REALIZED(GTK_WIDGET(link)))
    {
      if (link->priv->uri && link->priv->in_link)
	{
	  GdkCursor *cursor;
	  
	  /* set the proper cursor */
	  cursor = gdk_cursor_new(GDK_HAND2);
	  gdk_window_set_cursor(GTK_WIDGET(link)->window, cursor);
	  gdk_cursor_unref(cursor);
	}
      else
	gdk_window_set_cursor(GTK_WIDGET(link)->window, NULL);
    }
}

static gboolean
st_link_enter_notify_event_h (STLink *link,
			      GdkEventCrossing *event,
			      gpointer user_data)
{
  GtkWidget *event_widget;

  event_widget = gtk_get_event_widget((GdkEvent *) event);
  if (event_widget == (GtkWidget *) link && event->detail != GDK_NOTIFY_INFERIOR)
    link->priv->in_link = TRUE;

  st_link_update_cursor(link);

  return FALSE;
}

static gboolean
st_link_leave_notify_event_h (STLink *link,
			      GdkEventCrossing *event,
			      gpointer user_data)
{
  GtkWidget *event_widget;

  event_widget = gtk_get_event_widget((GdkEvent *) event);
  if (event_widget == (GtkWidget *) link && event->detail != GDK_NOTIFY_INFERIOR)
    link->priv->in_link = FALSE;

  st_link_update_cursor(link);

  return FALSE;
}

static void
st_link_open_activate_h (GtkMenuItem *item, gpointer user_data)
{
  STLink *link = user_data;
  st_link_open(link);
}

static void
st_link_copy_address_activate_h (GtkMenuItem *item, gpointer user_data)
{
  STLink *link = user_data;
  st_link_copy_address(link);
}

static gboolean
st_link_button_press_event_h (STLink *link,
			      GdkEventButton *event,
			      gpointer user_data)
{
  if (event->button == 3 && link->priv->in_link && link->priv->uri)
    gtk_item_factory_popup(link->priv->factory, event->x_root, event->y_root, event->button, event->time);
  
  return FALSE;
}

static gboolean
st_link_button_release_event_h (STLink *link,
				GdkEventButton *event,
				gpointer user_data)
{
  if (event->button == 1 && link->priv->in_link && link->priv->uri)
    {
      st_link_open(link);

      return TRUE;
    }

  return FALSE;
}

static void
st_link_open (STLink *link)
{
  GError *err = NULL;

  g_return_if_fail(ST_IS_LINK(link));
  g_return_if_fail(link->priv->uri != NULL);

  if (! st_action_run("view-web", link->priv->uri, &err))
    {
      char *normalized;
      
      normalized = st_dialog_normalize(err->message);
      
      st_error_dialog(_("Unable to visit the link."), "%s", normalized);
      
      g_free(normalized);
      g_error_free(err);
    }
}

static void
st_link_copy_address (STLink *link)
{
  GtkClipboard *clipboard;

  g_return_if_fail(ST_IS_LINK(link));
  g_return_if_fail(link->priv->uri != NULL);

  clipboard = gtk_clipboard_get(gdk_atom_intern("PRIMARY", TRUE));
  gtk_clipboard_set_text(clipboard, link->priv->uri, -1);
}

GtkWidget *
st_link_new (void)
{
  return g_object_new(ST_TYPE_LINK, NULL);
}

void
st_link_set_text (STLink *link, const char *text)
{
  g_return_if_fail(ST_IS_LINK(link));
  
  g_free(link->priv->text);
  link->priv->text = g_strdup(text);

  st_link_update_text(link);
}

void
st_link_set_uri (STLink *link, const char *uri)
{
  g_return_if_fail(ST_IS_LINK(link));
  
  g_free(link->priv->uri);
  link->priv->uri = g_strdup(uri);

  st_link_update_text(link);
  st_link_update_cursor(link);
}
