/*
 * $Id: st-browser-tab.c,v 1.141.2.3 2004/07/23 15:50:53 jylefort Exp $
 *
 * Copyright (c) 2003, 2004 Jean-Yves Lefort
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of Jean-Yves Lefort nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND
 * CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 * ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include "config.h"
#include <string.h>
#include <gtk/gtk.h>
#include "gettext.h"
#include "sgtk-util.h"
#include "sg-util.h"
#include "st-browser-tab.h"
#include "st-browser-tab-label.h"
#include "st-category-view.h"
#include "st-dialog-api.h"
#include "st-statusbar.h"
#include "st-stream-view.h"
#include "st-handler.h"
#include "st-thread.h"
#include "st-settings.h"
#include "st-category-store.h"
#include "st-stream-store.h"

/*** type definitions ********************************************************/

enum {
  REFRESHING_CHANGED,
  LAST_SIGNAL
};

enum {
  PROP_0,
  PROP_HANDLER
};
    
typedef struct
{
  STBrowserTab		*tab;
  STCategoryBag		*category_bag;
  GTimer		*ui_timer;
  gboolean		ui_first;
  STThread		*thread;
} RefreshThreadInfo;

struct _STBrowserTabPrivate
{
  GSList	*labels;		/* a list of tab labels */
  STThread	*thread;

  unsigned int	context_counts;
  unsigned int	context_thread;
};

/*** variable declarations ***************************************************/

static GObjectClass *parent_class = NULL;
static unsigned int browser_tab_signals[LAST_SIGNAL] = { 0 };

/*** function declarations ***************************************************/

static void st_browser_tab_class_init	(STBrowserTabClass	*class);
static void st_browser_tab_init		(STBrowserTab		*tab);

static GObject *st_browser_tab_constructor (GType type,
					    unsigned int n_construct_properties,
					    GObjectConstructParam *construct_params);

static void st_browser_tab_set_property	(GObject	*object,
					 unsigned int	prop_id,
					 const GValue	*value,
					 GParamSpec	*pspec);
static void st_browser_tab_finalize	(GObject	*object);
static void st_browser_tab_unrealize	(GtkWidget	*widget);

static void st_browser_tab_update_counters	(STBrowserTab    *tab);
static void st_browser_tab_set_refreshing	(STBrowserTab    *tab,
						 gboolean        refreshing);

static void st_browser_tab_finish_refresh	(STBrowserTab    *tab);

static void st_browser_tab_refresh_thread		(gpointer    data);
static void st_browser_tab_refresh_thread_print_cb	(const char  *str,
							 gpointer    data);
static void st_browser_tab_refresh_thread_progress_cb	(double      fraction,
							 gpointer    data);
static void st_browser_tab_refresh_thread_abort_cb	(gpointer    data);

static void st_browser_tab_category_selection_changed_h (STCategoryView *view,
							 gpointer user_data);

static void st_browser_tab_set_streams (STBrowserTab *tab,
					STStreamStore *streams);

/*** implementation **********************************************************/

GType
st_browser_tab_get_type (void)
{
  static GType browser_tab_type = 0;
  
  if (! browser_tab_type)
    {
      static const GTypeInfo browser_tab_info = {
	sizeof(STBrowserTabClass),
	NULL,
	NULL,
	(GClassInitFunc) st_browser_tab_class_init,
	NULL,
	NULL,
	sizeof(STBrowserTab),
	0,
	(GInstanceInitFunc) st_browser_tab_init,
      };
      
      browser_tab_type = g_type_register_static(GTK_TYPE_HPANED,
						"STBrowserTab",
						&browser_tab_info,
						0);
    }

  return browser_tab_type;
}

static void
st_browser_tab_class_init (STBrowserTabClass *class)
{
  GObjectClass *object_class = G_OBJECT_CLASS(class);
  GtkWidgetClass *widget_class = GTK_WIDGET_CLASS(class);

  parent_class = g_type_class_peek_parent(class);

  object_class->constructor = st_browser_tab_constructor;
  object_class->set_property = st_browser_tab_set_property;
  object_class->finalize = st_browser_tab_finalize;

  widget_class->unrealize = st_browser_tab_unrealize;

  g_object_class_install_property(object_class,
				  PROP_HANDLER,
				  g_param_spec_pointer("handler",
						       _("Handler"),
						       _("The tab's STHandler object"),
						       G_PARAM_WRITABLE | G_PARAM_CONSTRUCT_ONLY));

  browser_tab_signals[REFRESHING_CHANGED] = g_signal_new("refreshing-changed",
							 ST_TYPE_BROWSER_TAB,
							 G_SIGNAL_RUN_LAST,
							 G_STRUCT_OFFSET(STBrowserTabClass, refreshing_changed),
							 NULL,
							 NULL,
							 g_cclosure_marshal_VOID__BOOLEAN,
							 G_TYPE_NONE,
							 1,
							 G_TYPE_BOOLEAN);
}

static void
st_browser_tab_init (STBrowserTab *tab)
{
  tab->priv = g_new0(STBrowserTabPrivate, 1);

  tab->handler = NULL;
  tab->statusbar = NULL;
  tab->refreshing = FALSE;
  tab->category_view = NULL;
  tab->stream_view = NULL;
}

static GObject *
st_browser_tab_constructor (GType type,
			    unsigned int n_construct_properties,
			    GObjectConstructParam *construct_params)
{
  GObject *object;
  STBrowserTab *tab;
  GtkWidget *scrolled;

  object = G_OBJECT_CLASS(parent_class)->constructor(type,
						     n_construct_properties,
						     construct_params);
  tab = ST_BROWSER_TAB(object);

  tab->statusbar = st_statusbar_new();
  gtk_widget_show(tab->statusbar);
  g_object_ref(tab->statusbar);

  tab->priv->context_counts = gtk_statusbar_get_context_id(GTK_STATUSBAR(tab->statusbar), _("Counts"));
  tab->priv->context_thread = gtk_statusbar_get_context_id(GTK_STATUSBAR(tab->statusbar), _("Thread messages"));

  if (ST_HANDLER_HAS_CATEGORIES(tab->handler))
    {
      STCategoryStore *categories;

      scrolled = gtk_scrolled_window_new(FALSE, FALSE);
      gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrolled),
				     GTK_POLICY_AUTOMATIC,
				     GTK_POLICY_AUTOMATIC);

      categories = st_handler_get_categories(tab->handler);

      g_signal_connect_swapped(G_OBJECT(categories),
			       "row-deleted",
			       G_CALLBACK(st_browser_tab_update_counters),
			       tab);
      g_signal_connect_swapped(G_OBJECT(categories),
			       "row-inserted",
			       G_CALLBACK(st_browser_tab_update_counters),
			       tab);

      tab->category_view = (STCategoryView *) st_category_view_new(categories);
      g_object_unref(categories);

      gtk_container_add(GTK_CONTAINER(scrolled), GTK_WIDGET(tab->category_view));
      gtk_paned_add1(GTK_PANED(tab), scrolled);
      gtk_widget_show_all(scrolled);

      g_signal_connect(G_OBJECT(tab->category_view),
		       "selection-changed",
		       G_CALLBACK(st_browser_tab_category_selection_changed_h),
		       tab);
    }

  scrolled = gtk_scrolled_window_new(FALSE, FALSE);
  gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrolled),
				 GTK_POLICY_AUTOMATIC,
				 GTK_POLICY_AUTOMATIC);

  tab->stream_view = (STStreamView *) st_stream_view_new(tab->handler);

  gtk_container_add(GTK_CONTAINER(scrolled), GTK_WIDGET(tab->stream_view));
  gtk_paned_add2(GTK_PANED(tab), scrolled);
  gtk_widget_show_all(scrolled);

  gtk_paned_set_position(GTK_PANED(tab), st_handler_get_paned_position(tab->handler));

  return object;
}

static void
st_browser_tab_set_property (GObject *object,
			     unsigned int prop_id,
			     const GValue *value,
			     GParamSpec *pspec)
{
  STBrowserTab *tab = ST_BROWSER_TAB(object);

  switch (prop_id)
    {
    case PROP_HANDLER:
      tab->handler = g_value_get_pointer(value);
      break;

    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
      break;
    }
}

static void
st_browser_tab_finalize (GObject *object)
{
  STBrowserTab *tab = ST_BROWSER_TAB(object);

  g_slist_free(tab->priv->labels);
  g_free(tab->priv);
  g_object_unref(tab->statusbar);

  G_OBJECT_CLASS(parent_class)->finalize(object);
}

static void
st_browser_tab_unrealize (GtkWidget *widget)
{
  STBrowserTab *tab = ST_BROWSER_TAB(widget);

  st_handler_set_paned_position(tab->handler, gtk_paned_get_position(GTK_PANED(tab)));

  GTK_WIDGET_CLASS(parent_class)->unrealize(widget);
}

GtkWidget *
st_browser_tab_get_label (STBrowserTab *tab, gboolean use_event_box)
{
  GtkWidget *label;

  g_return_val_if_fail(ST_IS_BROWSER_TAB(tab), NULL);

  /*
   * A widget can't be shared by multiple containers, so we create a
   * new label for every call of this function.
   */

  label = st_browser_tab_label_new(tab, use_event_box);
  tab->priv->labels = g_slist_append(tab->priv->labels, label);

  return label;
}

void
st_browser_tab_set_label_icon_visible (STBrowserTab *tab, gboolean visible)
{
  GSList *l;

  g_return_if_fail(ST_IS_BROWSER_TAB(tab));

  SG_LIST_FOREACH(l, tab->priv->labels)
    st_browser_tab_label_set_icon_visible(ST_BROWSER_TAB_LABEL(l->data), visible);
}

GtkWidget *
st_browser_tab_new (STHandler *handler)
{
  return g_object_new(ST_TYPE_BROWSER_TAB,
		      "handler", handler,
		      NULL);
}

void
st_browser_tab_update_sensitivity (STBrowserTab *tab)
{
  g_return_if_fail(ST_IS_BROWSER_TAB(tab));

  st_stream_view_update_sensitivity(tab->stream_view);
}

void
st_browser_tab_refresh (STBrowserTab *tab)
{
  STCategoryBag *category_bag;
  RefreshThreadInfo *info;

  g_return_if_fail(ST_IS_BROWSER_TAB(tab));

  category_bag = st_handler_get_selected_category(tab->handler);
  g_return_if_fail(category_bag != NULL);

  if (! st_category_bag_apply_url_cb(category_bag))
    {
      g_object_unref(category_bag);
      return;
    }

  info = g_new(RefreshThreadInfo, 1);

  g_object_ref(tab);
  info->tab = tab;
  info->category_bag = category_bag;
  info->ui_timer = g_timer_new();
  info->ui_first = TRUE;
  info->thread = st_thread_new(tab->handler);
  info->thread->thread_cb = st_browser_tab_refresh_thread;
  info->thread->print_cb = st_browser_tab_refresh_thread_print_cb;
  info->thread->progress_cb = st_browser_tab_refresh_thread_progress_cb;
  info->thread->abort_cb = st_browser_tab_refresh_thread_abort_cb;
  info->thread->cb_data = info;

  if (tab->refreshing)
    {
      st_thread_forget(tab->priv->thread);
      gtk_progress_bar_set_fraction(ST_STATUSBAR(tab->statusbar)->progress_bar, 0);
    }
  else
    st_browser_tab_set_refreshing(tab, TRUE);

  tab->priv->thread = info->thread;
  st_thread_run(tab->priv->thread);
}

static void
st_browser_tab_finish_refresh (STBrowserTab *tab)
{
  g_return_if_fail(ST_IS_BROWSER_TAB(tab));

  gtk_statusbar_pop(GTK_STATUSBAR(tab->statusbar), tab->priv->context_thread);
  st_browser_tab_set_refreshing(tab, FALSE);
}

static void
st_browser_tab_refresh_thread (gpointer data)
{
  RefreshThreadInfo *info = data;
  gboolean status;
  GNode *categories;
  STStreamStore *stream_store;
  GError *err = NULL;

  status = st_handler_refresh(info->tab->handler,
			      info->category_bag,
			      &categories,
			      &stream_store,
			      &err);

  st_thread_validate_callback_return(info->thread, status, err);
  if (! st_thread_is_aborted(info->thread))
    {
      GDK_THREADS_ENTER();
      st_browser_tab_finish_refresh(info->tab);
      gdk_flush();
      GDK_THREADS_LEAVE();
    }
  
  if (status)
    {
      STCategoryStore *category_store;

      GDK_THREADS_ENTER();

      /* set categories */

      category_store = st_handler_get_categories(info->tab->handler);
      st_category_store_clear(category_store);
      st_category_store_append_node(category_store, categories);

      g_object_unref(category_store);
      sg_objects_free_node(categories);

      /* set streams */

      st_browser_tab_set_streams(info->tab, stream_store);
      if (stream_store)
	g_object_unref(stream_store);

      gdk_flush();
      GDK_THREADS_LEAVE();
    }
  else if (err)
    {
      char *normalized;

      normalized = st_dialog_normalize(err->message);
      st_error_dialog(_("Unable to refresh."), "%s", normalized);
      
      g_free(normalized);
      g_error_free(err);
    }
  
  /* cleanup */

  if (info->category_bag)
    g_object_unref(info->category_bag);
  g_object_unref(info->tab);
  g_timer_destroy(info->ui_timer);
  g_free(info);
}


static void
st_browser_tab_refresh_thread_print_cb (const char *str, gpointer data)
{
  RefreshThreadInfo *info = data;
  
  st_statusbar_print(ST_STATUSBAR(info->tab->statusbar), info->tab->priv->context_thread, str);
}

static void
st_browser_tab_refresh_thread_progress_cb (double fraction, gpointer data)
{
  RefreshThreadInfo *info = data;

  if (fraction == -1)		/* -1 means "pulse" */
    {
      if (info->ui_first || g_timer_elapsed(info->ui_timer, 0) > 0.1)
	{
	  gtk_progress_bar_pulse(ST_STATUSBAR(info->tab->statusbar)->progress_bar);
	  g_timer_start(info->ui_timer);
	  info->ui_first = FALSE;
	}
    }
  else
    gtk_progress_bar_set_fraction(ST_STATUSBAR(info->tab->statusbar)->progress_bar, fraction);
}

static void
st_browser_tab_refresh_thread_abort_cb (gpointer data)
{
  RefreshThreadInfo *info = data;

  st_browser_tab_finish_refresh(info->tab);
}

gboolean
st_browser_tab_can_stop (STBrowserTab *tab)
{
  g_return_val_if_fail(ST_IS_BROWSER_TAB(tab), FALSE);

  return tab->refreshing;
}

void
st_browser_tab_stop (STBrowserTab *tab)
{
  g_return_if_fail(ST_IS_BROWSER_TAB(tab));
  g_return_if_fail(st_browser_tab_can_stop(tab));

  st_thread_abort(tab->priv->thread);
}

static void
st_browser_tab_set_refreshing (STBrowserTab *tab, gboolean refreshing)
{
  GSList *l;

  g_return_if_fail(ST_IS_BROWSER_TAB(tab));

  tab->refreshing = refreshing;

  st_statusbar_set_active(ST_STATUSBAR(tab->statusbar), tab->refreshing);
  SG_LIST_FOREACH(l, tab->priv->labels)
    st_browser_tab_label_set_blinking(ST_BROWSER_TAB_LABEL(l->data), tab->refreshing);

  g_signal_emit(tab, browser_tab_signals[REFRESHING_CHANGED], 0, tab->refreshing);
}

void
st_browser_tab_update (STBrowserTab *tab)
{
  STCategoryBag *category_bag;
  STStreamStore *streams;

  g_return_if_fail(ST_IS_BROWSER_TAB(tab));

  category_bag = st_handler_get_selected_category(tab->handler);
  g_return_if_fail(category_bag != NULL);

  streams = st_handler_get_streams(tab->handler, ST_CATEGORY(category_bag)->name);
  g_object_unref(category_bag);

  if (tab->refreshing)
    st_browser_tab_stop(tab);

  st_browser_tab_set_streams(tab, streams);
  if (streams)
    g_object_unref(streams);

  if (ST_HANDLER_EVENT_HAS_REFRESH(tab->handler) && ((! streams) || st_settings.always_refresh))
    st_browser_tab_refresh(tab);
}

static void
st_browser_tab_update_counters (STBrowserTab *tab)
{
  GtkTreeModel *model;
  int n_streams;
  char *streams;
  char *str;

  g_return_if_fail(ST_IS_BROWSER_TAB(tab));

  model = gtk_tree_view_get_model(GTK_TREE_VIEW(tab->stream_view));
  n_streams = model ? st_stream_store_count(ST_STREAM_STORE(model)) : 0;
  streams = g_strdup_printf(ngettext("%i stream",
				     "%i streams",
				     n_streams),
			    n_streams);

  if (tab->category_view)
    {
      int n_categories;
      char *categories;

      model = gtk_tree_view_get_model(GTK_TREE_VIEW(tab->category_view));
      n_categories = model ? st_category_store_count(ST_CATEGORY_STORE(model)) : 0;
      categories = g_strdup_printf(ngettext("%i category",
					    "%i categories",
					    n_categories),
				   n_categories);

      str = g_strdup_printf(_("%s, %s"), categories, streams);
      g_free(categories);
    }
  else
    str = g_strdup(streams);
  
  st_statusbar_print(ST_STATUSBAR(tab->statusbar), tab->priv->context_counts, str);

  g_free(streams);
  g_free(str);
}

static void
st_browser_tab_category_selection_changed_h (STCategoryView *view,
					     gpointer user_data)
{
  STBrowserTab *tab = user_data;

  st_browser_tab_update(tab);
}

static void
st_browser_tab_set_streams (STBrowserTab *tab, STStreamStore *streams)
{
  GtkTreeModel *model;

  g_return_if_fail(ST_IS_BROWSER_TAB(tab));

  model = gtk_tree_view_get_model(GTK_TREE_VIEW(tab->stream_view));
  if (model)
    g_signal_handlers_disconnect_by_func(model, st_browser_tab_update_counters, tab);

  st_stream_view_set_store(tab->stream_view, streams);
  if (streams)
    {
      g_signal_connect_swapped(G_OBJECT(streams),
			       "row-deleted",
			       G_CALLBACK(st_browser_tab_update_counters),
			       tab);
      g_signal_connect_swapped(G_OBJECT(streams),
			       "row-inserted",
			       G_CALLBACK(st_browser_tab_update_counters),
			       tab);
    }

  st_browser_tab_update_counters(tab);
}
