/*
 * $Id: pst.c,v 1.83 2004/01/27 13:52:34 jylefort Exp $
 *
 * Copyright (c) 2003, 2004 Jean-Yves Lefort
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of Jean-Yves Lefort nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND
 * CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 * ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include <Python.h>
#include <streamtuner.h>
#include "gettext.h"
#include "spy.h"
#include "pg.h"

#define PST_API_MAJOR_VERSION		0
#define PST_API_MINOR_VERSION		2

typedef struct
{
  PyObject *cb;
  PyObject *data;
} PythonBinding;

typedef struct
{
  STCategory	category;

  PyObject	*p;
} PythonCategory;

typedef struct
{
  STStream	stream;

  PyObject	*p;
  PyObject	*fields;
} PythonStream;

/*** st-action-api.h *********************************************************/

static PyObject *pst_action_register (PyObject *dummy, PyObject *args);
static PyObject *pst_action_run (PyObject *dummy, PyObject *args);

/*** st-category-api.h *******************************************************/

static PyObject *pst_category_new (PyObject *dummy, PyObject *args);
static PyObject *pst_category_set_name (PyObject *dummy, PyObject *args);
static PyObject *pst_category_get_name (PyObject *dummy, PyObject *args);
static PyObject *pst_category_set_label (PyObject *dummy, PyObject *args);
static PyObject *pst_category_get_label (PyObject *dummy, PyObject *args);
static PyObject *pst_category_set_url_postfix (PyObject *dummy, PyObject *args);
static PyObject *pst_category_get_url_postfix (PyObject *dummy, PyObject *args);

static PythonCategory *pst_category_new_cb (gpointer data);
static void pst_category_free_cb (PythonCategory *category, gpointer data);

/*** st-dialog-api.h *********************************************************/

static PyObject *pst_notice (PyObject *dummy, PyObject *args);
static PyObject *pst_error_dialog (PyObject *dummy, PyObject *args);
static PyObject *pst_search_dialog (PyObject *dummy, PyObject *args);

/* deprecated API */
static PyObject *pst_warning (PyObject *dummy, PyObject *args);
static PyObject *pst_error (PyObject *dummy, PyObject *args);

/*** st-handler-api.h ********************************************************/

static PyObject *pst_handler_new (PyObject *dummy, PyObject *args);
static PyObject *pst_handler_set_label (PyObject *dummy, PyObject *args);
static PyObject *pst_handler_set_copyright (PyObject *dummy, PyObject *args);
static PyObject *pst_handler_set_description (PyObject *dummy, PyObject *args);
static PyObject *pst_handler_set_home (PyObject *dummy, PyObject *args);
static PyObject *pst_handler_set_icon_from_file (PyObject *dummy, PyObject *args);
static PyObject *pst_handler_set_stock_categories (PyObject *dummy, PyObject *args);
static PyObject *pst_handler_add_field (PyObject *dummy, PyObject *args);
static PyObject *pst_handler_bind (PyObject *dummy, PyObject *args);

static gboolean pst_handler_refresh_cb (PythonCategory *category,
					GNode **categories,
					GList **streams,
					PythonBinding *binding,
					GError **err);

static gpointer pst_thread_begin_cb (void);
static void pst_thread_end_cb (gpointer data);

/*** st-handler-field-api.h **************************************************/

static PyObject *pst_handler_field_new (PyObject *dummy, PyObject *args);

/*** st-handlers-api.h *******************************************************/

static PyObject *pst_handlers_add (PyObject *dummy, PyObject *args);

/*** st-m3u-api.h ************************************************************/

static PyObject *pst_m3u_mktemp (PyObject *dummy, PyObject *args);

/*** st-pls-api.h ************************************************************/

static PyObject *pst_pls_parse (PyObject *dummy, PyObject *args);

/*** st-programs-api.h *******************************************************/

/* deprecated API */
static PyObject *pst_programs_register (PyObject *dummy, PyObject *args);
static PyObject *pst_programs_run (PyObject *dummy, PyObject *args);

/*** st-state-api.h **********************************************************/

static PyObject *pst_is_aborted (PyObject *dummy, PyObject *args);

/*** st-stream-api.h *********************************************************/

static PyObject *pst_stream_new (PyObject *dummy, PyObject *args);
static PyObject *pst_stream_set_name (PyObject *dummy, PyObject *args);
static PyObject *pst_stream_get_name (PyObject *dummy, PyObject *args);
static PyObject *pst_stream_set_field (PyObject *dummy, PyObject *args);
static PyObject *pst_stream_get_field (PyObject *dummy, PyObject *args);

static PythonStream *pst_stream_new_cb (gpointer data);
static void pst_stream_field_set_cb (PythonStream *stream,
				     STHandlerField *field,
				     const GValue *value,
				     gpointer data);
static void pst_stream_field_get_cb (PythonStream *stream,
				     STHandlerField *field,
				     GValue *value,
				     gpointer data);
static void pst_stream_free_cb (PythonStream *stream, gpointer data);

static gboolean pst_stream_cb (PythonStream *stream,
			       PythonBinding *binding,
			       GError **err);
static gboolean pst_stream_tune_in_multiple_cb (GSList *streams,
						PythonBinding *binding,
						GError **err);

/*** st-transfer-api.h *******************************************************/

static PyObject *pst_transfer_session_new (PyObject *dummy, PyObject *args);

static PyObject *pst_transfer_session_get (PyObject *dummy, PyObject *args);
static PyObject *pst_transfer_session_get_by_line (PyObject *dummy, PyObject *args);
static void pst_transfer_session_get_by_line_cb (const char *line, gpointer data);

static PyObject *pst_transfer_escape (PyObject *dummy, PyObject *args);

/* deprecated API */
static PyObject *pst_transfer_get_full (PyObject *dummy, PyObject *args);
static PyObject *pst_transfer_get_lines (PyObject *dummy, PyObject *args);

/*** Python-specific API *****************************************************/

static PyObject *pst_find_icon (PyObject *dummy, PyObject *args);

static PyMethodDef pst_methods[] = {
  { "st_action_register", pst_action_register, METH_VARARGS, "st_action_register()", },
  { "st_action_run", pst_action_run, METH_VARARGS, "st_action_run()", },

  { "st_category_new", pst_category_new, METH_VARARGS, "st_category_new()", },
  { "st_category_set_name", pst_category_set_name, METH_VARARGS, "set STCategory->name", },
  { "st_category_get_name", pst_category_get_name, METH_VARARGS, "get STCategory->name", },
  { "st_category_set_label", pst_category_set_label, METH_VARARGS, "set STCategory->label", },
  { "st_category_get_label", pst_category_get_label, METH_VARARGS, "get STCategory->label", },
  { "st_category_set_url_postfix", pst_category_set_url_postfix, METH_VARARGS, "set STCategory->url_postfix", },
  { "st_category_get_url_postfix", pst_category_get_url_postfix, METH_VARARGS, "get STCategory->url_postfix", },

  { "st_notice", pst_notice, METH_VARARGS, "st_notice()", },
  { "st_error_dialog", pst_error_dialog, METH_VARARGS, "st_error_dialog()", },
  { "st_search_dialog", pst_search_dialog, METH_VARARGS, "st_search_dialog()", },
  /* deprecated dialog functions */
  { "st_warning", pst_warning, METH_VARARGS, "st_warning()", },
  { "st_error", pst_error, METH_VARARGS, "st_error()", },

  { "st_handler_new", pst_handler_new, METH_VARARGS, "st_handler_new()", },
  { "st_handler_set_label", pst_handler_set_label, METH_VARARGS, "st_handler_set_label()", },
  { "st_handler_set_copyright", pst_handler_set_copyright, METH_VARARGS, "st_handler_set_copyright()", },
  { "st_handler_set_description", pst_handler_set_description, METH_VARARGS, "st_handler_set_description()", },
  { "st_handler_set_home", pst_handler_set_home, METH_VARARGS, "st_handler_set_home()", },
  { "st_handler_set_icon_from_file", pst_handler_set_icon_from_file, METH_VARARGS, "st_handler_set_icon_from_file()", },
  { "st_handler_set_stock_categories", pst_handler_set_stock_categories, METH_VARARGS, "st_handler_set_stock_categories()", },
  { "st_handler_add_field", pst_handler_add_field, METH_VARARGS, "st_handler_add_field()", },
  { "st_handler_bind", pst_handler_bind, METH_VARARGS, "st_handler_bind()", },
  
  { "st_handler_field_new", pst_handler_field_new, METH_VARARGS, "st_handler_field_new()", },

  { "st_handlers_add", pst_handlers_add, METH_VARARGS, "st_handlers_add()", },

  { "st_m3u_mktemp", pst_m3u_mktemp, METH_VARARGS, "st_m3u_mktemp()", },

  { "st_pls_parse", pst_pls_parse, METH_VARARGS, "st_pls_parse()", },

  /* deprecated st_programs functions */
  { "st_programs_register", pst_programs_register, METH_VARARGS, "st_programs_register()", },
  { "st_programs_run", pst_programs_run, METH_VARARGS, "st_programs_run()", },

  { "st_is_aborted", pst_is_aborted, METH_VARARGS, "st_is_aborted()", },

  { "st_stream_new", pst_stream_new, METH_VARARGS, "new PythonStream", },
  { "st_stream_set_name", pst_stream_set_name, METH_VARARGS, "set STStream->name", },
  { "st_stream_get_name", pst_stream_get_name, METH_VARARGS, "get STStream->name", },
  { "st_stream_set_field", pst_stream_set_field, METH_VARARGS, "set to STStream->fields", },
  { "st_stream_get_field", pst_stream_get_field, METH_VARARGS, "get from STStream->fields", },
  
  { "st_transfer_session_new", pst_transfer_session_new, METH_VARARGS, "st_transfer_session_new()", },
  { "st_transfer_session_get", pst_transfer_session_get, METH_VARARGS, "st_transfer_session_get()", },
  { "st_transfer_session_get_by_line", pst_transfer_session_get_by_line, METH_VARARGS, "st_transfer_session_get_by_line()", },
  { "st_transfer_escape", pst_transfer_escape, METH_VARARGS, "st_transfer_escape()", },
  /* deprecated transfer functions*/
  { "st_transfer_get_full", pst_transfer_get_full, METH_VARARGS, "st_transfer_get_full()", },
  { "st_transfer_get_lines", pst_transfer_get_lines, METH_VARARGS, "st_transfer_get_lines()", },

  { "st_find_icon", pst_find_icon, METH_VARARGS, "foo", },
  
  { NULL, NULL, 0, NULL },
};

static char *private_icons_dir = NULL;

void
pst_init (void)
{
  PyObject *module;

  private_icons_dir = g_build_filename(st_settings_get_private_dir(),
				       "streamtuner-python",
				       "icons",
				       NULL);

  module = Py_InitModule("streamtuner", pst_methods);
  
  PyModule_AddIntConstant(module, "ST_MAJOR_VERSION", st_major_version);
  PyModule_AddIntConstant(module, "ST_MINOR_VERSION", st_minor_version);
  PyModule_AddIntConstant(module, "ST_MICRO_VERSION", st_micro_version);

  PyModule_AddIntConstant(module, "ST_API_MAJOR_VERSION", PST_API_MAJOR_VERSION);
  PyModule_AddIntConstant(module, "ST_API_MINOR_VERSION", PST_API_MINOR_VERSION);

  PyModule_AddIntConstant(module, "ST_HANDLER_EVENT_REFRESH", ST_HANDLER_EVENT_REFRESH);
  PyModule_AddIntConstant(module, "ST_HANDLER_EVENT_RELOAD", ST_HANDLER_EVENT_REFRESH);	/* deprecated */
  PyModule_AddIntConstant(module, "ST_HANDLER_EVENT_STREAM_TUNE_IN", ST_HANDLER_EVENT_STREAM_TUNE_IN);
  PyModule_AddIntConstant(module, "ST_HANDLER_EVENT_STREAM_TUNE_IN_MULTIPLE", ST_HANDLER_EVENT_STREAM_TUNE_IN_MULTIPLE);
  PyModule_AddIntConstant(module, "ST_HANDLER_EVENT_STREAM_RECORD", ST_HANDLER_EVENT_STREAM_RECORD);
  PyModule_AddIntConstant(module, "ST_HANDLER_EVENT_STREAM_BROWSE", ST_HANDLER_EVENT_STREAM_BROWSE);

  PyModule_AddIntConstant(module, "ST_TRANSFER_PASS_NEWLINE", ST_TRANSFER_PASS_NEWLINE);
}

/*** st-action-api.h *********************************************************/

static PyObject *
pst_action_register (PyObject *dummy, PyObject *args)
{
  const char *id;
  const char *label;
  const char *command;

  if (! PyArg_ParseTuple(args, "sss", &id, &label, &command))
    return NULL;

  st_action_register(id, label, command);

  Py_INCREF(Py_None);
  return Py_None;
}

static PyObject *
pst_action_run (PyObject *dummy, PyObject *args)
{
  const char *id;
  const char *uri;
  
  GError *err = NULL;

  if (! PyArg_ParseTuple(args, "ss", &id, &uri))
    return NULL;

  if (! st_action_run(id, uri, &err))
    {
      PyErr_SetString(PyExc_RuntimeError, err->message);
      g_error_free(err);

      return NULL;
    }
  
  Py_INCREF(Py_None);
  return Py_None;
}

/*** st-category-api.h *******************************************************/

static PyObject *
pst_category_new (PyObject *dummy, PyObject *args)
{
  /* no Python arguments */

  PythonCategory *category;
 
  if (! PyArg_ParseTuple(args, ""))
    return NULL;

  category = pst_category_new_cb(NULL);

  Py_INCREF(category->p);
  return category->p;
}

static PyObject *
pst_category_set_name (PyObject *dummy, PyObject *args)
{
  PyObject *pcategory;
  const char *name;

  STCategory *category;

  if (! PyArg_ParseTuple(args, "Os", &pcategory, &name))
    return NULL;

  category = PyCObject_AsVoidPtr(pcategory);
  category->name = g_strdup(name);

  Py_INCREF(Py_None);
  return Py_None;
}

static PyObject *
pst_category_get_name (PyObject *dummy, PyObject *args)
{
  PyObject *pcategory;

  STCategory *category;

  if (! PyArg_ParseTuple(args, "O", &pcategory))
    return NULL;
  
  category = PyCObject_AsVoidPtr(pcategory);
  
  if (category->name)
    return PyString_FromString(category->name);
  else
    {
      Py_INCREF(Py_None);
      return Py_None;
    }
}

static PyObject *
pst_category_set_label (PyObject *dummy, PyObject *args)
{
  PyObject *pcategory;
  const char *label;

  STCategory *category;

  if (! PyArg_ParseTuple(args, "Os", &pcategory, &label))
    return NULL;

  category = PyCObject_AsVoidPtr(pcategory);
  category->label = g_strdup(label);

  Py_INCREF(Py_None);
  return Py_None;
}

static PyObject *
pst_category_get_label (PyObject *dummy, PyObject *args)
{
  PyObject *pcategory;

  STCategory *category;

  if (! PyArg_ParseTuple(args, "O", &pcategory))
    return NULL;
  
  category = PyCObject_AsVoidPtr(pcategory);
  
  if (category->label)
    return PyString_FromString(category->label);
  else
    {
      Py_INCREF(Py_None);
      return Py_None;
    }
}

static PyObject *
pst_category_set_url_postfix (PyObject *dummy, PyObject *args)
{
  PyObject *pcategory;
  const char *url_postfix;

  STCategory *category;

  if (! PyArg_ParseTuple(args, "Os", &pcategory, &url_postfix))
    return NULL;

  category = PyCObject_AsVoidPtr(pcategory);
  category->url_postfix = g_strdup(url_postfix);

  Py_INCREF(Py_None);
  return Py_None;
}

static PyObject *
pst_category_get_url_postfix (PyObject *dummy, PyObject *args)
{
  PyObject *pcategory;
  
  STCategory *category;

  if (! PyArg_ParseTuple(args, "O", &pcategory))
    return NULL;
  
  category = PyCObject_AsVoidPtr(pcategory);
  
  if (category->url_postfix)
    return PyString_FromString(category->url_postfix);
  else
    {
      Py_INCREF(Py_None);
      return Py_None;
    }
}

static PythonCategory *
pst_category_new_cb (gpointer data)
{
  PythonCategory *category;

  category = g_new0(PythonCategory, 1);
  category->p = PyCObject_FromVoidPtr(category, NULL);

  return category;
}

static void
pst_category_free_cb (PythonCategory *category, gpointer data)
{
  Py_DECREF(category->p);
  st_category_free((STCategory *) category);
}

/*** st-dialog-api.h *********************************************************/

static PyObject *
pst_notice (PyObject *dummy, PyObject *args)
{
  const char *message;
  
  if (! PyArg_ParseTuple(args, "s", &message))
    return NULL;
  
  st_notice("%s", message);
  
  Py_INCREF(Py_None);
  return Py_None;
}

static PyObject *
pst_error_dialog (PyObject *dummy, PyObject *args)
{
  const char *primary;
  const char *secondary;
  
  if (! PyArg_ParseTuple(args, "ss", &primary, &secondary))
    return NULL;
  
  st_error_dialog(primary, "%s", secondary);
  
  Py_INCREF(Py_None);
  return Py_None;
}

static PyObject *
pst_search_dialog (PyObject *dummy, PyObject *args)
{
  /* no Python arguments */

  char *str;

  if (! PyArg_ParseTuple(args, ""))
    return NULL;

  str = st_search_dialog();
  if (str)
    {
      PyObject *pstr;
      
      pstr = PyString_FromString(str);
      g_free(str);

      return pstr;
    }
  else
    {
      Py_INCREF(Py_None);
      return Py_None;
    }
}

/*
 * Deprecated.
 */
static PyObject *
pst_warning (PyObject *dummy, PyObject *args)
{
  const char *message;
  
  if (! PyArg_ParseTuple(args, "s", &message))
    return NULL;
  
  st_warning("%s", message);
  
  Py_INCREF(Py_None);
  return Py_None;
}

/*
 * Deprecated.
 */
static PyObject *
pst_error (PyObject *dummy, PyObject *args)
{
  const char *message;
  
  if (! PyArg_ParseTuple(args, "s", &message))
    return NULL;
  
  st_error("%s", message);
  
  Py_INCREF(Py_None);
  return Py_None;
}

/*** st-handler-api.h ********************************************************/

static PyObject *
pst_handler_new (PyObject *dummy, PyObject *args)
{
  const char *name;

  PyObject *phandler;
  STHandler *handler;

  if (! PyArg_ParseTuple(args, "s", &name))
    return NULL;
  
  handler = st_handler_new(name);

  st_handler_bind(handler, ST_HANDLER_EVENT_CATEGORY_NEW,
		  pst_category_new_cb, NULL);
  st_handler_bind(handler, ST_HANDLER_EVENT_CATEGORY_FREE,
		  pst_category_free_cb, NULL);
  st_handler_bind(handler, ST_HANDLER_EVENT_STREAM_NEW,
		  pst_stream_new_cb, NULL);
  st_handler_bind(handler, ST_HANDLER_EVENT_STREAM_FIELD_SET,
		  pst_stream_field_set_cb, NULL);
  st_handler_bind(handler, ST_HANDLER_EVENT_STREAM_FIELD_GET,
		  pst_stream_field_get_cb, NULL);
  st_handler_bind(handler, ST_HANDLER_EVENT_STREAM_FREE,
		  pst_stream_free_cb, NULL);
  st_handler_bind(handler, ST_HANDLER_EVENT_THREAD_BEGIN,
		  pst_thread_begin_cb, NULL);
  st_handler_bind(handler, ST_HANDLER_EVENT_THREAD_END,
		  pst_thread_end_cb, NULL);

  phandler = PyCObject_FromVoidPtr(handler, NULL);

  return phandler;
}

static PyObject *
pst_handler_set_label (PyObject *dummy, PyObject *args)
{
  PyObject *phandler;
  const char *label;

  STHandler *handler;

  if (! PyArg_ParseTuple(args, "Os", &phandler, &label))
    return NULL;

  handler = PyCObject_AsVoidPtr(phandler);
  st_handler_set_label(handler, label);

  Py_INCREF(Py_None);
  return Py_None;
}

static PyObject *
pst_handler_set_copyright (PyObject *dummy, PyObject *args)
{
  PyObject *phandler;
  const char *copyright;

  STHandler *handler;

  if (! PyArg_ParseTuple(args, "Os", &phandler, &copyright))
    return NULL;

  handler = PyCObject_AsVoidPtr(phandler);
  st_handler_set_copyright(handler, copyright);

  Py_INCREF(Py_None);
  return Py_None;
}

static PyObject *
pst_handler_set_description (PyObject *dummy, PyObject *args)
{
  PyObject *phandler;
  const char *description;

  STHandler *handler;

  if (! PyArg_ParseTuple(args, "Os", &phandler, &description))
    return NULL;

  handler = PyCObject_AsVoidPtr(phandler);
  st_handler_set_description(handler, description);

  Py_INCREF(Py_None);
  return Py_None;
}

static PyObject *
pst_handler_set_home (PyObject *dummy, PyObject *args)
{
  PyObject *phandler;
  const char *home;

  STHandler *handler;

  if (! PyArg_ParseTuple(args, "Os", &phandler, &home))
    return NULL;

  handler = PyCObject_AsVoidPtr(phandler);
  st_handler_set_home(handler, home);

  Py_INCREF(Py_None);
  return Py_None;
}

static PyObject *
pst_handler_set_icon_from_file (PyObject *dummy, PyObject *args)
{
  PyObject *phandler;
  const char *filename;

  STHandler *handler;
  GError *err = NULL;

  if (! PyArg_ParseTuple(args, "Os", &phandler, &filename))
    return NULL;

  handler = PyCObject_AsVoidPtr(phandler);

  if (! st_handler_set_icon_from_file(handler, filename, &err))
    {
      PyErr_SetString(PyExc_RuntimeError, err->message);
      g_error_free(err);

      return NULL;
    }
  
  Py_INCREF(Py_None);
  return Py_None;
}

static PyObject *
pst_handler_set_stock_categories (PyObject *dummy, PyObject *args)
{
  PyObject *phandler;
  PyObject *pcategories;

  STHandler *handler;
  GNode *categories;

  if (! PyArg_ParseTuple(args, "OO", &phandler, &pcategories))
    return NULL;

  handler = PyCObject_AsVoidPtr(phandler);
  categories = PyCObject_AsVoidPtr(pcategories);

  st_handler_set_stock_categories(handler, categories);

  Py_INCREF(Py_None);
  return Py_None;
}

static PyObject *
pst_handler_add_field (PyObject *dummy, PyObject *args)
{
  PyObject *phandler;
  PyObject *pfield;

  STHandler *handler;
  STHandlerField *field;

  if (! PyArg_ParseTuple(args, "OO", &phandler, &pfield))
    return NULL;

  handler = PyCObject_AsVoidPtr(phandler);
  field = PyCObject_AsVoidPtr(pfield);

  st_handler_add_field(handler, field);

  Py_INCREF(Py_None);
  return Py_None;
}

static PyObject *
pst_handler_bind (PyObject *dummy, PyObject *args)
{
  PyObject *phandler;
  STHandlerEvent event;
  PyObject *pcb;
  PyObject *pdata;

  STHandler *handler;
  PythonBinding *binding;

  if (! PyArg_ParseTuple(args, "OiOO", &phandler, &event, &pcb, &pdata))
    return NULL;
  
  Py_INCREF(pcb);
  Py_INCREF(pdata);

  handler = PyCObject_AsVoidPtr(phandler);

  binding = g_new(PythonBinding, 1);
  binding->cb = pcb;
  binding->data = pdata;

  switch (event)
    {
    case ST_HANDLER_EVENT_REFRESH:
      st_handler_bind(handler, event, pst_handler_refresh_cb, binding);
      break;

    case ST_HANDLER_EVENT_STREAM_TUNE_IN:
    case ST_HANDLER_EVENT_STREAM_RECORD:
    case ST_HANDLER_EVENT_STREAM_BROWSE:
      st_handler_bind(handler, event, pst_stream_cb, binding);
      break;

    case ST_HANDLER_EVENT_STREAM_TUNE_IN_MULTIPLE:
      st_handler_bind(handler, event, pst_stream_tune_in_multiple_cb, binding);
      break;

    default:
      PyErr_Format(PyExc_RuntimeError, _("unknown event %i"), event);
      return NULL;
    }

  Py_INCREF(Py_None);
  return Py_None;
}

static gboolean
pst_handler_refresh_cb (PythonCategory *category,
			GNode **categories,
			GList **streams,
			PythonBinding *binding,
			GError **err)
{
  PyObject *pargs;
  PyObject *presult;

  PyObject *pcategories;
  PyObject *pstreams;

  gboolean status;

  /* might have been aborted while we were waiting for the GIL */
  if (st_is_aborted())
    return FALSE;

  /* create Python arguments */

  pcategories = PGNode_New(Py_None);
  pstreams = PyList_New(0);

  /* call Python callback */

  pargs = Py_BuildValue("(OOOO)", category->p, pcategories, pstreams, binding->data);
  presult = PyEval_CallObject(binding->cb, pargs);
  Py_DECREF(pargs);

  if (! presult)
    {
      PyErr_Print(); /*
		      * FIXME: of course we need to return the msg
		      * into ERR instead of doing this
		      */
      
      g_set_error(err, 0, 0, _("Python exception occurred"));

      return FALSE;
    }
  
  status = presult != Py_None;
  Py_DECREF(presult);

  /* set back the C arguments */

  *categories = PyCObject_AsVoidPtr(pcategories);
  *streams = PyList_AsGList(pstreams);
  Py_DECREF(pcategories);
  Py_DECREF(pstreams);
  
  return status;
}

static gpointer
pst_thread_begin_cb (void)
{
  PyThreadState *state;

  PyEval_AcquireLock();

  state = PyThreadState_New(spy_main_thread_state->interp);
  PyThreadState_Swap(state);

  return state;
}

static void
pst_thread_end_cb (gpointer data)
{
  PyThreadState *state = data;

  g_assert(state != NULL);

  PyThreadState_Swap(NULL);

  PyThreadState_Clear(state);
  PyThreadState_Delete(state);

  PyEval_ReleaseLock();
}

/*** st-handler-field-api.h **************************************************/

static PyObject *
pst_handler_field_new (PyObject *dummy, PyObject *args)
{
  int id;
  const char *label;
  GType type;
  gboolean visible;

  PyObject *pfield;
  STHandlerField *field;

  /* FIXME: visible doesn't accepts None */
  if (! PyArg_ParseTuple(args, "isii", &id, &label, &type, &visible))
    return NULL;
  
  field = st_handler_field_new(id, label, type, visible);
  pfield = PyCObject_FromVoidPtr(field, NULL);

  return pfield;
}

/*** st-handlers-api.h *******************************************************/

static PyObject *
pst_handlers_add (PyObject *dummy, PyObject *args)
{
  PyObject *phandler;

  STHandler *handler;

  if (! PyArg_ParseTuple(args, "O", &phandler))
    return NULL;
  
  handler = PyCObject_AsVoidPtr(phandler);
  st_handlers_add(handler);

  Py_INCREF(Py_None);
  return Py_None;
}

/*** st-m3u-api.h ************************************************************/

static PyObject *
pst_m3u_mktemp (PyObject *dummy, PyObject *args)
{
  const char *prefix;
  GSList *uri_list;

  GSList *l;
  GError *err = NULL;
  char *filename;
  PyObject *pfilename;

  if (! PyArg_ParseTuple(args, "sO&", &prefix, PyStrings_AsGSList, &uri_list))
    return NULL;
  
  filename = st_m3u_mktemp(prefix, uri_list, &err);

  /* free the list */
  for (l = uri_list; l; l = l->next)
    g_free(l->data);
  g_slist_free(uri_list);

  if (! filename)
    {
      PyErr_SetString(PyExc_RuntimeError, err->message);
      g_error_free(err);

      return NULL;
    }
      
  pfilename = PyString_FromString(filename);
  g_free(filename);

  return pfilename;
}

/*** st-pls-api.h ************************************************************/

static PyObject *
pst_pls_parse (PyObject *dummy, PyObject *args)
{
  const char *playlist;
  
  GSList *url_list;
  GSList *l;
  PyObject *tuple;

  if (! PyArg_ParseTuple(args, "s", &playlist))
    return NULL;

  url_list = st_pls_parse(playlist);
  tuple = PyTuple_FromGSList(url_list);
  
  /* free the list */
  for (l = url_list; l; l = l->next)
    g_free(l->data);
  g_slist_free(url_list);

  return tuple;
}

/*** st-programs-api.h *******************************************************/

/*
 * Deprecated.
 */
static PyObject *
pst_programs_register (PyObject *dummy, PyObject *args)
{
  const char *action;
  const char *command;

  if (! PyArg_ParseTuple(args, "ss", &action, &command))
    return NULL;

  st_programs_register(action, command);

  Py_INCREF(Py_None);
  return Py_None;
}

/*
 * Deprecated.
 */
static PyObject *
pst_programs_run (PyObject *dummy, PyObject *args)
{
  const char *action;
  const char *locator;
  
  GError *err = NULL;

  if (! PyArg_ParseTuple(args, "ss", &action, &locator))
    return NULL;

  if (! st_programs_run(action, locator, &err))
    {
      PyErr_SetString(PyExc_RuntimeError, err->message);
      g_error_free(err);

      return NULL;
    }
  
  Py_INCREF(Py_None);
  return Py_None;
}

/*** st-state-api.h **********************************************************/

static PyObject *
pst_is_aborted (PyObject *dummy, PyObject *args)
{
  /* no Python arguments */

  if (! PyArg_ParseTuple(args, ""))
    return NULL;

  if (st_is_aborted())
    return PyInt_FromLong(1);

  Py_INCREF(Py_None);
  return Py_None;
}

/*** st-stream-api.h *********************************************************/

static PyObject *
pst_stream_new (PyObject *dummy, PyObject *args)
{
  /* no Python arguments */

  PythonStream *stream;

  if (! PyArg_ParseTuple(args, ""))
    return NULL;

  stream = pst_stream_new_cb(NULL);

  Py_INCREF(stream->p);
  return stream->p;
}

static PyObject *
pst_stream_set_name (PyObject *dummy, PyObject *args)
{
  PyObject *pstream;
  const char *name;

  STStream *stream;

  if (! PyArg_ParseTuple(args, "Os", &pstream, &name))
    return NULL;

  stream = PyCObject_AsVoidPtr(pstream);
  stream->name = g_strdup(name);

  Py_INCREF(Py_None);
  return Py_None;
}

static PyObject *
pst_stream_get_name (PyObject *dummy, PyObject *args)
{
  PyObject *pstream;

  STStream *stream;

  if (! PyArg_ParseTuple(args, "O", &pstream))
    return NULL;
  
  stream = PyCObject_AsVoidPtr(pstream);
  
  if (stream->name)
    return PyString_FromString(stream->name);
  else
    {
      Py_INCREF(Py_None);
      return Py_None;
    }
}

static PyObject *
pst_stream_set_field (PyObject *dummy, PyObject *args)
{
  PyObject *pstream;
  PyObject *pid;
  PyObject *pdata;

  PythonStream *stream;

  if (! PyArg_ParseTuple(args, "OOO", &pstream, &pid, &pdata))
    return NULL;

  stream = PyCObject_AsVoidPtr(pstream);
  PyDict_SetItem(stream->fields, pid, pdata);

  Py_INCREF(Py_None);
  return Py_None;
}

static PyObject *
pst_stream_get_field (PyObject *dummy, PyObject *args)
{
  PyObject *pstream;
  PyObject *pid;

  PyObject *pdata;
  PythonStream *stream;

  if (! PyArg_ParseTuple(args, "OO", &pstream, &pid))
    return NULL;

  stream = PyCObject_AsVoidPtr(pstream);
  pdata = PyDict_GetItem(stream->fields, pid);
  
  if (! pdata)
    {
      PyErr_SetString(PyExc_LookupError, _("no such field"));
      return NULL;
    }
  
  Py_INCREF(pdata);
  return pdata;
}

static PythonStream *
pst_stream_new_cb (gpointer data)
{
  PythonStream *stream;

  stream = g_new0(PythonStream, 1);
  stream->fields = PyDict_New();
  stream->p = PyCObject_FromVoidPtr(stream, NULL);

  return stream;
}

static void
pst_stream_field_set_cb (PythonStream *stream,
			 STHandlerField *field,
			 const GValue *value,
			 gpointer data)
{
  PyObject *pid;
  PyObject *pdata;

  if (field->type == G_TYPE_INT)
    pdata = PyInt_FromLong(g_value_get_int(value));
  else if (field->type == G_TYPE_STRING)
    pdata = PyString_FromString(g_value_get_string(value));
  else
    g_return_if_fail(FALSE);

  pid = PyInt_FromLong(field->id);
  PyDict_SetItem(stream->fields, pid, pdata);
  Py_DECREF(pid);
}

static void
pst_stream_field_get_cb (PythonStream *stream,
			 STHandlerField *field,
			 GValue *value,
			 gpointer data)
{
  PyObject *pid;
  PyObject *pdata;

  pid = PyInt_FromLong(field->id);
  pdata = PyDict_GetItem(stream->fields, pid);
  Py_DECREF(pid);

  g_return_if_fail(pdata != NULL);
  
  if (PyInt_Check(pdata))
    g_value_set_int(value, PyInt_AsLong(pdata));
  else if (PyString_Check(pdata))
    g_value_set_string(value, PyString_AsString(pdata));
  else
    g_return_if_fail(FALSE);
}

static void
pst_stream_free_cb (PythonStream *stream, gpointer data)
{
  Py_DECREF(stream->fields);
  Py_DECREF(stream->p);
  st_stream_free((STStream *) stream);
}

static gboolean
pst_stream_cb (PythonStream *stream, PythonBinding *binding, GError **err)
{
  PyObject *pargs;
  PyObject *presult;

  gboolean status;

  /* might have been aborted while we were waiting for the GIL */
  if (st_is_aborted())
    return FALSE;

  pargs = Py_BuildValue("(OO)", stream->p, binding->data);
  presult = PyEval_CallObject(binding->cb, pargs);
  Py_DECREF(pargs);
  
  if (! presult)
    {
      PyErr_Print();

      g_set_error(err, 0, 0, _("Python exception occurred"));

      return FALSE;
    }
  
  status = presult != Py_None;
  Py_DECREF(presult);
  
  return status;
}

static gboolean
pst_stream_tune_in_multiple_cb (GSList *streams, PythonBinding *binding, GError **err)
{
  PyObject *pargs;
  PyObject *presult;

  gboolean status;
  PyObject *tuple;
  GSList *l;
  int i = 0;

  /* might have been aborted while we were waiting for the GIL */
  if (st_is_aborted())
    return FALSE;

  tuple = PyTuple_New(g_slist_length(streams));
  for (l = streams; l; l = l->next)
    {
      PythonStream *stream = l->data;
      
      Py_INCREF(stream->p); /* because PyTuple_SetItem() steals a ref */
      PyTuple_SetItem(tuple, i++, stream->p);
    }
  
  pargs = Py_BuildValue("(OO)", tuple, binding->data);
  Py_DECREF(tuple);

  presult = PyEval_CallObject(binding->cb, pargs);
  Py_DECREF(pargs);
  
  if (! presult)
    {
      PyErr_Print();

      g_set_error(err, 0, 0, _("Python exception occurred"));

      return FALSE;
    }
  
  status = presult != Py_None;
  Py_DECREF(presult);
  
  return status;
}

/*** st-transfer-api.h *******************************************************/

static PyObject *
pst_transfer_session_new (PyObject *dummy, PyObject *args)
{
  /* no Python arguments */

  STTransferSession *session;
  PyObject *psession;

  if (! PyArg_ParseTuple(args, ""))
    return NULL;

  session = st_transfer_session_new();
  psession = PyCObject_FromVoidPtr(session, (void (*) (void *)) st_transfer_session_free);

  return psession;
}

static PyObject *
pst_transfer_session_get (PyObject *dummy, PyObject *args)
{
  PyObject *psession;
  const char *url;
  int flags;
  gboolean fetch_headers;

  STTransferSession *session;
  gboolean status;
  char *headers = NULL;
  char *body;
  GError *err = NULL;
  PyObject *tuple;

  if (! PyArg_ParseTuple(args, "Osii", &psession, &url, &flags, &fetch_headers))
    return NULL;

  session = PyCObject_AsVoidPtr(psession);
  status = st_transfer_session_get(session,
				   url,
				   flags,
				   fetch_headers ? &headers : NULL,
				   &body,
				   &err);
  if (! status)
    {
      if (err)
	{
	  PyErr_SetString(PyExc_RuntimeError, err->message);
	  g_error_free(err);

	  return NULL;
	}
      
      Py_INCREF(Py_None);
      return Py_None;
    }
  
  tuple = Py_BuildValue("(ss)", headers, body);
  g_free(headers);
  g_free(body);

  return tuple;
}

static PyObject *
pst_transfer_session_get_by_line (PyObject *dummy, PyObject *args)
{
  PyObject *psession;
  const char *url;
  int flags;
  PyObject *header_cb;
  PyObject *header_data;
  PyObject *body_cb;
  PyObject *body_data;

  STTransferSession *session;
  PythonBinding header_binding;
  PythonBinding body_binding;
  GError *err = NULL;
  gboolean status;

  if (! PyArg_ParseTuple(args, "OsiOOOO", &psession, &url, &flags, &header_cb, &header_data, &body_cb, &body_data))
    return NULL;
  
  session = PyCObject_AsVoidPtr(psession);

  if (header_cb != Py_None)
    {
      header_binding.cb = header_cb;
      header_binding.data = header_data;
    }
  
  body_binding.cb = body_cb;
  body_binding.data = body_data;

  status = st_transfer_session_get_by_line(session,
					   url,
					   flags,
					   header_cb != Py_None ? pst_transfer_session_get_by_line_cb : NULL,
					   header_cb != Py_None ? &header_binding : NULL,
					   pst_transfer_session_get_by_line_cb,
					   &body_binding,
					   &err);
  if (! status)
    {
      if (err)
	{
	  PyErr_SetString(PyExc_RuntimeError, err->message);
	  g_error_free(err);

	  return NULL;
	}
      
      Py_INCREF(Py_None);
      return Py_None;
    }
  
  return PyInt_FromLong(1);
}

static void
pst_transfer_session_get_by_line_cb (const char *line, gpointer data)
{
  PythonBinding *binding = data;
  PyObject *pargs;
  PyObject *presult;

  pargs = Py_BuildValue("sO", line, binding->data);
  presult = PyEval_CallObject(binding->cb, pargs);
  Py_DECREF(pargs);

  if (presult)
    Py_DECREF(presult);
  else
    PyErr_Print();
}

/*
 * Deprecated.
 */
static PyObject *
pst_transfer_get_full (PyObject *dummy, PyObject *args)
{
  const char *url;

  GError *err = NULL;
  char *data;
  PyObject *string;

  if (! PyArg_ParseTuple(args, "s", &url))
    return NULL;

  data = st_transfer_get_full(url, &err);

  if (! data)
    {
      if (err)
	{
	  PyErr_SetString(PyExc_RuntimeError, err->message);
	  g_error_free(err);

	  return NULL;
	}
      
      Py_INCREF(Py_None);
      return Py_None;
    }

  string = PyString_FromString(data);
  g_free(data);

  return string;
}

/*
 * Deprecated.
 */
static PyObject *
pst_transfer_get_lines (PyObject *dummy, PyObject *args)
{
  const char *url;
  PyObject *pcb;
  PyObject *pdata;

  PythonBinding binding;
  GError *err = NULL;
  gboolean status;

  if (! PyArg_ParseTuple(args, "sOO", &url, &pcb, &pdata))
    return NULL;
  
  binding.cb = pcb;
  binding.data = pdata;

  status = st_transfer_get_lines(url, pst_transfer_session_get_by_line_cb, &binding, &err);

  if (! status)
    {
      if (err)
	{
	  PyErr_SetString(PyExc_RuntimeError, err->message);
	  g_error_free(err);

	  return NULL;
	}
      
      Py_INCREF(Py_None);
      return Py_None;
    }
  
  return PyInt_FromLong(1);
}

static PyObject *
pst_transfer_escape (PyObject *dummy, PyObject *args)
{
  const char *url;

  char *escaped;
  PyObject *string;

  if (! PyArg_ParseTuple(args, "s", &url))
    return NULL;

  escaped = st_transfer_escape(url);
  string = PyString_FromString(escaped);
  g_free(escaped);

  return string;
}

/*** Python-specific API *****************************************************/

static PyObject *
pst_find_icon (PyObject *dummy, PyObject *args)
{
  const char *filename;

  char *pathname;
  PyObject *result = NULL;

  if (! PyArg_ParseTuple(args, "s", &filename))
    return NULL;

  g_assert(private_icons_dir != NULL);

  pathname = g_build_filename(ICONS_DIR, filename, NULL);

  if (g_file_test(pathname, G_FILE_TEST_EXISTS))
    result = PyString_FromString(pathname);
  else
    {
      g_free(pathname);
      pathname = g_build_filename(private_icons_dir, filename, NULL);
      
      if (g_file_test(pathname, G_FILE_TEST_EXISTS))
	result = PyString_FromString(pathname);
    }

  g_free(pathname);

  if (! result)
    {
      Py_INCREF(Py_None);
      result = Py_None;
    }
  
  return result;
}
