/*
 * $Id: spy-scripts.c,v 1.3 2004/01/27 13:52:34 jylefort Exp $
 *
 * This file is part of streamtuner-python.
 *
 * Copyright (c) 2003 Jean-Yves Lefort
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of Jean-Yves Lefort nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND
 * CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 * ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include <streamtuner.h>
#include <string.h>
#include "gettext.h"
#include "spy.h"

static gboolean spy_scripts_scan (const char *dirname, GError **err);

gboolean
spy_scripts_load (GError **err)
{
  GError *tmp_err = NULL;
  char *private_scriptsdir;

  if (g_file_test(SCRIPTS_DIR, G_FILE_TEST_IS_DIR))
    {
      if (! spy_scripts_scan(SCRIPTS_DIR, &tmp_err))
	{
	  st_warning(_("Unable to scan system Python scripts directory: %s"),
		     tmp_err->message);
	  g_clear_error(&tmp_err);
	}
    }

  private_scriptsdir = g_build_filename(st_settings_get_private_dir(),
					"streamtuner-python", "scripts", NULL);
  if (g_file_test(private_scriptsdir, G_FILE_TEST_IS_DIR))
    {
      if (! spy_scripts_scan(private_scriptsdir, &tmp_err))
	{
	  st_warning(_("Unable to scan user Python scripts directory: %s"),
		     tmp_err->message);
	  g_clear_error(&tmp_err);
	}
    }
  g_free(private_scriptsdir);
  
  return TRUE;
}

static gboolean
spy_scripts_scan (const char *dirname, GError **err)
{
  GDir *dir;
  const char *filename;

  g_return_val_if_fail(dirname != NULL, FALSE);

  if (! (dir = g_dir_open(dirname, 0, err)))
    return FALSE;

  while ((filename = g_dir_read_name(dir)))
    {
      GError *load_err = NULL;
      char *pathname;
      char *extension;

      pathname = g_build_filename(dirname, filename, NULL);

      if (g_file_test(pathname, G_FILE_TEST_IS_REGULAR)
	  && (extension = strrchr(filename, '.'))
	  && (! strcmp(++extension, "py")))
	{
	  if (! spy_load_script(pathname, &load_err))
	    {
	      st_warning(_("Unable to load script %s: %s"),
			 pathname, load_err->message);
	      g_clear_error(&load_err);
	    }
	}
      
      g_free(pathname);
    }

  g_dir_close(dir);
  
  return TRUE;
}
