/****************************************************************
*
*                  Container World Project
*
*           (c) 2002-2004 Imperial College London
*
* $RCSfile: Misc.java,v $
* $Revision: 1.3 $
* $Date: 2003/06/11 14:26:39 $
* $Author: marsha02 $
* Log: See end of file
*
****************************************************************/

package cosmic.util;

import java.lang.Double;
import java.lang.Integer;
import java.lang.Math;
import org.w3c.dom.Node;

/**
 * General utility class with various value conversion and input methods.
 * Author: James.A.Marshall@imperial.ac.uk
 *
 */

public class Misc
{
	/** The node currently being read in a DOM document (for error reporting) */
	private static Node msNodeReading = null;

	/**
	 * Converts a source integer to a binary string in two's complement
	 *
	 * @param Source integer (-2^(numBits - 1) <= source < 2^(numBits - 1))
	 * @param Number of bits
	 *
	 * @return Binary string in two's complement
	 *
	 * @throws IllegalArgumentException if source integer cannot be encoded in the specified number of bits
	 *
	 */
	public static String getTwosComplementString(int sourceInt, int numBits)
	{
		if (sourceInt < (Math.pow(2, numBits) / 2) * -1 || sourceInt >= Math.pow(2, numBits) / 2)
		{
			throw new IllegalArgumentException(sourceInt + " cannot be encoded in " + numBits + " bits using two's complement");
		}
		String twosComplementString;

		if (sourceInt < 0)
		{
			// number is negative
			sourceInt *= -1;
			sourceInt = ~sourceInt;
			sourceInt += 1;
		}
		twosComplementString = Integer.toBinaryString(sourceInt);
		if (twosComplementString.length() < numBits)
		{
			int l;
			char[] paddingBits = new char[numBits - twosComplementString.length()];
			for (l = 0; l < numBits - twosComplementString.length(); l++)
			{
				paddingBits[l] = '0';
			}
			String paddingString = new String(paddingBits);
			twosComplementString = paddingString.concat(twosComplementString);
		}

		return twosComplementString.substring(twosComplementString.length() - numBits);
	}

	/**
	 * Converts a binary string in two's complement into an integer
	 *
	 * @param Binary string in two's complement
	 *
	 * @return Integer value of two's complement binary string
	 *
	 * @throws NumberFormatException if string is not a binary string
	 *
	 */
	public static int getIntFromTwosComplementString(String string) throws NumberFormatException
	{
		int intValue, paddingBits;

		try
		{
			intValue = Integer.parseInt(string, 2);
		}
		catch (NumberFormatException exception)
		{
			throw exception;
		}
		if (string.charAt(0) == '1')
		{
			// number is negative
			paddingBits = Integer.MAX_VALUE;
			paddingBits = paddingBits << string.length();
			intValue += paddingBits;
			intValue = ~intValue;
			intValue += 1;
			intValue *= -1;
		}

		return intValue;
	}

	/**
     * Gets an integer from the specified string
     *
     * @param string (!= null)
     *
     * @return integer value of string
     *
     * @throws NumberFormatException if string does not represent a valid integer
     *
     */
	public static int getIntegerFromString(String string) throws NumberFormatException
	{
		if (string == null)
		{
			throw new IllegalArgumentException("getIntegerFromString called with null string");
		}
		int value;

		try
		{
			value = Integer.parseInt(string);
		}
		catch (NumberFormatException exception)
		{
			throw exception;
		}

		return value;
	}

	/**
     * Gets an integer from the specified string in the specified interval
     *
     * @param string (!= null)
     * @param lower bound of interval
     * @param upper bound of interval
     * @param lower interval bound closed?
     * @param upper interval bound closed?
     *
     * @return integer value of string
     *
     * @throws NumberFormatException if string does not represent a valid integer
     * @throws IllegalArgumentException if integer is not within interval
     *
     */
	public static int getIntegerFromStringInInterval(String string, int lowerBound, int upperBound, boolean closedLowerBound, boolean closedUpperBound) throws NumberFormatException
	{
		if (string == null)
		{
			throw new IllegalArgumentException("getIntegerFromStringInInterval called with null string");
		}
		int value;

		try
		{
			value = Integer.parseInt(string);
		}
		catch (NumberFormatException exception)
		{
			throw exception;
		}
		if (((closedLowerBound && value < lowerBound) || (!closedLowerBound && value <= lowerBound)) || ((closedUpperBound && value > upperBound) || (!closedUpperBound && value >= upperBound)))
		{
			String exceptionString = new String("value violates interval: " + lowerBound);
			if (closedLowerBound)
			{
				exceptionString = exceptionString.concat(" <= " + value);
			}
			else
			{
				exceptionString = exceptionString.concat(" < " + value);
			}
			if (closedUpperBound)
			{
				exceptionString = exceptionString.concat(" <= " + upperBound);
			}
			else
			{
				exceptionString = exceptionString.concat(" < " + upperBound);
			}
			throw new IllegalArgumentException(exceptionString);
		}

		return value;
	}

	/**
     * Gets an integer from the specified string with the specified lower bound
     *
     * @param string (!= null)
     * @param lower bound of interval
     * @param lower interval bound closed?
     *
     * @return integer value of string
     *
     * @throws NumberFormatException if string does not represent a valid integer
     * @throws IllegalArgumentException if integer violates lower bound
     *
     */
	public static int getIntegerFromStringWithLowerBound(String string, int lowerBound, boolean closedLowerBound)
	{
		if (string == null)
		{
			throw new IllegalArgumentException("getIntegerFromStringWithLowerBound called with null string");
		}
		int value;

		try
		{
			value = Integer.parseInt(string);
		}
		catch (NumberFormatException exception)
		{
			throw exception;
		}
		if ((closedLowerBound && value < lowerBound) || (!closedLowerBound && value <= lowerBound))
		{
			String exceptionString = new String("value violates lower bound: " + lowerBound);
			if (closedLowerBound)
			{
				exceptionString = exceptionString.concat(" <= " + value);
			}
			else
			{
				exceptionString = exceptionString.concat(" < " + value);
			}
			throw new IllegalArgumentException(exceptionString);
		}

		return value;
	}

	/**
     * Gets an integer from the specified string with the specified upper bound
     *
     * @param string (!= null)
     * @param upper bound of interval
     * @param upper interval bound closed?
     *
     * @return integer value of string
     *
     * @throws NumberFormatException if string does not represent a valid integer
     * @throws IllegalArgumentException if integer violates upper bound
     *
     */
	public static int getIntegerFromStringWithUpperBound(String string, int upperBound, boolean closedUpperBound)
	{
		if (string == null)
		{
			throw new IllegalArgumentException("getIntegerFromStringWithUpperBound called with null string");
		}
		int value;

		try
		{
			value = Integer.parseInt(string);
		}
		catch (NumberFormatException exception)
		{
			throw exception;
		}
		if ((closedUpperBound && value > upperBound) || (!closedUpperBound && value >= upperBound))
		{
			String exceptionString = new String("value violates upper bound: " + value);
			if (closedUpperBound)
			{
				exceptionString = exceptionString.concat(" <= " + upperBound);
			}
			else
			{
				exceptionString = exceptionString.concat(" < " + upperBound);
			}
			throw new IllegalArgumentException(exceptionString);
		}

		return value;
	}

	/**
     * Gets a double from the specified string
     *
     * @param string (!= null)
     *
     * @return double value of string
     *
     * @throws NumberFormatException if string does not represent a valid double
     *
     */
	public static double getDoubleFromString(String string) throws NumberFormatException
	{
		if (string == null)
		{
			throw new IllegalArgumentException("getDoubleFromString called with null string");
		}
		double value;

		try
		{
			value = Double.parseDouble(string);
		}
		catch (NumberFormatException exception)
		{
			throw exception;
		}

		return value;
	}

	/**
     * Gets a double from the specified string in the specified interval
     *
     * @param string (!= null)
     * @param lower bound of interval
     * @param upper bound of interval
     * @param lower interval bound closed?
     * @param upper interval bound closed?
     *
     * @return double value of string
     *
     * @throws NumberFormatException if string does not represent a valid double
     * @throws IllegalArgumentException if double is not within interval
     *
     */
	public static double getDoubleFromStringInInterval(String string, double lowerBound, double upperBound, boolean closedLowerBound, boolean closedUpperBound) throws NumberFormatException
	{
		if (string == null)
		{
			throw new IllegalArgumentException("getDoubleFromStringInInterval called with null string");
		}
		double value;

		try
		{
			value = Double.parseDouble(string);
		}
		catch (NumberFormatException exception)
		{
			throw exception;
		}
		if (((closedLowerBound && value < lowerBound) || (!closedLowerBound && value <= lowerBound)) || ((closedUpperBound && value > upperBound) || (!closedUpperBound && value >= upperBound)))
		{
			String exceptionString = new String("value violates interval: " + lowerBound);
			if (closedLowerBound)
			{
				exceptionString = exceptionString.concat(" <= " + value);
			}
			else
			{
				exceptionString = exceptionString.concat(" < " + value);
			}
			if (closedUpperBound)
			{
				exceptionString = exceptionString.concat(" <= " + upperBound);
			}
			else
			{
				exceptionString = exceptionString.concat(" < " + upperBound);
			}
			throw new IllegalArgumentException(exceptionString);
		}

		return value;
	}

	/**
     * Gets a double from the specified string with the specified lower bound
     *
     * @param string (!= null)
     * @param lower bound of interval
     * @param lower interval bound closed?
     *
     * @return double value of string
     *
     * @throws NumberFormatException if string does not represent a valid double
     * @throws IllegalArgumentException if double violates lower bound
     *
     */
	public static double getDoubleFromStringWithLowerBound(String string, double lowerBound, boolean closedLowerBound)
	{
		if (string == null)
		{
			throw new IllegalArgumentException("getDoubleFromStringWithLowerBound called with null string");
		}
		double value;

		try
		{
			value = Double.parseDouble(string);
		}
		catch (NumberFormatException exception)
		{
			throw exception;
		}
		if ((closedLowerBound && value < lowerBound) || (!closedLowerBound && value <= lowerBound))
		{
			String exceptionString = new String("value violates lower bound: " + lowerBound);
			if (closedLowerBound)
			{
				exceptionString = exceptionString.concat(" <= " + value);
			}
			else
			{
				exceptionString = exceptionString.concat(" < " + value);
			}
			throw new IllegalArgumentException(exceptionString);
		}

		return value;
	}

	/**
     * Gets a double from the specified string with the specified upper bound
     *
     * @param string (!= null)
     * @param upper bound of interval
     * @param upper interval bound closed?
     *
     * @return double value of string
     *
     * @throws NumberFormatException if string does not represent a valid double
     * @throws IllegalArgumentException if double violates upper bound
     *
     */
	public static double getDoubleFromStringWithUpperBound(String string, double upperBound, boolean closedUpperBound)
	{
		if (string == null)
		{
			throw new IllegalArgumentException("getDoubleFromStringWithUpperBound called with null string");
		}
		double value;

		try
		{
			value = Double.parseDouble(string);
		}
		catch (NumberFormatException exception)
		{
			throw exception;
		}
		if ((closedUpperBound && value > upperBound) || (!closedUpperBound && value >= upperBound))
		{
			String exceptionString = new String("value violates upper bound: " + value);
			if (closedUpperBound)
			{
				exceptionString = exceptionString.concat(" <= " + upperBound);
			}
			else
			{
				exceptionString = exceptionString.concat(" < " + upperBound);
			}
			throw new IllegalArgumentException(exceptionString);
		}

		return value;
	}

	/**
     * Gets the first child element node of the given input node in a DOM document
     *
     * @param element node in DOM document (!= null)
     *
     * @return first child element node of input node (null indicates no child element node)
     *
     */
	public static Node getFirstChildElementNode(Node node)
	{
		if (node == null)
		{
			throw new IllegalArgumentException("getFirstChildElementNode called with null node");
		}

		return Misc.getNextElementNode(node.getFirstChild());
	}

	/**
     * Gets the next element node after the given input node in a DOM document
     *
     * @param element node in DOM document (!= null)
     *
     * @return next element node after input node (null indicates no next element node)
     *
     */
	public static Node getNextElementNode(Node node)
	{
		if (node == null)
		{
			throw new IllegalArgumentException("getNextElementNode called with null node");
		}

		node = node.getNextSibling();
		while (node != null && node.getNodeType() != org.w3c.dom.Node.ELEMENT_NODE)
		{
			node = node.getNextSibling();
		}
		msNodeReading = node;

		return node;
	}

	/**
     * Gets the name of the node currently being read in a DOM document
     *
     * @return name of the node currently being read (null indicates no node currently being read)
     *
     */
	public static String getNodeReadingName()
	{
		if (msNodeReading == null)
		{
			return null;
		}
		else
		{
			return msNodeReading.getNodeName();
		}
	}

	/**
     * Private constructor for Misc class to prevent instantiation of Misc objects
     *
     */
	private Misc()
	{
	}
}

/****************************************************************
*
*                              File log
*
* $Log: Misc.java,v $
* Revision 1.3  2003/06/11 14:26:39  marsha02
* James Marshall: changed two's complement methods to be static
*
* Revision 1.2  2003/05/29 15:20:34  marsha02
* James Marshall: corrected errors in comments
*
* Revision 1.1  2003/05/29 10:40:57  marsha02
* James Marshall: new class containing miscellaneous methods relating to value conversion and input
*
*
*
*
****************************************************************/