/*  LibKiazma 0.2
 *  Copyright (C) 2007 Roberto -MadBob- Guido <m4db0b@users.sourceforge.net>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

/**
	@file	kiazma_metadata_set_collector.c
*/

#include "libkiazma.h"
#include "kiazma_metadata_set_collector.h"

enum {
	METADATA_STRING_POS,
	METADATA_POS,
	OPERATOR_STRING_POS,
	OPERATOR_POS,
	VALUE_POS,
};

enum {
	KIAZMA_MSC_ADD,
	KIAZMA_MSC_DEL,
	KIAZMA_MSC_END_SIGNALS
};

static guint KiazmaMetadataSetCollectorSignals [ KIAZMA_MSC_END_SIGNALS ];

/**
	@internal

	Init the class for KiazmaMetadataSetCollector

	@param	klass		Instance of the class to init
*/
static void kiazma_metadata_set_collector_class_init ( KiazmaMetadataSetCollectorClass *klass ) {
	GObjectClass *gobject_class;

	gobject_class = ( GObjectClass* ) klass;

	KiazmaMetadataSetCollectorSignals [ KIAZMA_MSC_ADD ] = g_signal_new ( g_intern_static_string ( "item_added" ),
						G_TYPE_FROM_CLASS ( gobject_class ), G_SIGNAL_RUN_FIRST, 0, NULL, NULL,
						g_cclosure_marshal_VOID__VOID, G_TYPE_NONE, 0 );

	KiazmaMetadataSetCollectorSignals [ KIAZMA_MSC_DEL ] = g_signal_new ( g_intern_static_string ( "item_removed" ),
						G_TYPE_FROM_CLASS ( gobject_class ), G_SIGNAL_RUN_FIRST, 0, NULL, NULL,
						g_cclosure_marshal_VOID__VOID, G_TYPE_NONE, 0 );
}

/**
*/
static GtkWidget* do_edit_dialog ( UINT64 init_meta, HYPPO_COMPARISON_OPERATOR init_op,
					gchar *init_value, GtkWidget **chooser ) {

	GtkWidget *dialog;

	dialog = gtk_dialog_new_with_buttons ( _( "Edit Collected Metadata Set" ), NULL, GTK_DIALOG_MODAL,
						GTK_STOCK_CANCEL, GTK_RESPONSE_REJECT, GTK_STOCK_OK, GTK_RESPONSE_OK, NULL );

	*chooser = kiazma_metadata_set_chooser_new_full ();

	if ( init_value )
		kiazma_metadata_set_chooser_set ( KIAZMA_METADATA_SET_CHOOSER ( *chooser ), init_meta, init_op, init_value );

	gtk_box_pack_start ( GTK_BOX ( GTK_DIALOG ( dialog )->vbox ), *chooser, TRUE, TRUE, 0 );
	gtk_widget_show_all ( dialog );
	return dialog;
}

/**
*/
static void add_metadata_in_collection ( GtkButton *button, KiazmaMetadataSetCollector *ttt ) {
	UINT64 meta;
	gchar *value				= NULL;
	gchar *meta_str				= NULL;
	gchar *op_str				= NULL;
	HYPPO_COMPARISON_OPERATOR op;
	GtkWidget *dialog;
	GtkWidget *sel;
	GtkTreeIter iter;

	dialog = do_edit_dialog ( 0LL, 0, NULL, &sel );
	if ( gtk_dialog_run ( GTK_DIALOG ( dialog ) ) == GTK_RESPONSE_OK ) {
		if ( kiazma_metadata_set_chooser_get ( KIAZMA_METADATA_SET_CHOOSER ( sel ), &meta, &op, &value ) ) {
			gtk_tree_store_append ( ttt->list_model, &iter, NULL );
			meta_str = hyppo_metadata_translate_in_string ( meta );
			op_str = hyppo_operator_to_string ( op );
			gtk_tree_store_set ( ttt->list_model, &iter,
						METADATA_STRING_POS, meta_str,
						METADATA_POS, meta,
						OPERATOR_STRING_POS, op_str,
						OPERATOR_POS, op,
						VALUE_POS, value,
						-1 );

			g_signal_emit ( ttt, KiazmaMetadataSetCollectorSignals [ KIAZMA_MSC_ADD ], 0 );
		}
	}

	gtk_widget_destroy ( dialog );
}

/**
*/
static void remove_metadata_from_collection ( GtkButton *button, KiazmaMetadataSetCollector *ttt ) {
	GtkTreeIter child;

	if ( gtk_tree_selection_get_selected ( ttt->list_selector, ( GtkTreeModel** ) &( ttt->list_model ), &child ) ) {
		gtk_tree_store_remove ( ttt->list_model, &child );
		g_signal_emit ( ttt, KiazmaMetadataSetCollectorSignals [ KIAZMA_MSC_DEL ], 0 );
	}
}

/**
*/
static void edit_metadata_in_collection ( GtkButton *button, KiazmaMetadataSetCollector *ttt ) {
	UINT64 meta;
	gchar *value;
	HYPPO_COMPARISON_OPERATOR op;
	GtkWidget *dialog;
	GtkWidget *sel;
	GtkTreeIter child;

	if ( gtk_tree_selection_get_selected ( ttt->list_selector, ( GtkTreeModel** ) &( ttt->list_model ), &child ) ) {
		gtk_tree_model_get ( ( GtkTreeModel* ) ttt->list_model, &child, OPERATOR_POS, &op,
					VALUE_POS, &value, METADATA_POS, &meta, -1 );
		dialog = do_edit_dialog ( meta, op, value, &sel );
		g_free ( value );

		if ( gtk_dialog_run ( GTK_DIALOG ( dialog ) ) == GTK_RESPONSE_OK ) {
			if ( kiazma_metadata_set_chooser_get ( KIAZMA_METADATA_SET_CHOOSER ( sel ), &meta, &op, &value ) ) {
				gtk_tree_store_set ( ttt->list_model, &child,
							METADATA_STRING_POS, hyppo_metadata_translate_in_string ( meta ),
							METADATA_POS, meta,
							OPERATOR_STRING_POS, hyppo_operator_to_string ( op ),
							OPERATOR_POS, op,
							VALUE_POS, value,
							-1 );

				g_free ( value );
			}
		}

		gtk_widget_destroy ( dialog );
	}
}

/**
	@internal

	@param	ttt
*/
static void kiazma_metadata_set_collector_init ( KiazmaMetadataSetCollector *ttt ) {
	GtkWidget *buttons;

	gtk_container_set_border_width ( GTK_CONTAINER ( ttt ), 0 );
	gtk_box_set_homogeneous ( GTK_BOX ( ttt ), FALSE );

	/*** lista medatati ***/

	ttt->list_model = gtk_tree_store_new ( 5, G_TYPE_STRING, G_TYPE_UINT64, G_TYPE_STRING, G_TYPE_INT, G_TYPE_STRING );
	ttt->list = gtk_tree_view_new_with_model ( GTK_TREE_MODEL ( ttt->list_model ) );

	ttt->list_selector = gtk_tree_view_get_selection ( GTK_TREE_VIEW ( ttt->list ) );
	gtk_tree_selection_set_mode ( ttt->list_selector, GTK_SELECTION_BROWSE );
	gtk_tree_view_insert_column_with_attributes ( GTK_TREE_VIEW ( ttt->list ), 0, _( "Metadata" ),
							gtk_cell_renderer_text_new (), "text", METADATA_STRING_POS, NULL );
	gtk_tree_view_insert_column_with_attributes ( GTK_TREE_VIEW ( ttt->list ), 1, _( "Operator" ),
							gtk_cell_renderer_text_new (), "text", OPERATOR_STRING_POS, NULL );
	gtk_tree_view_insert_column_with_attributes ( GTK_TREE_VIEW ( ttt->list ), 2, _( "Value" ),
							gtk_cell_renderer_text_new (), "text", VALUE_POS, NULL );
	gtk_box_pack_start ( GTK_BOX ( ttt ), ttt->list, TRUE, TRUE, 0 );

	/*** pulsanti ***/

	buttons = gtk_vbox_new ( TRUE, 2 );

	ttt->add_button = kiazma_button_stock_new ( GTK_STOCK_ADD );
	gtk_box_pack_start ( GTK_BOX ( buttons ), ttt->add_button, FALSE, FALSE, 0 );
	g_signal_connect ( G_OBJECT ( ttt->add_button ), "clicked", G_CALLBACK ( add_metadata_in_collection ), ttt );
	ttt->remove_button = kiazma_button_stock_new ( GTK_STOCK_REMOVE );
	gtk_box_pack_start ( GTK_BOX ( buttons ), ttt->remove_button, FALSE, FALSE, 0 );
	g_signal_connect ( G_OBJECT ( ttt->remove_button ), "clicked", G_CALLBACK ( remove_metadata_from_collection ), ttt );
	ttt->edit_button = kiazma_button_stock_new ( GTK_STOCK_EDIT );
	gtk_box_pack_start ( GTK_BOX ( buttons ), ttt->edit_button, FALSE, FALSE, 0 );
	g_signal_connect ( G_OBJECT ( ttt->edit_button ), "clicked", G_CALLBACK ( edit_metadata_in_collection ), ttt );

	gtk_box_pack_start ( GTK_BOX ( ttt ), buttons, FALSE, FALSE, 0 );
}

/**
	@return
*/
GType kiazma_metadata_set_collector_get_type () {
	static GType ttt_type = 0;

	if ( !ttt_type ) {
		static const GTypeInfo ttt_info = {
			sizeof ( KiazmaMetadataSetCollectorClass ),
			NULL,
			NULL,
			( GClassInitFunc ) kiazma_metadata_set_collector_class_init,
			NULL,
			NULL,
			sizeof ( KiazmaMetadataSetCollector ),
			0,
			( GInstanceInitFunc ) kiazma_metadata_set_collector_init,
		};

		ttt_type = g_type_register_static ( GTK_TYPE_HBOX, "KiazmaMetadataSetCollector", &ttt_info, 0 );
	}

	return ttt_type;
}

/**
	@return
*/
GtkWidget* kiazma_metadata_set_collector_new () {
	return GTK_WIDGET ( g_object_new ( kiazma_metadata_set_collector_get_type (), NULL ) );
}

/**
	@param	item
	@param	meta
*/
void kiazma_metadata_set_collector_add_item ( KiazmaMetadataSetCollector *item, UINT64 meta,
						HYPPO_COMPARISON_OPERATOR op, gchar *value ) {

	GtkTreeIter iter;

	gtk_tree_store_append ( item->list_model, &iter, NULL );
	gtk_tree_store_set ( item->list_model, &iter,
				METADATA_STRING_POS, hyppo_metadata_translate_in_string ( meta ),
				METADATA_POS, meta,
				OPERATOR_STRING_POS, hyppo_operator_to_string ( op ),
				OPERATOR_POS, op,
				VALUE_POS, value,
				-1 );
}

/**
	@param	item
	@param	meta
	@param	op
	@param	value

	@return
*/
gboolean kiazma_metadata_set_collector_get_first ( KiazmaMetadataSetCollector *item, UINT64 *meta,
							HYPPO_COMPARISON_OPERATOR *op, gchar **value ) {

	if ( gtk_tree_model_get_iter_first ( GTK_TREE_MODEL ( item->list_model ), &( item->current ) ) ) {
		gtk_tree_model_get ( GTK_TREE_MODEL ( item->list_model ), &( item->current ),
					OPERATOR_POS, op, VALUE_POS, value, METADATA_POS, meta, -1 );
		return TRUE;
	}
	else
		return FALSE;
}

/**
	@param	item
	@param	meta
	@param	op
	@param	value

	@return
*/
gboolean kiazma_metadata_set_collector_get_next ( KiazmaMetadataSetCollector *item, UINT64 *meta,
							HYPPO_COMPARISON_OPERATOR *op, gchar **value ) {

	if ( gtk_tree_model_iter_next ( GTK_TREE_MODEL ( item->list_model ), &( item->current ) ) ) {
		gtk_tree_model_get ( GTK_TREE_MODEL ( item->list_model ), &( item->current ),
					OPERATOR_POS, op, VALUE_POS, value, METADATA_POS, meta, -1 );
		return TRUE;
	}
	else
		return FALSE;
}

/**
*/
void kiazma_metadata_set_collector_reset ( KiazmaMetadataSetCollector *item ) {
	gtk_tree_store_clear ( item->list_model );
}
