/*  Synapse 0.2
 *  Copyright (C) 2007 Roberto -MadBob- Guido <m4db0b@users.sourceforge.net>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include "synapse.h"

/**
	Name of the file which contains the actions definitions
*/
#define DEFAULT_NAME_FOR_ACTIONS_FILE		"actions"

/**
	List of loaded actions
*/
GList		*ActionsList			= NULL;

/**
*/
void write_action_to_file ( ActionDesc *act, FILE *file ) {
	GList *inner_iter;
	ActionConstraintDesc *cons;

	fprintf ( file, "\nname: %s\n", act->name );

	for ( inner_iter = g_list_first ( act->constraints ); inner_iter; inner_iter = g_list_next ( inner_iter ) ) {
		cons = ( ActionConstraintDesc* ) inner_iter->data;
		fprintf ( file, "constraint: %llu %s\n", cons->meta, cons->value );
	}

	if ( act->command )
		fprintf ( file, "command: %s\n", act->command );

	fprintf ( file, "interm: %s\n", act->in_terminal ? "TRUE" : "FALSE" );
}

/**
*/
GList* read_actions_from_file ( GIOChannel *file ) {
	gchar *string				= NULL;
	gchar *sep				= NULL;
	ActionDesc *azione			= NULL;
	ActionConstraintDesc *desc		= NULL;
	GList *list				= NULL;
	GError *error				= NULL;
	GIOStatus reading;

	g_clear_error ( &error );
	reading = g_io_channel_read_line ( file, &string, NULL, NULL, &error );

	while ( reading != G_IO_STATUS_EOF && reading != G_IO_STATUS_ERROR ) {
		string [ strlen ( string ) - 1 ] = '\0';

		if ( strncmp ( string, "name: ", 6 ) == 0 ) {
			azione = g_new0 ( ActionDesc, 1 );
			azione->name = g_strdup ( string + 6 );
			list = g_list_prepend ( list, azione );
		}

		else if ( strncmp ( string, "command: ", 9 ) == 0 ) {
			if ( azione ) {
				if ( azione->command )
					g_warning ( "Error while reading actions file: double definition of 'command'" );
				else
					azione->command = g_strdup ( string + 9 );
			}
			else
				g_warning ( "Error while reading actions file: 'command' specified outside 'action' context" );
		}

		else if ( strncmp ( string, "interm: ", 8 ) == 0 ) {
			if ( azione )
				azione->in_terminal = ( strcmp ( string + 8, "TRUE" ) == 0 ) ? TRUE : FALSE;
			else
				g_warning ( "Error while reading actions file: 'command' specified outside 'action' context" );
		}

		else if ( strncmp ( string, "constraint: ", 12 ) == 0 ) {
			if ( azione ) {
				desc = g_new0 ( ActionConstraintDesc, 1 );
				desc->meta = strtoull ( string + 12, &sep, 10 );
				desc->value = g_strdup ( sep + 1 );
				azione->constraints = g_list_prepend ( azione->constraints, desc );
			}
			else
				g_warning ( "Error while reading actions file: 'constraint' specified outside 'action' context" );
		}

		g_free ( string );
		reading = g_io_channel_read_line ( file, &string, NULL, NULL, &error );
	}

	return list;
}

/**
	Load all saved actions from the configuration file

	@todo		Use an XML file to store informations about actions
			instead the plain text here parsed
*/
void load_actions () {
	gchar *path			= NULL;
	GIOChannel *file		= NULL;
	GError *error			= NULL;

	path = get_system_path ( DEFAULT_NAME_FOR_ACTIONS_FILE );
	file = g_io_channel_new_file ( path, "r", &error );

	if ( file == NULL ) {
		xfce_err ( _( "Unable to open the actions file: %s.\nA new one will be created..." ), error->message );
		g_clear_error ( &error );
		fclose ( fopen ( path, "w" ) );
	}
	else {
		ActionsList = read_actions_from_file ( file );
		g_io_channel_unref ( file );
	}

	g_free ( path );
}

/**
	Callback for the contextual menu's items linked to a file, execute
	the command defined for the selected action

	@param	item		Menu item selected by the user
	@param	act		Description of the action item rappresented

	@return			FALSE
*/
gboolean do_action_from_menu ( GtkMenuItem *item, ActionDesc *act ) {
	gboolean ret;
	gchar *path;
	gchar *true_path;
	gchar *command;
	gchar *tmp_command;
	GError *error			= NULL;

	if ( act->command ) {
		if ( strstr ( act->command, "%s" ) ) {
			path = hyppo_vfs_get_mounted_path ();
			true_path = g_strdup_printf ( "%s/%llu", path, act->current_file );
			command = hyppo_substitute_string ( act->command, "%s", true_path );
			g_free ( true_path );
			g_free ( path );
		}
		else
			command = g_strdup ( act->command );

		if ( act->in_terminal ) {
			if ( strlen ( Confs.action_in_terminal_command ) == 0 ) {
				xfce_err ( _( "Undefined command to invoke the terminal" ) );
				g_free ( command );
				return FALSE;
			}
			else {
				tmp_command = g_strconcat ( Confs.action_in_terminal_command, " ", command, NULL );
				g_free ( command );
				command = tmp_command;
			}
		}

		ret = g_spawn_command_line_async ( command, &error );
		g_free ( command );

		if ( ret == FALSE ) {
			xfce_err ( _( "Failed to execute action '%s': %s" ), act->name, error->message );
			g_clear_error ( &error );
		}
	}

	return FALSE;
}

/**
	Add the actions references to the contextual menu linked to the
	specified file ID. Is assumed the menu is already inited, and the
	actions names are delimited by two GtkMenuSeparator's

	@param	menu		The contextual menu where add the actions
				names
	@param	fileid		File ID to check, looking for matching
				actions to add to the menu
	@param	metas		List of metadata assigned to the file ID. If
				NULL, the list is collected directly from
				filesystem with hyppo_vfs_listxattr()
*/
void add_menu_actions ( GtkWidget *menu, UINT64 fileid, GList *metas ) {
	gboolean enter_sep			= FALSE;
	gboolean metas_here_collected		= FALSE;
	UINT64 met;
	gchar value [ 100 ];
	GList *metas_iter;
	GList *actions_iter;
	GList *cons_iter;
	GtkWidget *subitem;
	ActionDesc *act;
	ActionConstraintDesc *cons;

	if ( ActionsList == NULL || metas == NULL )
		return;

	if ( metas == NULL ) {
		metas = hyppo_vfs_listxattr ( fileid );
		metas_here_collected = TRUE;
	}

	/**
		@todo	With a better organization of mapped actions I can avoid to
			loop too much times into the lists to match the constraints
	*/
	for ( actions_iter = g_list_first ( ActionsList ); actions_iter; actions_iter = g_list_next ( actions_iter ) ) {
		act = ( ActionDesc* ) actions_iter->data;

		for ( cons_iter = g_list_first ( act->constraints ); cons_iter; cons_iter = g_list_next ( cons_iter ) ) {
			cons = ( ActionConstraintDesc* ) cons_iter->data;

			for ( metas_iter = g_list_first ( metas ); metas_iter; metas_iter = g_list_next ( metas_iter ) ) {
				met = *( ( UINT64* ) metas_iter->data );

				if ( cons->meta == met ) {
					if ( hyppo_vfs_getxattr ( fileid, met, value, 200 ) ) {
						if ( strcmp ( value, cons->value ) == 0 ) {
							if ( enter_sep == FALSE ) {
								subitem = gtk_separator_menu_item_new ();
								gtk_widget_show ( subitem );
								gtk_menu_shell_append ( GTK_MENU_SHELL ( menu ), subitem );
								enter_sep = TRUE;
							}

							subitem = gtk_menu_item_new_with_label ( _( act->name ) );
							g_signal_connect ( G_OBJECT ( subitem ), "activate",
										G_CALLBACK ( do_action_from_menu ), act );
							act->current_file = fileid;

							gtk_widget_show ( subitem );
							gtk_menu_shell_append ( GTK_MENU_SHELL ( menu ), subitem );
							break;
						}
					}
				}
			}
		}
	}

	if ( enter_sep ) {
		subitem = gtk_separator_menu_item_new ();
		gtk_widget_show ( subitem );
		gtk_menu_shell_append ( GTK_MENU_SHELL ( menu ), subitem );
	}

	if ( metas_here_collected )
		hyppo_vfs_free_xattrs ( metas );
}

/**
	Destroy from memory the list of constraints defined for the specified
	action item

	@param	action		The action item from which destroy the
				constraints list
*/
void action_destroy_constraints ( ActionDesc *action ) {
	GList *inner_iter;
	ActionConstraintDesc *desc;

	for ( inner_iter = g_list_first ( action->constraints ); inner_iter; inner_iter = g_list_next ( inner_iter ) ) {
		desc = inner_iter->data;
		g_free ( desc->value );
	}

	g_list_free ( action->constraints );
	action->constraints = NULL;
}

/**
	Destroy from memory the action item provided

	@param	garbage		An action item to be destroyed
*/
void action_destroy ( ActionDesc *garbage ) {
	if ( garbage->name )
		g_free ( garbage->name );

	if ( garbage->command )
		g_free ( garbage->command );

	if ( garbage->constraints )
		action_destroy_constraints ( garbage );

	g_free ( garbage );
}

/**
	Destroy from the memory a list of actions

	@param	list		The list to free
*/
void action_destroy_list ( GList *list ) {
	GList *iter;
	ActionDesc *azione;

	for ( iter = g_list_first ( list ); iter; iter = g_list_next ( iter ) ) {
		azione = ( ActionDesc* ) iter->data;
		action_destroy ( azione );
	}

	g_list_free ( list );
}

/**
	Close the Synapse actions subsystem
*/
void close_actions () {
	action_destroy_list ( ActionsList );
	ActionsList = NULL;
}

/**
	Duplicate an actions list in memory: the returned list can be
	manipulated without modifying the original one

	@param	original	The actions list to duplicate

	@return			A copy of the whole list, have to be freed
				with action_destroy_list() when no longer in
				use
*/
GList* action_duplicate_list ( GList *original ) {
	GList *list			= NULL;
	GList *iter;
	GList *inner;
	ActionDesc *action;
	ActionDesc *nuova;
	ActionConstraintDesc *desc;
	ActionConstraintDesc *a_nuovo;

	if ( original ) {
		for ( iter = g_list_first ( original ); iter; iter = g_list_next ( iter ) ) {
			action = ( ActionDesc* ) iter->data;
			nuova = g_new0 ( ActionDesc, 1 );

			if ( action->name )
				nuova->name = g_strdup ( action->name );
			if ( action->command )
				nuova->command = g_strdup ( action->command );

			nuova->in_terminal = action->in_terminal;

			if ( action->constraints ) {
				for ( inner = g_list_first ( action->constraints ); inner; inner = g_list_next ( inner ) ) {
					desc = ( ActionConstraintDesc* ) inner->data;
					a_nuovo = g_new0 ( ActionConstraintDesc, 1 );

					a_nuovo->meta = desc->meta;

					if ( desc->value )
						a_nuovo->value = g_strdup ( desc->value );

					nuova->constraints = g_list_prepend ( nuova->constraints, a_nuovo );
				}

				nuova->constraints = g_list_reverse ( nuova->constraints );
			}

			list = g_list_prepend ( list, nuova );
		}

		return g_list_reverse ( list );
	}
	else
		return NULL;
}

/**
	Return a copy of the currently enabled actions list

	@return			A copy of the internal list. She needs to be
				freed when no longer in use with
				action_destroy_list()
*/
GList* action_get_all () {
	return action_duplicate_list ( ActionsList );
}

/**
	Save the actions list to the permanent file
*/
void save_actions_to_file () {
	int fd;
	char name [ 50 ];
	gchar *path;
	FILE *file;
	GList *iter;
	ActionDesc *ptr;

	( void ) snprintf ( name, 50, "%s", "/tmp/synapse_actions_XXXXXX" );

	fd = mkstemp ( name );
	if ( fd == -1 ) {
		xfce_err ( _( "Error while saving actions: %s" ), strerror ( errno ) );
		return;
	}

	file = fdopen ( fd, "w" );

	for ( iter = g_list_first ( ActionsList ); iter; iter = g_list_next ( iter ) ) {
		ptr = ( ActionDesc* ) iter->data;

		if ( ptr->name == NULL ) {
			action_destroy ( ptr );
			ActionsList = g_list_delete_link ( ActionsList, iter );
			continue;
		}

		write_action_to_file ( ptr, file );
	}

	fclose ( file );
	path = get_system_path ( DEFAULT_NAME_FOR_ACTIONS_FILE );
	remove ( path );
	rename ( name, path );
	g_free ( path );
}

/**
	Replace the currently enabled actions list with the provided one. The
	submitted list is directly assigned, so have not to be freed

	@param	nuova		The new list to used
*/
void action_replace ( GList *nuova ) {
	action_destroy_list ( ActionsList );
	ActionsList = nuova;
	save_actions_to_file ();
}
