// BC_database.cpp
//
// Copyright 2012-2013 Roan Trail, Inc.
//
// This file is part of Tovero.
//
// Tovero is free software; you can redistribute it and/or modify it
// under the terms of the GNU Lesser General Public License
// version 2.1 as published by the Free Software Foundation.
//
// Tovero is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
// Lesser General Public License for more details.  You should have
// received a copy of the GNU Lesser General Public License along with
// Tovero. If not, see <http://www.gnu.org/licenses/>.

// Note: a goal of this code is to have geometric equivalence
// converting back and forth from Tovero to BRL-CAD, not necessarily
// preserve the exact database structures.

#include <tovero/graphics/brlcad/BC_database.hpp>
#include <tovero/graphics/brlcad/BC_database_reader.hpp>
#include <tovero/graphics/brlcad/BC_database_writer.hpp>
#include <tovero/math/geometry/Area.hpp>
#include <tovero/math/geometry/Distance.hpp>
#include <tovero/math/geometry/Geometric_tolerances.hpp>
#include <tovero/math/geometry/Solid.hpp>
#include <tovero/math/geometry/Unitless.hpp>
#include <tovero/math/geometry/Units_mapper.hpp>
#include <tovero/support/common.hpp>
#include <tovero/support/Reference_counting_list.hpp>
#include <tovero/support/error/Graphics_error.hpp>
#include <list>
#include <string>
#include <vector>

using std::list;
using std::string;
using std::vector;
using Roan_trail::Tovero_support::Error_param;
using Roan_trail::Tovero_support::Graphics_error;
using Roan_trail::Tovero_support::Reference_counting_list;
using Roan_trail::Tovero_math::Area;
using Roan_trail::Tovero_math::Distance;
using Roan_trail::Tovero_math::Geometric_tolerances;
using Roan_trail::Tovero_math::Solid;
using Roan_trail::Tovero_math::Unitless;
using Roan_trail::Tovero_math::Units_mapper;

using namespace Roan_trail::Tovero_graphics;

namespace
{
  const string ic_BC_default_length_units = "mm";
}

//
// Constructor/destructor/initialization
//
BC_database::BC_database(const string& title)
  : Database(title),
    m_simplify_solids_on_read(false),
    m_tolerate_database_errors(true)
{
  // change the database tolerances for BRL-CAD, based on BRL-CAD
  // (version 7.20.4) source ./include/raytrace.h" (RT_LEN_TOL and
  // RT_DOT_TOL)
  Geometric_tolerances tolerances;
  tolerances.distance = Distance(1E-8 * Distance::millimeter);
  tolerances.area = tolerances.distance * tolerances.distance;
  tolerances.cosine = Unitless(1E-3);
  tolerances.unitless = Unitless(1E-3);
  set_tolerances(tolerances);

  // set default units
  units_mapper().set_length_units(ic_BC_default_length_units);
}

//
// Read/write
//

bool BC_database::write(const string& path,
                        bool overwrite,
                        Error_param& return_error)
{
  precondition(!return_error());

  return write(path,
               overwrite,
               tolerances(),
               return_error);
}

bool BC_database::write(const string& path,
                        bool overwrite,
                        const Geometric_tolerances& tolerances,
                        Error_param& return_error)
{
  precondition(!return_error());

  bool return_value = false;

  start_error_block();

  Error_param error;
  BC_database_writer writer(title(),
                            m_tolerate_database_errors,
                            tolerances,
                            top_solids());
  const bool did_write = writer.write(path,
                                      overwrite,
                                      units_mapper(),
                                      error);
  on_error(!did_write, new Graphics_error(error_location(),
                                          Graphics_error::general,
                                          error()));
  return_value = true;
  goto exit_point;

  end_error_block();

  default_error_handler_and_cleanup(return_error,
                                    return_value,
                                    false);
  goto exit_point;

 exit_point:
  postcondition(return_error.is_valid_at_return(return_value));
  return return_value;
}

bool BC_database::write_solid(Solid* solid,
                              const string& path,
                              bool overwrite,
                              Error_param& return_error)
{
  precondition(!return_error());

  return write_solid(solid,
                     path,
                     overwrite,
                     tolerances(),
                     return_error);
}

bool BC_database::write_solid(Solid* solid,
                              const string& path,
                              bool overwrite,
                              const Geometric_tolerances& tolerances,
                              Error_param& return_error)
{
  precondition(!return_error());

  bool return_value = false;

  Reference_counting_list<Solid> solids;
  solids.push_back(*solid);

  start_error_block();

  Error_param error;
  BC_database_writer writer(title(),
                            m_tolerate_database_errors,
                            tolerances,
                            solids);
  const bool did_write = writer.write(path,
                                      overwrite,
                                      units_mapper(),
                                      error);
  on_error(!did_write, new Graphics_error(error_location(),
                                          Graphics_error::general,
                                          error()));
  return_value = true;
  goto exit_point;

  end_error_block();

  default_error_handler_and_cleanup(return_error,
                                    return_value,
                                    false);
  goto exit_point;

 exit_point:
  postcondition(return_error.is_valid_at_return(return_value));
  return return_value;
}

bool BC_database::read(const string& path,
                       const vector<string>& objects,
                       Error_param& return_error)
{
  precondition(!return_error());

  return read(path,
              objects,
              tolerances(),
              return_error);
}

bool BC_database::read(const string& path,
                       const vector<string>& objects,
                       const Geometric_tolerances& tolerances,
                       Error_param& return_error)
{
  precondition(!return_error());

  bool return_value = false;

  start_error_block();

  Reference_counting_list<Solid> solids;

  BC_database_reader reader(m_simplify_solids_on_read,
                            m_tolerate_database_errors,
                            tolerances);
  Error_param error;

  const bool did_read = reader.read(path,
                                    objects,
                                    solids,
                                    units_mapper(),
                                    error);
  on_error(!did_read, new Graphics_error(error_location(),
                                         Graphics_error::general,
                                         error()));

  set_title(reader.title());

  Reference_counting_list<Solid>& tops = top_solids();
  tops = solids;

  return_value = true;
  goto exit_point;

  end_error_block();

  default_error_handler_and_cleanup(return_error,
                                    return_value,
                                    false);
  goto exit_point;

 exit_point:
  postcondition(return_error.is_valid_at_return(return_value));
  return return_value;
}

