// BC_database_writer.cpp
//
// Copyright 2012-2013 Roan Trail, Inc.
//
// This file is part of Tovero.
//
// Tovero is free software; you can redistribute it and/or modify it
// under the terms of the GNU Lesser General Public License
// version 2.1 as published by the Free Software Foundation.
//
// Tovero is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
// Lesser General Public License for more details.  You should have
// received a copy of the GNU Lesser General Public License along with
// Tovero. If not, see <http://www.gnu.org/licenses/>.

// Note: values stored in the BRL-CAD database are in millimeter units,
//       all values supplied to mk_* functions must be in millimeter units

// BRL-CAD headers (declared first to avoid macro conflicts)
#include <wdb.h>
#include <bu.h>
//
#include <tovero/graphics/brlcad/BC_database_writer.hpp>
#include <tovero/graphics/brlcad/BC_combination_attributes.hpp>
#include <tovero/graphics/base/Combination.hpp>
#include <tovero/graphics/base/Combination_attributes.hpp>
#include <tovero/graphics/base/Database.hpp>
#include <tovero/graphics/base/Node_mapper.hpp>
#include <tovero/graphics/base/Operation.hpp>
#include <tovero/support/Logger.hpp>
#include <tovero/math/geometry/Axis_aligned_box.hpp>
#include <tovero/math/geometry/Boundary_triangle_mesh.hpp>
#include <tovero/math/geometry/Box.hpp>
#include <tovero/math/geometry/Cone.hpp>
#include <tovero/math/geometry/Distance.hpp>
#include <tovero/math/geometry/Cylinder.hpp>
#include <tovero/math/geometry/Ellipsoid.hpp>
#include <tovero/math/geometry/Ellipsoid_base.hpp>
#include <tovero/math/geometry/Elliptical_cone.hpp>
#include <tovero/math/geometry/Elliptical_cylinder.hpp>
#include <tovero/math/geometry/General_cone.hpp>
#include <tovero/math/geometry/General_cone_base.hpp>
#include <tovero/math/geometry/Geometric_tolerances.hpp>
#include <tovero/math/geometry/Half_space.hpp>
#include <tovero/math/geometry/Oblique_cone.hpp>
#include <tovero/math/geometry/Oblique_cylinder.hpp>
#include <tovero/math/geometry/Plate_triangle_mesh.hpp>
#include <tovero/math/geometry/Polyhedron.hpp>
#include <tovero/math/geometry/Polyhedron_base.hpp>
#include <tovero/math/geometry/Solid.hpp>
#include <tovero/math/geometry/Solid_combination.hpp>
#include <tovero/math/geometry/Solid_operand.hpp>
#include <tovero/math/geometry/Solid_operation.hpp>
#include <tovero/math/geometry/Solid_operator.hpp>
#include <tovero/math/geometry/Solid_member.hpp>
#include <tovero/math/geometry/Sphere.hpp>
#include <tovero/math/geometry/Torus.hpp>
#include <tovero/math/geometry/Transformation.hpp>
#include <tovero/math/geometry/Triangle_face.hpp>
#include <tovero/math/geometry/Triangle_mesh.hpp>
#include <tovero/math/geometry/Units_mapper.hpp>
#include <tovero/math/geometry/Unit_vector.hpp>
#include <tovero/math/geometry/Vector.hpp>
#include <tovero/math/geometry/Wedge.hpp>
#include <tovero/support/common.hpp>
#include <tovero/support/Reference_counting_list.hpp>
#include <tovero/support/error/Graphics_error.hpp>
#include <tovero/support/error/Posix_error.hpp>
#include <list>
#include <sstream>
#include <string>
#include <vector>
#include <cstddef>

using std::endl;
using std::ios_base;
using std::list;
using std::string;
using std::stringstream;
using std::vector;
using Roan_trail::Tovero_support::Error_param;
using Roan_trail::Tovero_support::Error;
using Roan_trail::Tovero_support::Graphics_error;
using Roan_trail::Tovero_support::Logger;
using Roan_trail::Tovero_support::Reference_counting_list;
using Roan_trail::Tovero_math::Axis_aligned_box;
using Roan_trail::Tovero_math::Box;
using Roan_trail::Tovero_math::Boundary_triangle_mesh;
using Roan_trail::Tovero_math::Cone;
using Roan_trail::Tovero_math::Cylinder;
using Roan_trail::Tovero_math::Distance;
using Roan_trail::Tovero_math::Ellipsoid;
using Roan_trail::Tovero_math::Ellipsoid_base;
using Roan_trail::Tovero_math::Elliptical_cone;
using Roan_trail::Tovero_math::Elliptical_cylinder;
using Roan_trail::Tovero_math::General_cone;
using Roan_trail::Tovero_math::General_cone_base;
using Roan_trail::Tovero_math::Half_space;
using Roan_trail::Tovero_math::Geometric_tolerances;
using Roan_trail::Tovero_math::Matrix;
using Roan_trail::Tovero_math::Oblique_cone;
using Roan_trail::Tovero_math::Oblique_cylinder;
using Roan_trail::Tovero_math::Plate_triangle_mesh;
using Roan_trail::Tovero_math::Point;
using Roan_trail::Tovero_math::Polyhedron;
using Roan_trail::Tovero_math::Polyhedron_base;
using Roan_trail::Tovero_math::Solid;
using Roan_trail::Tovero_math::Solid_combination;
using Roan_trail::Tovero_math::Solid_member;
using Roan_trail::Tovero_math::Solid_operand;
using Roan_trail::Tovero_math::Solid_operation;
using Roan_trail::Tovero_math::Solid_operator;
using Roan_trail::Tovero_math::Solid_visitor;
using Roan_trail::Tovero_math::Sphere;
using Roan_trail::Tovero_math::Torus;
using Roan_trail::Tovero_math::Transformation;
using Roan_trail::Tovero_math::Triangle_face;
using Roan_trail::Tovero_math::Triangle_mesh;
using Roan_trail::Tovero_math::Units_mapper;
using Roan_trail::Tovero_math::Unit_vector;
using Roan_trail::Tovero_math::Vector;
using Roan_trail::Tovero_math::Wedge;
using namespace Roan_trail::Tovero_graphics;

//
// Internal helpers
//

namespace
{
  const string ic_BC_default_length_units = "mm";

  int ih_convert_to_BC_op(const Solid_operator::Solid_operator_type op)
  {
    int BC_op = WMOP_UNION;
    switch (op)
    {
    case Solid_operator::intersection_op:
      BC_op = WMOP_INTERSECT;
      break;
    case Solid_operator::difference_op:
      BC_op = WMOP_SUBTRACT;
      break;
    default:
      break;
    }
    return BC_op;
  }

  inline void ih_copy_point(const Point& P,
                            const Distance& factor,
                            point_t& return_BC_P)
  {
    return_BC_P[0] = (P[0] / factor).value();
    return_BC_P[1] = (P[1] / factor).value();
    return_BC_P[2] = (P[2] / factor).value();
  }

  inline void ih_copy_unit_vector(const Unit_vector& u, vect_t& return_BC_u)
  {
    return_BC_u[0] = u[0].value();
    return_BC_u[1] = u[1].value();
    return_BC_u[2] = u[2].value();
  }

  inline void ih_copy_vector(const Vector& v,
                             const Distance& factor,
                             vect_t& return_BC_v)
  {
    return_BC_v[0] = (v[0] / factor).value();
    return_BC_v[1] = (v[1] / factor).value();
    return_BC_v[2] = (v[2] / factor).value();
  }

  void ih_copy_transform(const Transformation& transform,
                         const Distance& conversion_factor,
                         mat_t& return_BC_matrix)
  {
    Matrix m = transform.matrix();
    const Distance& translation_units = transform.translation_units();

    // convert units for translation only
    const double translate_factor = (translation_units / conversion_factor).value();
    m[3] *= translate_factor;
    m[7] *= translate_factor;
    m[11] *= translate_factor;

    for (size_t i = 0; i < 16; ++i)
    {
      return_BC_matrix[i] = m[i];
    }
  }
}

//
// Constructor/destructor/initialization
//

BC_database_writer::BC_database_writer(const string& title,
                                       bool tolerate_database_errors,
                                       const Geometric_tolerances& tolerances,
                                       const Reference_counting_list<Solid>& solids)
  : m_title(title),
    m_tolerate_database_errors(tolerate_database_errors),
    m_tolerances(tolerances),
    m_solids(&solids),
    m_file(),
    m_node_mapper(0),
    m_IO_error(0),
    m_logger(Roan_trail::Tovero_support::Logger::default_logger()),
    m_distance_conversion_factor(new Distance(1.0 * Distance::meter))
{
  postcondition(mf_invariant(false));
}

BC_database_writer::File_struct::File_struct()
  : path(""),
    overwrite(false),
    database(0)
{
}

BC_database_writer::~BC_database_writer()
{
  delete m_distance_conversion_factor;
  delete m_IO_error;
  delete m_node_mapper;

  if (m_file.database)
  {
    wdb_close(m_file.database);
  }

  postcondition(mf_invariant(false));
}


//
// Write
//

bool BC_database_writer::write(const string& path,
                               bool overwrite,
                               const Units_mapper& units_mapper,
                               Error_param& return_error)
{
  precondition(!return_error()
               && mf_invariant());

  bool return_value = false;

  start_error_block();

  on_error(path == "", new Graphics_error(error_location(),
                                          Graphics_error::write,
                                          "empty path for BRL-CAD database when opening for write"));

  mf_setup_for_write(path,
                     overwrite,
                     units_mapper);

  Error_param error;
  const bool opened = mf_open_for_write(units_mapper,
                                        error);
  on_error(!opened, new Graphics_error(error_location(),
                                       Graphics_error::general,
                                       error()));

  // traverse the tree for each solid
  for (Reference_counting_list<Solid>::const_iterator i = m_solids->begin();
       i != m_solids->end();
       ++i)
  {
    const Solid& solid = **i;
    solid.accept(*this); // TODO: check return code?
    on_error(m_IO_error, new Graphics_error(error_location(),
                                            Graphics_error::general,
                                            m_IO_error->clone()));
  }

  return_value = true;
  goto exit_point;

  end_error_block();

  default_error_handler(return_error);

 error_cleanup:
  if (m_IO_error)
  {
    delete m_IO_error;
    m_IO_error = 0;
  }
  if (!return_error.need_error())
  {
    delete handler_error;
  }
  return_value = false;
  goto exit_point;

 exit_point:
  postcondition(return_error.is_valid_at_return(return_value)
                && mf_invariant());
  return return_value;
}

//
// Protected member functions
//

bool BC_database_writer::mf_invariant(bool check_base_class) const
{
  static_cast<void>(check_base_class); // avoid unused warning

  bool return_value = false;

  if (!m_solids
      || !m_distance_conversion_factor)
  {
    goto exit_point;
  }

  return_value = true;
  goto exit_point;

 exit_point:
  return return_value;
}

//
// Private member functions
//

bool BC_database_writer::mf_open_for_write(const Units_mapper& units_mapper,
                                           Error_param& return_error)
{
  precondition(!return_error()
               && !m_file.database);

  struct rt_wdb* stream = 0;

  bool return_value = false;

  start_error_block();

  // Note: this isn't really a satisfying way to check for a file's existence
  //       because it allows a race condition, but...
  if (!m_file.overwrite)
  {
    struct stat stat_buffer;
    const int stat_return = stat(m_file.path.c_str(), &stat_buffer);
    on_error(stat_return, new Graphics_error(error_location(),
                                             Graphics_error::write,
                                             "file exists and overwrite not allowed",
                                             m_file.path));
  }
  stream = wdb_fopen(m_file.path.c_str());
  on_error(!stream, new Graphics_error(error_location(),
                                       Graphics_error::write,
                                       "could not open BRL-CAD database",
                                       m_file.path));

  string length_units = units_mapper.length_units();
  if (!(bu_units_conversion(length_units.c_str()) > 0.0))
  {
    const string diagnostic = string("Invalid length units: ") + length_units
      + string(" for BRL-CAD database");
    on_error(!m_tolerate_database_errors, new Graphics_error(error_location(),
                                                             Graphics_error::write,
                                                             diagnostic));
    m_logger << Logger::warning << diagnostic << endl;
    length_units = ic_BC_default_length_units;
  }
  assert((bu_units_conversion(length_units.c_str()) > 0.0) && "invalid default BRL-CAD length units");

  const int mk_id_units_return = mk_id_units(stream,
                                             m_title.c_str(),
                                             length_units.c_str());
  if (mk_id_units_return)
  {
    const string diagnostic = string("Could not set BRL-CAD database title to: ") + m_title
      + string(" and length units to: ") + length_units;
    on_error(!m_tolerate_database_errors, new Graphics_error(error_location(),
                                                             Graphics_error::write,
                                                             diagnostic));
    m_logger << Logger::warning << diagnostic << endl;
  }

  m_file.database = stream;

  return_value = true;
  goto exit_point;

  end_error_block();

  default_error_handler(return_error);

 error_cleanup:
  if (!return_error.need_error())
  {
    delete handler_error;
  }
  if (stream)
  {
    wdb_close(stream);
  }
  goto exit_point;

 exit_point:
  postcondition(((return_value && m_file.database)
                 || (!return_value && !m_file.database))
                && return_error.is_valid_at_return(return_value));
  return return_value;
}

//
//   setup/cleanup
//

void BC_database_writer::mf_setup_for_write(const string& path,
                                            bool overwrite,
                                            const Units_mapper& units_mapper)
{
  m_file.path = path;
  m_file.overwrite = overwrite;
  if (m_file.database)
  {
    wdb_close(m_file.database);
    m_file.database = 0;
  }

  delete m_node_mapper;
  m_node_mapper = new Node_mapper<const Solid*>;

  delete m_IO_error;
  m_IO_error = 0;

  // BRL-CAD stores all length values internally as millimeters, set up a
  // conversion factor
  Units_mapper::Length_map_type::const_iterator i = units_mapper.length_map.find("mm");
  assert((i != units_mapper.length_map.end()) && "error, mm unit conversion factor not found");
  *m_distance_conversion_factor = i->second;
}

void BC_database_writer::mf_cleanup_after_write()
{
  if (m_file.database)
  {
    wdb_close(m_file.database);
    m_file.database = 0;
  }
  m_file.path = "";

  delete m_node_mapper;
  m_node_mapper = 0;
}

//
//   visitor visit member functions
//

//
//     composites
//

Solid_visitor::Visit_result BC_database_writer::visit_exit(const Solid_operation& operation)
{
  Solid_visitor::Visit_result return_value = Solid_visitor::abort;

  start_error_block();

  string operation_name;
  m_node_mapper->add_node(&operation,
                          "Solid_operation",
                          operation_name);

  struct wmember head;
  // initialize the list of members
  BU_LIST_INIT(&head.l);

  // add members to the list
  Reference_counting_list<Solid_member> member_list;
  Solid_member* left_member = new Solid_member(operation.left_operand(), Solid_operator::union_op);
  Solid_member* right_member = new Solid_member(operation.right_operand(), operation.op());
  member_list.push_back(*left_member);
  member_list.push_back(*right_member);

  for (Reference_counting_list<Solid_member>::const_iterator i = member_list.begin();
       i != member_list.end();
       ++i)
  {
    const Solid_member& current_member = **i;
    // get operand for this combination
    const Solid_operand& operand = current_member.operand();

    string solid_name;
    const bool found = m_node_mapper->find_node_name(&operand.solid(), solid_name);
    static_cast<void>(found); // Avoid unused warning
    assert(found && "error, member solid not named");

    // get operator for this combination
    Solid_operator::Solid_operator_type op = current_member.op();
    const int BC_op = ih_convert_to_BC_op(op);

    // determine transform for this node
    const Transformation* transformation = operand.transformation();

    fastf_t* BC_transform_matrix = 0;
    if (transformation)
    {
      // since the transformation is not null, copy it to the matrix array
      mat_t matrix;
      ih_copy_transform(*transformation,
                        *m_distance_conversion_factor,
                        matrix);
      BC_transform_matrix = matrix;
    }

    // add the member to the list
    const struct wmember* member = mk_addmember(solid_name.c_str(),
                                                &head.l,
                                                BC_transform_matrix,
                                                BC_op);
    if (member == WMEMBER_NULL)
    {
      const string diagnostic = string("Could not add member: ") + solid_name
        + string(" to operation (combination): ") + operation_name;
      on_error(!m_tolerate_database_errors, new Graphics_error(error_location(),
                                                               Graphics_error::write,
                                                               diagnostic));
      m_logger << Logger::warning << diagnostic << endl;
      // tolerate it, but don't add the overall combination
      return_value = Solid_visitor::success;
      goto exit_point;
    }
  }

  bool is_region = false;
  string shader_name;
  string shader_args;
  unsigned char *BC_color = 0;
  unsigned char BC_color_values[3];
  int material = 0;
  bool children_inherit = false;

  // set the attributes, if any
  Combination_attributes* attributes = 0;
  // Note: the run time type check and reinterpret cast method used here
  // is not ideal, but it allows us to extend the visitor pattern
  // outside the math library in a relatively simple way.
  //
  // TODO: is this an exception to C++ code standards?
  if (operation.solid_class() == "Operation")
  {
    const Operation* attributed_operation = reinterpret_cast<const Operation*>(&operation);
    attributes = attributed_operation->attributes();
    if (attributes)
    {
      BC_combination_attributes BC_attributes(*attributes);
      is_region = BC_attributes.is_region();
      shader_name = BC_attributes.shader_name();
      shader_args = BC_attributes.shader_args();
      BC_attributes.copy_color(BC_color_values);
      BC_color = BC_color_values;
      material = BC_attributes.material();
      children_inherit = BC_attributes.children_inherit();
    }
  }

  const char *BC_shader_name = ("" == shader_name) ? 0 : shader_name.c_str();
  const char *BC_shader_args = ((!BC_shader_name) || ("" == shader_args)) ? 0 : shader_args.c_str();
  // TODO: consider mk_comb (includes additional parameters)
  const int combination_result = mk_lrcomb(m_file.database,        // database
                                           operation_name.c_str(), // node name
                                           &head,                  // pointer to list of elements
                                           is_region,              // region (part) flag
                                           BC_shader_name,         // shader name
                                           BC_shader_args,         // shader args
                                           BC_color,               // color
                                           0,                      // region ID
                                           0,                      // air code
                                           material,               // material code
                                           0,                      // (los) line of sight thickness
                                           children_inherit);      // children inherit?
  if (combination_result)
  {
    const string diagnostic = string("Could not make combination for operation: ") + operation_name;
    on_error(!m_tolerate_database_errors, new Graphics_error(error_location(),
                                                             Graphics_error::write,
                                                             diagnostic));
    m_logger << Logger::warning << diagnostic << endl;
  }

  return_value = Solid_visitor::success;
  goto exit_point;

  end_error_block();

 error_handler:
  delete m_IO_error;
  m_IO_error = handler_error;
  goto error_cleanup;

 error_cleanup:
  return_value = Solid_visitor::abort;
  goto exit_point;

 exit_point:
  return return_value;
}

Solid_visitor::Visit_result BC_database_writer::visit_exit(const Solid_combination& combination)
{
  Solid_visitor::Visit_result return_value = Solid_visitor::abort;

  start_error_block();

  string combination_name;
  m_node_mapper->add_node(&combination,
                          "Solid_combination",
                          combination_name);

  struct wmember head;
  // initialize the list of members
  BU_LIST_INIT(&head.l);

  // add members to the list
  const Reference_counting_list<Solid_member> member_list = combination.members();
  for (Reference_counting_list<Solid_member>::const_iterator i = member_list.begin();
       i != member_list.end();
       ++i)
  {
    const Solid_member& current_member = **i;

    // get operand for this combination
    const Solid_operand& operand = current_member.operand();

    string solid_name;
    const bool found = m_node_mapper->find_node_name(&operand.solid(), solid_name);
    static_cast<void>(found); // Avoid unused warning
    assert(found && "error, member solid not named");

    // get operator for this combination
    Solid_operator::Solid_operator_type op;
    if (member_list.begin() == i)
    {
      // force to union regardless of value if the first member of the combination
      op = Solid_operator::union_op;
    }
    else
    {
      op = current_member.op();
    }
    const int BC_op = ih_convert_to_BC_op(op);

    // determine transform for this node
    const Transformation* transformation = operand.transformation();

    fastf_t* BC_transform_matrix = 0;
    if (transformation)
    {
      // since the transformation is not null, copy it to the matrix array
      mat_t matrix;
      ih_copy_transform(*transformation,
                        *m_distance_conversion_factor,
                        matrix);
      BC_transform_matrix = matrix;
    }

    // add the member to the list
    const struct wmember* member = mk_addmember(solid_name.c_str(),
                                                &head.l,
                                                BC_transform_matrix,
                                                BC_op);
    if (member == WMEMBER_NULL)
    {
      const string diagnostic = string("Could not add member: ") + solid_name
        + string(" to combination: ") + combination_name;
      on_error(!m_tolerate_database_errors, new Graphics_error(error_location(),
                                                               Graphics_error::write,
                                                               diagnostic));
      m_logger << Logger::warning << diagnostic << endl;
      // tolerate it, but don't add the overall combination
      return_value = Solid_visitor::success;
      goto exit_point;
    }
  }

  bool is_region = false;
  string shader_name;
  string shader_args;
  unsigned char* BC_color = 0;
  unsigned char BC_color_values[3];
  int material = 0;
  bool children_inherit = false;

  // set the attributes, if present
  Combination_attributes* attributes = 0;
  // Note: the run time type check and reinterpret cast method used here
  // is not ideal, but it allows us to extend the visitor pattern
  // outside the math library in a relatively simple way.
  //
  // TODO: is this an exception to C++ code standards?
  if (combination.solid_class() == "Combination")
  {
    const Combination* attributed_combination = reinterpret_cast<const Combination*>(&combination);
    attributes = attributed_combination->attributes();
    if (attributes)
    {
      BC_combination_attributes BC_attributes(*attributes);
      is_region = BC_attributes.is_region();
      shader_name = BC_attributes.shader_name();
      shader_args = BC_attributes.shader_args();
      BC_attributes.copy_color(BC_color_values);
      BC_color = BC_color_values;
      material = BC_attributes.material();
      children_inherit = BC_attributes.children_inherit();
    }
  }

  const char *BC_shader_name = ("" == shader_name) ? 0 : shader_name.c_str();
  const char *BC_shader_args = ((!BC_shader_name) || ("" == shader_args)) ? 0 : shader_args.c_str();
  const int combination_result = mk_lrcomb(m_file.database,          // database
                                           combination_name.c_str(), // node name
                                           &head,                  // pointer to list of elements
                                           is_region,              // region (part) flag
                                           BC_shader_name,         // shader name
                                           BC_shader_args,         // shader args
                                           BC_color,               // color
                                           0,                      // region ID
                                           0,                      // air code
                                           material,               // material code
                                           0,                      // (los) line of sight thickness
                                           children_inherit);      // children inherit?
  if (combination_result)
  {
    const string diagnostic = "Could not make combination: " + combination_name;
    on_error(!m_tolerate_database_errors, new Graphics_error(error_location(),
                                                             Graphics_error::write,
                                                             diagnostic));
    m_logger << Logger::warning << diagnostic << endl;
  }

  return_value = Solid_visitor::success;
  goto exit_point;

  end_error_block();

 error_handler:
  delete m_IO_error;
  m_IO_error = handler_error;
  goto error_cleanup;

 error_cleanup:
  return_value = Solid_visitor::abort;
  goto exit_point;

 exit_point:
  return return_value;
}

//
//     primitive solids
//

Solid_visitor::Visit_result BC_database_writer::visit(const Axis_aligned_box& box)
{
  return mf_visit_base(box);
}

Solid_visitor::Visit_result BC_database_writer::visit(const Boundary_triangle_mesh& mesh)
{
  return mf_visit_base(mesh,
                       false, // is not solid (boundary)
                       false, // does not have thicknesses
                       vector<Distance>(),
                       vector<bool>());
}

Solid_visitor::Visit_result BC_database_writer::visit(const Box& box)
{
  return mf_visit_base(box);
}

Solid_visitor::Visit_result BC_database_writer::visit(const Cone& cone)
{
  return mf_visit_base(cone);
}

Solid_visitor::Visit_result BC_database_writer::visit(const Cylinder& cylinder)
{
  return mf_visit_base(cylinder);
}

Solid_visitor::Visit_result BC_database_writer::visit(const Ellipsoid& ellipsoid)
{
  return mf_visit_base(ellipsoid);
}

Solid_visitor::Visit_result BC_database_writer::visit(const Elliptical_cone& cone)
{
  return mf_visit_base(cone);
}

Solid_visitor::Visit_result BC_database_writer::visit(const Elliptical_cylinder& cylinder)
{
  return mf_visit_base(cylinder);
}

Solid_visitor::Visit_result BC_database_writer::visit(const General_cone& cone)
{
  return mf_visit_base(cone);
}

Solid_visitor::Visit_result BC_database_writer::visit(const Half_space& space)
{
  Solid_visitor::Visit_result return_value = Solid_visitor::abort;

  start_error_block();

  string solid_name;
  Error_param error;
  const bool success = mf_add_and_validate_solid(space,
                                                 solid_name,
                                                 error);
  on_error(!success, new Graphics_error(error_location(),
                                        Graphics_error::general,
                                        error()));

  // set normal
  const Unit_vector& normal = space.normal();
  vect_t BC_normal;
  ih_copy_unit_vector(normal, BC_normal);
  // set distance
  const Distance& distance = space.distance();
  const fastf_t BC_distance = (distance / *m_distance_conversion_factor).value();

  // make the BC solid
  const int return_code = mk_half(m_file.database,
                                  solid_name.c_str(),
                                  BC_normal,
                                  BC_distance);
  if (return_code)
  {
    const string diagnostic = string("Could not create ") + space.solid_class() + string(": ") + solid_name;
    on_error(!m_tolerate_database_errors, new Graphics_error(error_location(),
                                                             Graphics_error::write,
                                                             diagnostic));
    m_logger << Logger::warning << diagnostic << endl;
  }

  return_value = Solid_visitor::success;
  goto exit_point;

  end_error_block();

 error_handler:
  delete m_IO_error;
  m_IO_error = handler_error;
  goto error_cleanup;

 error_cleanup:
  return_value = Solid_visitor::abort;
  goto exit_point;

 exit_point:
  return return_value;
}

Solid_visitor::Visit_result BC_database_writer::visit(const Oblique_cone& cone)
{
  return mf_visit_base(cone);
}

Solid_visitor::Visit_result BC_database_writer::visit(const Oblique_cylinder& cylinder)
{
  return mf_visit_base(cylinder);
}

Solid_visitor::Visit_result BC_database_writer::visit(const Plate_triangle_mesh& mesh)
{
  return mf_visit_base(mesh,
                       true, // is solid
                       true, // has thicknesses
                       mesh.thicknesses(),
                       mesh.append_thicknesses());
}

Solid_visitor::Visit_result BC_database_writer::visit(const Polyhedron& polyhedron)
{
  return mf_visit_base(polyhedron);
}

Solid_visitor::Visit_result BC_database_writer::visit(const Sphere& sphere)
{
  return mf_visit_base(sphere);
}

Solid_visitor::Visit_result BC_database_writer::visit(const Torus& torus)
{
  Solid_visitor::Visit_result return_value = Solid_visitor::abort;

  start_error_block();

  string solid_name;
  Error_param error;
  const bool success = mf_add_and_validate_solid(torus,
                                                 solid_name,
                                                 error);
  on_error(!success, new Graphics_error(error_location(),
                                        Graphics_error::general,
                                        error()));

  // set center
  const Point& center = torus.center();
  point_t BC_center;
  ih_copy_point(center,
                *m_distance_conversion_factor,
                BC_center);
  // set normal
  const Unit_vector& normal = torus.normal();
  vect_t BC_normal;
  ih_copy_unit_vector(normal, BC_normal);
  // set radii
  const fastf_t BC_major_radius = (torus.major_radius() / *m_distance_conversion_factor).value();
  const fastf_t BC_minor_radius = (torus.minor_radius() / *m_distance_conversion_factor).value();
  // make the BC solid
  const int return_code = mk_tor(m_file.database,
                                 solid_name.c_str(),
                                 BC_center,
                                 BC_normal,
                                 BC_major_radius,
                                 BC_minor_radius);
  if (return_code)
  {
    const string diagnostic = string("Could not create ") + torus.solid_class() + string(": ") + solid_name;
    on_error(!m_tolerate_database_errors, new Graphics_error(error_location(),
                                                             Graphics_error::write,
                                                             diagnostic));
    m_logger << Logger::warning << diagnostic << endl;
  }

  return_value = Solid_visitor::success;
  goto exit_point;

  end_error_block();

 error_handler:
  delete m_IO_error;
  m_IO_error = handler_error;
  goto error_cleanup;

 error_cleanup:
  return_value = Solid_visitor::abort;
  goto exit_point;

 exit_point:
  return return_value;
}

Solid_visitor::Visit_result BC_database_writer::visit(const Triangle_mesh& mesh)
{
  return mf_visit_base(mesh,
                       true, // is solid
                       false, // does not have thicknesses
                       vector<Distance>(),
                       vector<bool>());
}

Solid_visitor::Visit_result BC_database_writer::visit(const Wedge& wedge)
{
  return mf_visit_base(wedge);
}

//
//   base
//

Solid_visitor::Visit_result BC_database_writer::mf_visit_base(const Ellipsoid_base& ellipsoid)
{
  Solid_visitor::Visit_result return_value = Solid_visitor::abort;

  start_error_block();

  string solid_name;
  Error_param error;
  const bool success = mf_add_and_validate_solid(ellipsoid,
                                                 solid_name,
                                                 error);
  on_error(!success, new Graphics_error(error_location(),
                                        Graphics_error::general,
                                        error()));

  Ellipsoid& general_ellipsoid = ellipsoid.generalize();
  // set center
  const Point& center = general_ellipsoid.center();
  point_t BC_center;
  ih_copy_point(center,
                *m_distance_conversion_factor,
                BC_center);
  // set axes
  const Vector& axis_a = general_ellipsoid.axis_a();
  vect_t BC_axis_a;
  ih_copy_vector(axis_a,
                 *m_distance_conversion_factor,
                 BC_axis_a);
  const Vector& axis_b = general_ellipsoid.axis_b();
  vect_t BC_axis_b;
  ih_copy_vector(axis_b,
                 *m_distance_conversion_factor,
                 BC_axis_b);
  const Vector& axis_c = general_ellipsoid.axis_c();
  vect_t BC_axis_c;
  ih_copy_vector(axis_c,
                 *m_distance_conversion_factor,
                 BC_axis_c);
  // make the BC solid
  const int return_code = mk_ell(m_file.database,
                                 solid_name.c_str(),
                                 BC_center,
                                 BC_axis_a,
                                 BC_axis_b,
                                 BC_axis_c);
  if (return_code)
  {
    const string diagnostic = "Could not create " + ellipsoid.solid_class() + string(": ") + solid_name;
    on_error(!m_tolerate_database_errors, new Graphics_error(error_location(),
                                                             Graphics_error::write,
                                                             diagnostic));
    m_logger << Logger::warning << diagnostic << endl;
  }

  return_value = Solid_visitor::success;
  goto exit_point;

  end_error_block();

 error_handler:
  delete m_IO_error;
  m_IO_error = handler_error;
  goto error_cleanup;

 error_cleanup:
  return_value = Solid_visitor::abort;
  goto exit_point;

 exit_point:
  return return_value;
}

Solid_visitor::Visit_result BC_database_writer::mf_visit_base(const General_cone_base& cone)
{
  Solid_visitor::Visit_result return_value = Solid_visitor::abort;

  start_error_block();

  string solid_name;
  Error_param error;
  const bool success = mf_add_and_validate_solid(cone,
                                                 solid_name,
                                                 error);
  on_error(!success, new Graphics_error(error_location(),
                                        Graphics_error::general,
                                        error()));

  General_cone& general_cone = cone.generalize();
  // set base point
  const Point& base = general_cone.base();
  point_t BC_vertex;
  ih_copy_point(base,
                *m_distance_conversion_factor,
                BC_vertex);
  // set height vector
  const Vector& height = general_cone.height();
  vect_t BC_height;
  ih_copy_vector(height,
                 *m_distance_conversion_factor,
                 BC_height);
  // set radii
  const Vector& base_a = general_cone.base_a();
  vect_t BC_base_a;
  ih_copy_vector(base_a,
                 *m_distance_conversion_factor,
                 BC_base_a);
  const Vector& base_b = general_cone.base_b();
  vect_t BC_base_b;
  ih_copy_vector(base_b,
                 *m_distance_conversion_factor,
                 BC_base_b);
  const Vector& top_c = general_cone.top_c();
  vect_t BC_top_c;
  ih_copy_vector(top_c,
                 *m_distance_conversion_factor,
                 BC_top_c);
  const Vector& top_d = general_cone.top_d();
  vect_t BC_top_d;
  ih_copy_vector(top_d,
                 *m_distance_conversion_factor,
                 BC_top_d);
  // make the BC solid
  const int return_code = mk_tgc(m_file.database,
                                 solid_name.c_str(),
                                 BC_vertex,
                                 BC_height,
                                 BC_base_a,
                                 BC_base_b,
                                 BC_top_c,
                                 BC_top_d);
  if (return_code)
  {
    const string diagnostic = "Could not create " + cone.solid_class() + string(": ") + solid_name;
    on_error(!m_tolerate_database_errors, new Graphics_error(error_location(),
                                                             Graphics_error::write,
                                                             diagnostic));
    m_logger << Logger::warning << diagnostic << endl;
  }

  return_value = Solid_visitor::success;
  goto exit_point;

  end_error_block();

 error_handler:
  delete m_IO_error;
  m_IO_error = handler_error;
  goto error_cleanup;

 error_cleanup:
  return_value = Solid_visitor::abort;
  goto exit_point;

 exit_point:
  return return_value;
}

Solid_visitor::Visit_result BC_database_writer::mf_visit_base(const Polyhedron_base& polyhedron)
{
  Solid_visitor::Visit_result return_value = Solid_visitor::abort;

  start_error_block();

  string solid_name;
  Error_param error;
  const bool success = mf_add_and_validate_solid(polyhedron,
                                                 solid_name,
                                                 error);
  on_error(!success, new Graphics_error(error_location(),
                                        Graphics_error::general,
                                        error()));

  Polyhedron& general_polyhedron = polyhedron.generalize();
  const vector<Point>& vertices = general_polyhedron.vertices();
  const size_t vertex_count = vertices.size();
  point_t BC_vertices[8];
  for (size_t i = 0; ((i < vertex_count) && (i < 8)); ++i)
  {
    ih_copy_point(vertices[i],
                  *m_distance_conversion_factor,
                  BC_vertices[i]);
  }
  // fill in any remaining (duplicate) vertices
  switch (vertex_count)
  {
  case 4:
    ih_copy_point(vertices[3],
                  *m_distance_conversion_factor,
                  BC_vertices[7]);
    ih_copy_point(vertices[3],
                  *m_distance_conversion_factor,
                  BC_vertices[6]);
    ih_copy_point(vertices[3],
                  *m_distance_conversion_factor,
                  BC_vertices[5]);
    ih_copy_point(vertices[3],
                  *m_distance_conversion_factor,
                  BC_vertices[4]);
    ih_copy_point(vertices[0],
                  *m_distance_conversion_factor,
                  BC_vertices[3]);
    break;
  case 5:
    ih_copy_point(vertices[4],
                  *m_distance_conversion_factor,
                  BC_vertices[7]);
    ih_copy_point(vertices[4],
                  *m_distance_conversion_factor,
                  BC_vertices[6]);
    ih_copy_point(vertices[4],
                  *m_distance_conversion_factor,
                  BC_vertices[5]);
    break;
  case 6:
    ih_copy_point(vertices[5],
                  *m_distance_conversion_factor,
                  BC_vertices[7]);
    ih_copy_point(vertices[5],
                  *m_distance_conversion_factor,
                  BC_vertices[6]);
    ih_copy_point(vertices[4],
                  *m_distance_conversion_factor,
                  BC_vertices[5]);
    break;
  case 7:
    ih_copy_point(vertices[4],
                  *m_distance_conversion_factor,
                  BC_vertices[7]);
    break;
  case 8:
    break;
  default:
    // should be caught in validation
    assert(false && "error, invalid number of vertices for polyhedron");
  }
  // make the BC solid
  const int return_code = mk_arb8(m_file.database,
                                  solid_name.c_str(),
                                  &BC_vertices[0][X]);
  if (return_code)
  {
    const string diagnostic = "Could not create " + polyhedron.solid_class() + string(": ") + solid_name;
    on_error(!m_tolerate_database_errors, new Graphics_error(error_location(),
                                                             Graphics_error::write,
                                                             diagnostic));
    m_logger << Logger::warning << diagnostic << endl;
  }

  return_value = Solid_visitor::success;
  goto exit_point;

  end_error_block();

 error_handler:
  delete m_IO_error;
  m_IO_error = handler_error;
  goto error_cleanup;

 error_cleanup:
  return_value = Solid_visitor::abort;
  goto exit_point;

 exit_point:
  return return_value;
}

//
//
//

bool BC_database_writer::mf_add_and_validate_solid(const Solid& solid,
                                                  string& return_solid_name,
                                                  Error_param& return_error)
{
  precondition(!return_error());

  bool return_value = false;

  start_error_block();

  // make solid if necessary (only make each node once)
  if (m_node_mapper->add_node(&solid,
                              solid.solid_class(),
                              return_solid_name))
  {
    // validate
    Error_param error;
    if (!solid.is_valid(m_tolerances, error))
    {
      on_error(!m_tolerate_database_errors, new Graphics_error(error_location(),
                                                               Graphics_error::write,
                                                               error()));
      string diagnostic = error()->error_dictionary()[Error::diagnostic_error_key];
      m_logger << Logger::warning << diagnostic << endl;
    }
  }

  return_value = true;
  goto exit_point;

  end_error_block();

  default_error_handler_and_cleanup(return_error, return_value, false);

 exit_point:
  postcondition(return_error.is_valid_at_return(return_value));
  return return_value;
}

Solid_visitor::Visit_result BC_database_writer::mf_visit_base(const Triangle_mesh& mesh,
                                                              bool is_solid,
                                                              bool has_thicknesses,
                                                              const vector<Distance>& thicknesses,
                                                              const vector<bool>& append_thicknesses)
{
  fastf_t* BC_vertices = 0;
  int* BC_faces = 0;
  fastf_t* BC_thicknesses = 0;
  bu_bitv* BC_face_mode = 0;
  fastf_t* BC_normals = 0;
  int* BC_face_normals = 0;

  Solid_visitor::Visit_result return_value = Solid_visitor::abort;

  start_error_block();

  string solid_name;
  Error_param error;
  const bool success = mf_add_and_validate_solid(mesh,
                                                 solid_name,
                                                 error);
  on_error(!success, new Graphics_error(error_location(),
                                        Graphics_error::general,
                                        error()));
  // mode
  unsigned char BC_mode;
  if (is_solid)
  {
    if (has_thicknesses)
    {
      BC_mode = RT_BOT_PLATE;
    }
    else
    {
      BC_mode = RT_BOT_SOLID;
    }
  }
  else
  {
    BC_mode = RT_BOT_SURFACE;
  }
  // triangle orientation
  unsigned char BC_orientation = RT_BOT_UNORIENTED;
  switch (mesh.triangle_direction())
  {
  case Triangle_face::direction_none:
    break;
  case Triangle_face::direction_clockwise:
    BC_orientation = RT_BOT_CW;
    break;
  case Triangle_face::direction_counterclockwise:
    BC_orientation = RT_BOT_CCW;
    break;
  default:
    assert(false && "invalid triangle orientation");
    break;
  }
  // vertices
  const vector<Point>& vertices = mesh.vertices();
  const size_t BC_vertex_count = vertices.size();
  if (BC_vertex_count)
  {
    BC_vertices = new fastf_t[BC_vertex_count * 3];
    for (size_t i = 0; i < BC_vertex_count; ++i)
    {
      const Point& vertex = vertices[i];
      size_t index = i * 3;
      BC_vertices[index] = (vertex[0] / *m_distance_conversion_factor).value();
      BC_vertices[index + 1] = (vertex[1] / *m_distance_conversion_factor).value();
      BC_vertices[index + 2] = (vertex[2] / *m_distance_conversion_factor).value();
    }
  }
  // faces
  const vector<Triangle_face>& faces = mesh.faces();
  const size_t BC_face_count = faces.size();
  if (BC_face_count)
  {
    BC_faces = new int[BC_face_count * 3];
    for (size_t i = 0; i < BC_face_count; ++i)
    {
      const Triangle_face& face = faces[i];
      size_t index = i * 3;
      BC_faces[index] = face.index_1();
      BC_faces[index + 1] = face.index_2();
      BC_faces[index + 2] = face.index_3();
    }
  }
  // thicknesses
  const size_t BC_thickness_count = thicknesses.size();
  if (BC_thickness_count)
  {
    BC_thicknesses = new fastf_t[BC_thickness_count];
    for (size_t i = 0; i < BC_thickness_count; ++i)
    {
      BC_thicknesses[i] = (thicknesses[i] / *m_distance_conversion_factor).value();
    }
  }
  const size_t BC_append_thicknesses_count = append_thicknesses.size();
  if (BC_append_thicknesses_count)
  {
    BC_face_mode = bu_bitv_new(BC_append_thicknesses_count);
    for (size_t i = 0; i < BC_append_thicknesses_count; ++i)
    {
      if (append_thicknesses[i])
      {
        BU_BITSET(BC_face_mode, i);
      }
      else
      {
        BU_BITCLR(BC_face_mode, i);
      }
    }
  }
  // normals
  const vector<Vector>& normals = mesh.normals();
  const size_t BC_normal_count = normals.size();
  unsigned char BC_bot_flags = 0;
  // flags
  if (mesh.has_surface_normals())
  {
    BC_bot_flags |= RT_BOT_HAS_SURFACE_NORMALS;
  }
  if (mesh.use_surface_normals())
  {
    BC_bot_flags |= RT_BOT_USE_NORMALS;
  }
  size_t BC_face_normal_count = 0;
  if (mesh.has_surface_normals() && BC_normal_count)
  {
    BC_normals = new fastf_t[BC_normal_count * 3];
    for (size_t i = 0; i < BC_normal_count; ++i)
    {
      const Vector& normal = normals[i];
      const size_t index = i * 3;
      BC_normals[index] = (normal[0] / *m_distance_conversion_factor).value();
      BC_normals[index + 1] = (normal[1] / *m_distance_conversion_factor).value();
      BC_normals[index + 2] = (normal[0] / *m_distance_conversion_factor).value();
    }
    const vector<Triangle_face>& face_normals = mesh.face_normals();
    BC_face_normal_count = face_normals.size();
    if (BC_face_normal_count)
    {
      BC_face_normals = new int[BC_face_count * 3];
      for (size_t i = 0; i < BC_face_count; ++i)
      {
        const Triangle_face& face_normal = face_normals[i];
        const size_t index = i * 3;
        BC_face_normals[index] = face_normal.index_1();
        BC_face_normals[index + 1] = face_normal.index_2();
        BC_face_normals[index + 2] = face_normal.index_3();
      }
    }
  }

  // create the solid
  const int return_code = mk_bot_w_normals(m_file.database,
                                           solid_name.c_str(),
                                           BC_mode,
                                           BC_orientation,
                                           BC_bot_flags,
                                           BC_vertex_count,
                                           BC_face_count,
                                           BC_vertices,
                                           BC_faces,
                                           BC_thicknesses,
                                           BC_face_mode,
                                           BC_normal_count,
                                           BC_normals,
                                           BC_face_normals);
  if (return_code)
  {
    const string diagnostic = "Could not create " + mesh.solid_class() + string(": ") + solid_name;
    on_error(!m_tolerate_database_errors, new Graphics_error(error_location(),
                                                             Graphics_error::write,
                                                             diagnostic));
    m_logger << Logger::warning << diagnostic << endl;
  }

  return_value = Solid_visitor::success;
  goto exit_point;

  end_error_block();

 error_handler:
  delete m_IO_error;
  m_IO_error = handler_error;
  goto error_cleanup;

 error_cleanup:
  return_value = Solid_visitor::abort;
  goto exit_point;

 exit_point:
  // cleanup
  delete [] BC_face_normals;
  delete [] BC_normals;
  if (BC_face_mode)
  {
    bu_bitv_free(BC_face_mode);
  }
  delete [] BC_thicknesses;
  delete [] BC_faces;
  delete [] BC_vertices;

  return return_value;
}
