// Kinetophone.cpp
//
// Copyright 2012-2013 Roan Trail, Inc.
//
// This file is part of Tovero.
//
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions are
// met:
//
// (1) Redistributions of source code must retain the above copyright
// notice, this list of conditions and the following disclaimer.
//
// (2) Redistributions in binary form must reproduce the above copyright
// notice, this list of conditions and the following disclaimer in
// the documentation and/or other materials provided with the
// distribution.
//
// (3) The name of the author may not be used to
// endorse or promote products derived from this software without
// specific prior written permission.
//
// THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
// IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
// WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
// DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT,
// INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
// (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
// SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
// HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
// STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING
// IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
// POSSIBILITY OF SUCH DAMAGE.

#include <tovero/graphics/brlcad/BC_database.hpp>
#include <tovero/graphics/base/Combination.hpp>
#include <tovero/graphics/base/Database.hpp>
#include <tovero/graphics/base/Color.hpp>
#include <tovero/graphics/base/Combination_attributes.hpp>
#include <tovero/graphics/base/Phong_shader.hpp>
#include <tovero/graphics/base/Toon_shader.hpp>
#include <tovero/support/error/Error.hpp>
#include <tovero/support/error/Graphics_error.hpp>
#include <tovero/math/geometry/Cone.hpp>
#include <tovero/math/geometry/Cylinder.hpp>
#include <tovero/math/geometry/Distance.hpp>
#include <tovero/math/geometry/Ellipsoid.hpp>
#include <tovero/math/geometry/General_cone.hpp>
#include <tovero/math/geometry/Polyhedron.hpp>
#include <tovero/math/geometry/Solid.hpp>
#include <tovero/math/geometry/Solid_combination.hpp>
#include <tovero/math/geometry/Solid_member.hpp>
#include <tovero/math/geometry/Solid_operation.hpp>
#include <tovero/math/geometry/Solid_operator.hpp>
#include <tovero/math/geometry/Sphere.hpp>
#include <tovero/math/geometry/Torus.hpp>
#include <tovero/math/geometry/Unit_vector.hpp>
#include <tovero/math/geometry/Unitless.hpp>
#include <tovero/math/geometry/Vector.hpp>
#include <tovero/support/Reference_counting_list.hpp>
#include <tovero/support/common.hpp>
#include <iostream>
#include <string>
#include <vector>
#include <cstddef>

using std::cout;
using std::endl;
using std::string;
using std::vector;
using Roan_trail::Tovero_support::Error;
using Roan_trail::Tovero_support::Error_param;
using Roan_trail::Tovero_support::Reference_counting_list;
using Roan_trail::Tovero_math::Cone;
using Roan_trail::Tovero_math::Cylinder;
using Roan_trail::Tovero_math::Distance;
using Roan_trail::Tovero_math::Ellipsoid;
using Roan_trail::Tovero_math::General_cone;
using Roan_trail::Tovero_math::Polyhedron;
using Roan_trail::Tovero_math::Point;
using Roan_trail::Tovero_math::Solid;
using Roan_trail::Tovero_math::Solid_combination;
using Roan_trail::Tovero_math::Solid_member;
using Roan_trail::Tovero_math::Solid_operation;
using Roan_trail::Tovero_math::Solid_operator;
using Roan_trail::Tovero_math::Sphere;
using Roan_trail::Tovero_math::Torus;
using Roan_trail::Tovero_math::Unit_vector;
using Roan_trail::Tovero_math::Unitless;
using Roan_trail::Tovero_math::Vector;

using namespace Roan_trail::Tovero_graphics;

void make_logo();

int main(int argc, char* argv[])
{
  make_logo();

  return 0;
}

void make_logo()
{
  Cylinder* base = new Cylinder(Point(),
                                Vector(0.0, 0.0, 2.0, Distance::meter),
                                Distance(1.3, Distance::meter),
                                "base.s");

  Torus* sound_tor1 = new Torus(Point(0.0, 0.0, 5.0, Distance::meter),
                                Unit_vector(Unitless(1.0), Unitless(0.0), Unitless(0.0)),
                                Distance(3.0, Distance::meter),
                                Distance(0.2, Distance::meter),
                                "sound_tor1.s");

  Torus* sound_tor2 = new Torus(Point(0.0, 0.0, 5.0, Distance::meter),
                                Unit_vector(Unitless(1.0), Unitless(0.0), Unitless(0.0)),
                                Distance(2.0, Distance::meter),
                                Distance(0.2, Distance::meter),
                                "sound_tor2.s");

  Torus* sound_tor3 = new Torus(Point(0.0, 0.0, 5.0, Distance::meter),
                                Unit_vector(Unitless(1.0), Unitless(0.0), Unitless(0.0)),
                                Distance(1.0, Distance::meter),
                                Distance(0.2, Distance::meter),
                                "sound_tor3.s");

  vector<Point> vertices;
  vertices.push_back(Point(-5.0, -5.0, 0.0, Distance::meter));
  vertices.push_back(Point(-5.0, 5.0, 0.0, Distance::meter));
  vertices.push_back(Point(5.0, 5.0, 0.0, Distance::meter));
  vertices.push_back(Point(5.0, -5.0, 0.0, Distance::meter));
  vertices.push_back(Point(-5.0, -5.0, 5.0, Distance::meter));
  vertices.push_back(Point(-5.0, 5.0, 5.0, Distance::meter));
  vertices.push_back(Point(5.0, 5.0, 5.0, Distance::meter));
  vertices.push_back(Point(5.0, -5.0, 5.0, Distance::meter));
  Polyhedron* sound_box = new Polyhedron(vertices, "sound_box.s");

  Solid_combination* sound_lines = new Solid_combination;
  sound_lines->set_name("sound_lines.r");
  sound_lines->add_member(*(new Solid_member(*sound_tor1)));
  sound_lines->add_member(*(new Solid_member(*sound_box, Solid_operator::difference_op)));
  sound_lines->add_member(*(new Solid_member(*sound_tor2)));
  sound_lines->add_member(*(new Solid_member(*sound_box, Solid_operator::difference_op)));
  sound_lines->add_member(*(new Solid_member(*sound_tor3)));
  sound_lines->add_member(*(new Solid_member(*sound_box, Solid_operator::difference_op)));

  Cone* outer_cone = new Cone(Point(0.0, 0.0, 2.0, Distance::meter),
                              Vector(0.0, 0.0, 2.0, Distance::meter),
                              Distance(1.3, Distance::meter),
                              Distance(2.3, Distance::meter),
                              "outer_cone.s");

  Cone* inner_cone = new Cone(Point(0.0, 0.0, 2.1, Distance::meter),
                              Vector(0.0, 0.0, 2.0, Distance::meter),
                              Distance(1.2382, Distance::meter),
                              Distance(2.2382, Distance::meter),
                              "inner_cone.s");

  Phong_shader* shader = &Phong_shader::plastic();
  Color* color = new Color(172, 118, 78, 0);
  Combination_attributes* attributes = new Combination_attributes;
  attributes->set_shader(shader);
  attributes->set_color(color);

  Combination* speaker = new Combination(attributes);
  speaker->set_name("speaker.r");
  speaker->add_member(*(new Solid_member(*base)));
  speaker->add_member(*(new Solid_member(*outer_cone)));
  speaker->add_member(*(new Solid_member(*inner_cone, Solid_operator::difference_op)));
  speaker->add_member(*(new Solid_member(*sound_lines)));

  BC_database BC_DB("Kinetophone Logo");
  Reference_counting_list<Solid>& out_solids = BC_DB.top_solids();
  out_solids.push_back(*speaker);

  Error_param error;
  const bool wrote_DB = BC_DB.write("kinetophone.g",
                                    true,
                                    error);
  if (!wrote_DB)
  {
    cout << endl << "Error writing kinetophone.g" << endl;
    cout << "Diagnostic: " << error()->error_dictionary()[Error::diagnostic_error_key] << endl;
    cout << *error() << endl;
  }
}
