// Angle.hpp
//
// Copyright 2012-2013 Roan Trail, Inc.
//
// This file is part of Tovero.
//
// Tovero is free software; you can redistribute it and/or modify it
// under the terms of the GNU Lesser General Public License
// version 2.1 as published by the Free Software Foundation.
//
// Tovero is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
// Lesser General Public License for more details.  You should have
// received a copy of the GNU Lesser General Public License along with
// Tovero. If not, see <http://www.gnu.org/licenses/>.

#ifndef TOVERO_MATH_ANGLE_HPP_
#define TOVERO_MATH_ANGLE_HPP_

#include <tovero/math/geometry/Quantity.hpp>
#include <tovero/math/geometry/Unitless.hpp>

namespace Roan_trail
{
  namespace Tovero_math
  {
    class Angle : public Quantity
    {
    public:
      // constructors
      Angle() : Quantity(0.0) {}
      //   meant for creating an angle from a value and a unit
      Angle(double value, const Angle& unit) : Quantity(value * unit.value()) {}
      static inline Angle from_value(double value);
      //  Note: use compiler generated copy constructor and destructor
      // operators
      inline Angle& operator=(const Angle& q);
      Angle operator+(const Angle& q) const { return Angle::from_value(value() + q.value()); }
      Angle operator-(const Angle& q) const { return Angle::from_value(value() - q.value()); }
      Angle operator-() const { return Angle::from_value(-value()); }
      Angle operator*(const Unitless& factor) const { return Angle::from_value(factor.value() * value()); }
      Angle operator*(double factor) const { return Angle::from_value(factor * value()); }
      Unitless operator/(const Angle &divisor) const
      { return Unitless::from_value(value() / divisor.value()); }
      Angle operator/(const Unitless &divisor) const
      { return Angle::from_value(value() / divisor.value()); }
      Angle operator/(double divisor) const { return Angle::from_value(value() / divisor); }
      inline Angle& operator+=(const Angle& addend);
      inline Angle& operator-=(const Angle& subtrahend);
      inline Angle& operator*=(const Unitless& factor);
      inline Angle& operator*=(double factor);
      inline Angle& operator/=(const Unitless& factor);
      inline Angle& operator/=(double factor);
      bool operator<(const Angle& q) const { return (value() < q.value()); }
      bool operator>(const Angle& q) const { return (value() > q.value()); }
      // class constants
      static const Angle radian;
      static const Angle degree;
    };

    // free functions

    inline Angle operator*(const Unitless& u, const Angle& d)
    { return Angle::from_value(d.value() * u.value()); }

    inline Angle operator*(double u, const Angle& d) { return Angle::from_value(d.value() * u); }

    //
    // Inline definitions
    //

    //
    // Constructors
    //

    inline Angle Angle::from_value(double value)
    {
      Angle d;
      d.set_value(value);

      return d;
    }

    //
    //   Operators
    //

    inline Angle& Angle::operator=(const Angle& d)
    {
      if (this != &d)
      {
        set_value(d.value());
      }

      return *this;
    }

    inline Angle& Angle::operator+=(const Angle& addend)
    {
      set_value(value() + addend.value());

      return *this;
    }

    inline Angle& Angle::operator-=(const Angle& subtrahend)
    {
      set_value(value() - subtrahend.value());

      return *this;
    }

    inline Angle& Angle::operator*=(const Unitless& factor)
    {
      set_value(value() * factor.value());

      return *this;
    }

    inline Angle& Angle::operator*=(double factor)
    {
      set_value(value() * factor);

      return *this;
    }

    inline Angle& Angle::operator/=(const Unitless& factor)
    {
      set_value(value() / factor.value());

      return *this;
    }

    inline Angle& Angle::operator/=(double factor)
    {
      set_value(value() / factor);

      return *this;
    }
  }
}

#endif // TOVERO_MATH_ANGLE_HPP_
