// Ellipsoid.cpp
//
// Copyright 2012-2013 Roan Trail, Inc.
//
// This file is part of Tovero.
//
// Tovero is free software; you can redistribute it and/or modify it
// under the terms of the GNU Lesser General Public License
// version 2.1 as published by the Free Software Foundation.
//
// Tovero is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
// Lesser General Public License for more details.  You should have
// received a copy of the GNU Lesser General Public License along with
// Tovero. If not, see <http://www.gnu.org/licenses/>.

#include <tovero/math/geometry/Ellipsoid.hpp>
#include <tovero/math/geometry/Area.hpp>
#include <tovero/math/geometry/Distance.hpp>
#include <tovero/math/geometry/Geometric_tolerances.hpp>
#include <tovero/math/geometry/Point.hpp>
#include <tovero/math/geometry/Vector.hpp>
#include <tovero/math/geometry/Unit_vector.hpp>
#include <tovero/math/geometry/Unitless.hpp>
#include <tovero/math/geometry/Sphere.hpp>
#include <tovero/support/common.hpp>
#include <tovero/support/error/Error.hpp>
#include <tovero/support/error/Math_error.hpp>
#include <sstream>
#include <string>
#include <cmath>

using std::string;
using std::stringstream;
using Roan_trail::Tovero_support::Error_param;
using Roan_trail::Tovero_support::Math_error;
using namespace Roan_trail::Tovero_math;

//
// Constructors
//

Ellipsoid::Ellipsoid(const string& name)
  : Ellipsoid_base(name),
    m_center(Point::O),
    m_axis_a(Unit_vector::x * Distance::meter),
    m_axis_b(Unit_vector::y * Distance::meter),
    m_axis_c(Unit_vector::z * Distance::meter)
{
}

Ellipsoid::Ellipsoid(const Point& center,
                     const Vector& axis_a,
                     const Vector& axis_b,
                     const Vector& axis_c,
                     const string& name)
  : Ellipsoid_base(name),
    m_center(center),
    m_axis_a(axis_a),
    m_axis_b(axis_b),
    m_axis_c(axis_c)
{
}

Ellipsoid::Ellipsoid(const Point& focus_A,
                     const Point& focus_B,
                     const Distance& axis_length,
                     const string& name)
  : Ellipsoid_base(name),
    m_center(),
    m_axis_a(),
    m_axis_b(),
    m_axis_c()
{
  Vector axis = focus_B - focus_A;
  axis *= Unitless(0.5);
  m_center = focus_A + axis;  // the base center is the midpoint of focus A and focus B
  Unit_vector unit_a;
  axis.normalize(unit_a);
  m_axis_a = unit_a * axis_length;
  Distance radius = sqrt(m_axis_a.length_squared() + axis.length_squared());
  Unit_vector unit_b = unit_a.perpendicular();
  m_axis_b = unit_b * radius;
  m_axis_c = unit_a.cross(unit_b) * radius;
}

Ellipsoid::Ellipsoid(const Point& center,
                     const Vector& axis_a,
                     const Distance& radius,
                     const std::string& name)
  : Ellipsoid_base(name),
    m_center(center),
    m_axis_a(axis_a),
    m_axis_b(),
    m_axis_c()
{
  Unit_vector unit_a;
  m_axis_a.normalize(unit_a);
  Unit_vector unit_b = unit_a.perpendicular();
  m_axis_b = unit_b * radius;
  m_axis_c = unit_a.cross(unit_b) * radius;
}

//
// Predicates
//

bool Ellipsoid::is_valid(const Geometric_tolerances& tolerances, Error_param& return_error) const
{
  precondition(!return_error());

  bool return_value = false;

  start_error_block();

  const Distance length_a = m_axis_a.length();
  const Distance length_b = m_axis_b.length();
  const Distance length_c = m_axis_c.length();
  // check for non-zero length axis
  if (tolerances.is_distance_near_zero(length_a)
      || tolerances.is_distance_near_zero(length_b)
      || tolerances.is_distance_near_zero(length_c))
  {
    stringstream diagnostic_stream;
    diagnostic_stream << "Invalid " << solid_class();
    const string& solid_name = name();
    diagnostic_stream << ((solid_name == "") ? "" : (string(" (") + solid_name + string(")")));
    diagnostic_stream << ": one or more axes is 0 length";
    on_error(true, new Math_error(error_location(),
                                  Math_error::validation,
                                  diagnostic_stream.str()));
  }

  Unit_vector axis_a_unit;
  m_axis_a.normalize(axis_a_unit);
  Unit_vector axis_b_unit;
  m_axis_b.normalize(axis_b_unit);
  Unit_vector axis_c_unit;
  m_axis_c.normalize(axis_c_unit);

  // check for mutually perpendicular axes
  if (!tolerances.is_cosine_near_zero(axis_a_unit.dot(axis_b_unit))
      || !tolerances.is_cosine_near_zero(axis_a_unit.dot(axis_c_unit))
      || !tolerances.is_cosine_near_zero(axis_b_unit.dot(axis_c_unit)))
  {
    stringstream diagnostic_stream;
    diagnostic_stream << "Invalid " << solid_class();
    const string& solid_name = name();
    diagnostic_stream << ((solid_name == "") ? "" : (string(" (") + solid_name + string(")")));
    diagnostic_stream << ": the axes are not mutually perpendicular";
    on_error(true, new Math_error(error_location(),
                                  Math_error::validation,
                                  diagnostic_stream.str()));
  }

  return_value = true;
  goto exit_point;

  end_error_block();

  default_error_handler_and_cleanup(return_error,
                                    return_value,
                                    false);

 exit_point:
  postcondition(return_error.is_valid_at_return(return_value));
  return return_value;
}

//
// Other
//

Solid& Ellipsoid::specialize(const Geometric_tolerances& tolerances) const
{
  Solid* return_value = const_cast<Solid*>(static_cast<const Solid*>((this)));

  const Distance length_a = m_axis_a.length();
  const Distance length_b = m_axis_b.length();
  const Distance length_c = m_axis_c.length();
  const Distance error_length_ab = length_a - length_b;
  const Distance error_length_ac = length_a - length_c;
  // check to see if the axes are equal length
  if (tolerances.is_distance_near_zero(error_length_ab)
      && tolerances.is_distance_near_zero(error_length_ac))
  {
    // sphere
    return_value = new Sphere(m_center,
                              length_a,
                              name());
  }

  return *return_value;
}
