// Ellipsoid.hpp
//
// Copyright 2012-2013 Roan Trail, Inc.
//
// This file is part of Tovero.
//
// Tovero is free software; you can redistribute it and/or modify it
// under the terms of the GNU Lesser General Public License
// version 2.1 as published by the Free Software Foundation.
//
// Tovero is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
// Lesser General Public License for more details.  You should have
// received a copy of the GNU Lesser General Public License along with
// Tovero. If not, see <http://www.gnu.org/licenses/>.

// A sphere-like solid, with 3 separate, mutally perpendicular "radial" axes.

#ifndef TOVERO_MATH_ELLIPSOID_HPP_
#define TOVERO_MATH_ELLIPSOID_HPP_

#include <tovero/math/geometry/Ellipsoid_base.hpp>
#include <tovero/math/geometry/Distance.hpp>
#include <tovero/math/geometry/Geometric_tolerances.hpp>
#include <tovero/math/geometry/Point.hpp>
#include <tovero/math/geometry/Vector.hpp>
#include <string>

namespace Roan_trail
{
  namespace Tovero_math
  {
    class Ellipsoid : public Ellipsoid_base
    {
    public:
      // Ellipsoid defaults to a sphere of radius 1.0 centered at the origin
      // constructors/copy
      //   by axis
      Ellipsoid(const std::string& name = "");
      Ellipsoid(const Point& center,
                const Vector& axis_a,
                const Vector& axis_b,
                const Vector& axis_c,
                const std::string& name = "");
      //   by foci
      Ellipsoid(const Point& focus_A,
                const Point& focus_B,
                const Distance& axis_length,
                const std::string& name = "");
      //   by radius of revolution
      Ellipsoid(const Point& center,
                const Vector& axis_a,
                const Distance& radius,
                const std::string& name = "");
      virtual Solid* clone_solid() const { return new Ellipsoid(*this); }
      //   Note: let compiler generate copy constructor and operator=
      // accessors/mutators
      const Point& center() const { return m_center; }
      void set_center(const Point& center) { m_center = center; }
      const Vector& axis_a() const { return m_axis_a; }
      void set_axis_a(const Vector& axis_a) { m_axis_a = axis_a; }
      const Vector& axis_b() const { return m_axis_b; }
      void set_axis_b(const Vector& axis_b) { m_axis_b = axis_b; }
      const Vector& axis_c() const { return m_axis_c; }
      void set_axis_c(const Vector& axis_c) { m_axis_c = axis_c; }
      // predicates
      virtual bool is_valid(const Geometric_tolerances& tolerances,
                            Roan_trail::Tovero_support::Error_param& return_error) const;
      // visitor
      virtual Solid_visitor::Visit_result accept(Solid_visitor& visitor) const { return visitor.visit(*this); }
      // other
      virtual std::string solid_class() const { return "Ellipsoid"; }
      virtual Ellipsoid& generalize() const { return *const_cast<Ellipsoid*>(this); }
      virtual Solid& specialize(const Geometric_tolerances& tolerances) const;
    protected:
      // destructor
      virtual ~Ellipsoid() {}
    private:
      // solid parameters
      Point m_center;
      Vector m_axis_a;
      Vector m_axis_b;
      Vector m_axis_c;
    };
  }
}

#endif // TOVERO_MATH_ELLIPSOID_HPP_
