// Matrix.cpp
//
// Copyright 2012-2013 Roan Trail, Inc.
//
// This file is part of Tovero.
//
// Tovero is free software; you can redistribute it and/or modify it
// under the terms of the GNU Lesser General Public License
// version 2.1 as published by the Free Software Foundation.
//
// Tovero is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
// Lesser General Public License for more details.  You should have
// received a copy of the GNU Lesser General Public License along with
// Tovero. If not, see <http://www.gnu.org/licenses/>.

#include <tovero/math/geometry/Matrix.hpp>
#include <tovero/math/geometry/Angle.hpp>
#include <tovero/math/geometry/Unit_vector.hpp>
#include <tovero/math/geometry/Vector.hpp>

using namespace Roan_trail::Tovero_math;

namespace
{
  inline double ih_multiply(const double lhs[16],
                            const double rhs[16],
                            size_t i,
                            size_t j)
  {
    return (lhs[4 * i    ] * rhs[j     ]
          + lhs[4 * i + 1] * rhs[j + 4 ]
          + lhs[4 * i + 2] * rhs[j + 8 ]
          + lhs[4 * i + 3] * rhs[j + 12]);
  }
}

//
// Constructors
//

Matrix::Matrix()
{
  for (size_t i = 0; i < 16; ++i)
  {
    m_matrix[i] = 0.0;
  }
}

Matrix::Matrix(// row 0
               double v_0_0,
               double v_0_1,
               double v_0_2,
               double v_0_3,
               // row 1
               double v_1_0,
               double v_1_1,
               double v_1_2,
               double v_1_3,
               // row 2
               double v_2_0,
               double v_2_1,
               double v_2_2,
               double v_2_3,
               // row 3
               double v_3_0,
               double v_3_1,
               double v_3_2,
               double v_3_3)
{
  // row 0
  m_matrix[ 0] = v_0_0;
  m_matrix[ 1] = v_0_1;
  m_matrix[ 2] = v_0_2;
  m_matrix[ 3] = v_0_3;
  // row 1
  m_matrix[ 4] = v_1_0;
  m_matrix[ 5] = v_1_1;
  m_matrix[ 6] = v_1_2;
  m_matrix[ 7] = v_1_3;
  // row 2
  m_matrix[ 8] = v_2_0;
  m_matrix[ 9] = v_2_1;
  m_matrix[10] = v_2_2;
  m_matrix[11] = v_2_3;
  // row 3
  m_matrix[12] = v_3_0;
  m_matrix[13] = v_3_1;
  m_matrix[14] = v_3_2;
  m_matrix[15] = v_3_3;
}

Matrix::Matrix(const Matrix& m)
{
  for (size_t i = 0; i < 16; ++i)
  {
    m_matrix[i] = m.m_matrix[i];
  }
}

//
// Operators
//

Matrix& Matrix::operator=(const Matrix& m)
{
  if (this != &m)
  {
    for (size_t i = 0; i < 16; ++i)
    {
      m_matrix[i] = m.m_matrix[i];
    }
  }
  return *this;
}

//
// Functions
//

void Matrix::transpose()
{
  exchange(m_matrix[ 4], m_matrix[ 1]);
  exchange(m_matrix[ 8], m_matrix[ 2]);
  exchange(m_matrix[12], m_matrix[ 3]);
  exchange(m_matrix[ 9], m_matrix[ 6]);
  exchange(m_matrix[13], m_matrix[ 7]);
  exchange(m_matrix[14], m_matrix[11]);
  exchange(m_matrix[ 3], m_matrix[12]);
}

void Matrix::set_rotation(const Unit_vector& axis, const Angle& a)
{
  const double sin_theta = sin((a.value() * Angle::radian).value());
  const double cos_theta = cos((a.value() * Angle::radian).value());
  const double one_minus_cos_theta = 1.0 - cos_theta;
  const double x = axis[0].value();
  const double y = axis[1].value();
  const double z = axis[2].value();
  const double x_x_one_minus_cos_theta = x * x * one_minus_cos_theta;
  const double y_y_one_minus_cos_theta = y * y * one_minus_cos_theta;
  const double z_z_one_minus_cos_theta = z * z * one_minus_cos_theta;
  const double x_y_one_minus_cos_theta = x * y * one_minus_cos_theta;
  const double x_z_one_minus_cos_theta = x * z * one_minus_cos_theta;
  const double y_z_one_minus_cos_theta = y * z * one_minus_cos_theta;
  const double x_sin_theta = x * sin_theta;
  const double y_sin_theta = y * sin_theta;
  const double z_sin_theta = z * sin_theta;

  // row 0
  m_matrix[ 0] = x_x_one_minus_cos_theta + cos_theta;   // 0, 0
  m_matrix[ 1] = x_y_one_minus_cos_theta - z_sin_theta; // 0, 1
  m_matrix[ 2] = x_z_one_minus_cos_theta + y_sin_theta; // 0, 2
  m_matrix[ 3] = 0.0;                                   // 0, 3
  // row 1
  m_matrix[ 4] = x_y_one_minus_cos_theta + z_sin_theta; // 1, 0
  m_matrix[ 5] = y_y_one_minus_cos_theta + cos_theta;   // 1, 1
  m_matrix[ 6] = y_z_one_minus_cos_theta - x_sin_theta; // 1, 2
  m_matrix[ 7] = 0.0;                                   // 1, 3
  // row 2
  m_matrix[ 8] = x_z_one_minus_cos_theta - y_sin_theta; // 2, 0
  m_matrix[ 9] = y_z_one_minus_cos_theta + x_sin_theta; // 2, 1
  m_matrix[10] = z_z_one_minus_cos_theta + cos_theta;   // 2, 2
  m_matrix[11] = 0.0;                                   // 2, 3
  // row 3
  m_matrix[12] = 0.0;                                   // 3, 0
  m_matrix[13] = 0.0;                                   // 3, 1
  m_matrix[14] = 0.0;                                   // 3, 2
  m_matrix[15] = 1.0;                                   // 3, 3
}

void Matrix::set_translation(double tx,
                             double ty,
                             double tz)
{
  // row 0
  m_matrix[ 0] = 1.0; // 0, 0
  m_matrix[ 1] = 0.0; // 0, 1
  m_matrix[ 2] = 0.0; // 0, 2
  m_matrix[ 3] = tx;  // 0, 3
  // row 1
  m_matrix[ 4] = 0.0; // 1, 0
  m_matrix[ 5] = 1.0; // 1, 1
  m_matrix[ 6] = 0.0; // 1, 2
  m_matrix[ 7] = ty;  // 1, 3
  // row 2
  m_matrix[ 8] = 0.0; // 2, 0
  m_matrix[ 9] = 0.0; // 2, 1
  m_matrix[10] = 1.0; // 2, 2
  m_matrix[11] = tz;  // 2, 3
  // row 3
  m_matrix[12] = 0.0; // 3, 0
  m_matrix[13] = 0.0; // 3, 1
  m_matrix[14] = 0.0; // 3, 2
  m_matrix[15] = 1.0; // 3, 3
}

void Matrix::set_scale(double sx,
                       double sy,
                       double sz)
{
  // row 0
  m_matrix[ 0] = sx;  // 0, 0
  m_matrix[ 1] = 0.0; // 0, 1
  m_matrix[ 2] = 0.0; // 0, 2
  m_matrix[ 3] = 0.0; // 0, 3
  // row 1
  m_matrix[ 4] = 0.0; // 1, 0
  m_matrix[ 5] = sy;  // 1, 1
  m_matrix[ 6] = 0.0; // 1, 2
  m_matrix[ 7] = 0.0; // 1, 3
  // row 2
  m_matrix[ 8] = 0.0; // 2, 0
  m_matrix[ 9] = 0.0; // 2, 1
  m_matrix[10] = sz;  // 2, 2
  m_matrix[11] = 0.0; // 2, 3
  // row 3
  m_matrix[12] = 0.0; // 3, 0
  m_matrix[13] = 0.0; // 3, 1
  m_matrix[14] = 0.0; // 3, 2
  m_matrix[15] = 1.0; // 3, 3
}

void Matrix::set_uniform_scale(double s)
{
  // row 0
  m_matrix[ 0] = s;   // 0, 0
  m_matrix[ 1] = 0.0; // 0, 1
  m_matrix[ 2] = 0.0; // 0, 2
  m_matrix[ 3] = 0.0; // 0, 3
  // row 1
  m_matrix[ 4] = 0.0; // 1, 0
  m_matrix[ 5] = s;   // 1, 1
  m_matrix[ 6] = 0.0; // 1, 2
  m_matrix[ 7] = 0.0; // 1, 3
  // row 2
  m_matrix[ 8] = 0.0; // 2, 0
  m_matrix[ 9] = 0.0; // 2, 1
  m_matrix[10] = s;   // 2, 2
  m_matrix[11] = 0.0; // 2, 3
  // row 3
  m_matrix[12] = 0.0; // 3, 0
  m_matrix[13] = 0.0; // 3, 1
  m_matrix[14] = 0.0; // 3, 2
  m_matrix[15] = 1.0; // 3, 3
}

Matrix& Matrix::multiply_right(const Matrix& m)
{
  double result[16];

  // row 0
  result[ 0] = ih_multiply(m_matrix, m.m_matrix, 0, 0);
  result[ 1] = ih_multiply(m_matrix, m.m_matrix, 0, 1);
  result[ 2] = ih_multiply(m_matrix, m.m_matrix, 0, 2);
  result[ 3] = ih_multiply(m_matrix, m.m_matrix, 0, 3);
  // row 1
  result[ 4] = ih_multiply(m_matrix, m.m_matrix, 1, 0);
  result[ 5] = ih_multiply(m_matrix, m.m_matrix, 1, 1);
  result[ 6] = ih_multiply(m_matrix, m.m_matrix, 1, 2);
  result[ 7] = ih_multiply(m_matrix, m.m_matrix, 1, 3);
  // row 2
  result[ 8] = ih_multiply(m_matrix, m.m_matrix, 2, 0);
  result[ 9] = ih_multiply(m_matrix, m.m_matrix, 2, 1);
  result[10] = ih_multiply(m_matrix, m.m_matrix, 2, 2);
  result[11] = ih_multiply(m_matrix, m.m_matrix, 2, 3);
  // row 3
  result[12] = ih_multiply(m_matrix, m.m_matrix, 3, 0);
  result[13] = ih_multiply(m_matrix, m.m_matrix, 3, 1);
  result[14] = ih_multiply(m_matrix, m.m_matrix, 3, 2);
  result[15] = ih_multiply(m_matrix, m.m_matrix, 3, 3);

  for (size_t i = 0; i < 16; ++i)
  {
    m_matrix[i] = result[i];
  }

  return *this;
}

Matrix& Matrix::multiply_left(const Matrix& m)
{
  double result[16];

  // row 0
  result[ 0] = ih_multiply(m.m_matrix, m_matrix, 0, 0);
  result[ 1] = ih_multiply(m.m_matrix, m_matrix, 0, 1);
  result[ 2] = ih_multiply(m.m_matrix, m_matrix, 0, 2);
  result[ 3] = ih_multiply(m.m_matrix, m_matrix, 0, 3);
  // row 1
  result[ 4] = ih_multiply(m.m_matrix, m_matrix, 1, 0);
  result[ 5] = ih_multiply(m.m_matrix, m_matrix, 1, 1);
  result[ 6] = ih_multiply(m.m_matrix, m_matrix, 1, 2);
  result[ 7] = ih_multiply(m.m_matrix, m_matrix, 1, 3);
  // row 2
  result[ 8] = ih_multiply(m.m_matrix, m_matrix, 2, 0);
  result[ 9] = ih_multiply(m.m_matrix, m_matrix, 2, 1);
  result[10] = ih_multiply(m.m_matrix, m_matrix, 2, 2);
  result[11] = ih_multiply(m.m_matrix, m_matrix, 2, 3);
  // row 3
  result[12] = ih_multiply(m.m_matrix, m_matrix, 3, 0);
  result[13] = ih_multiply(m.m_matrix, m_matrix, 3, 1);
  result[14] = ih_multiply(m.m_matrix, m_matrix, 3, 2);
  result[15] = ih_multiply(m.m_matrix, m_matrix, 3, 3);

  for (size_t i = 0; i < 16; ++i)
  {
    m_matrix[i] = result[i];
  }

  return *this;
}

//
// Public class constants
//

const Matrix Matrix::I;
