// Polyhedron.hpp
//
// Copyright 2012-2013 Roan Trail, Inc.
//
// This file is part of Tovero.
//
// Tovero is free software; you can redistribute it and/or modify it
// under the terms of the GNU Lesser General Public License
// version 2.1 as published by the Free Software Foundation.
//
// Tovero is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
// Lesser General Public License for more details.  You should have
// received a copy of the GNU Lesser General Public License along with
// Tovero. If not, see <http://www.gnu.org/licenses/>.

// Class that handles 5 polyhedron solids:
//
// 8 vertex
//   Vertices 1234 specify the front face, numbered counterclockwise from vertex 1, lower left.
//   Vertices 5678 specify the rear face, numbered counterclockwise from vertex 5, lower left.
//
// 7 vertex
//   Vertices 1234 specify the front face, 1 is bottom left, then numbered counterclockwise.
//   Vertices 567 specify the rear face, numbered clockwise from 5, lower left.
//
// 6 vertex
//   Vertices 1234 specify the front face, numbered counterclockwise from vertex 1, lower left.
//   Vertices 5 and 6 specify the rear edge, upper and lower, respectively.
//
// 5 vertex
//
// 4 vertex
//

#ifndef TOVERO_MATH_POLYHEDRON_HPP_
#define TOVERO_MATH_POLYHEDRON_HPP_

#include <tovero/math/geometry/Polyhedron_base.hpp>
#include <tovero/math/geometry/Point.hpp>
#include <string>
#include <vector>

namespace Roan_trail
{
  namespace Tovero_support
  {
    class Error_param;
  }

  namespace Tovero_math
  {
    class Geometric_tolerances;

    class Polyhedron : public Polyhedron_base
    {
    public:
      // constructors/copy
      //   defaults to a an axis aligned box with sides 2.0 meters, centered around the origin
      Polyhedron(const std::string& name = "");
      explicit Polyhedron(const std::vector<Point>& vertices,
                          const std::string& name = "");
      virtual Solid* clone_solid() const { return new Polyhedron(*this); }
      //   Note: let compiler generate copy constructor and operator=
      // accessors
      const std::vector<Point>& vertices() const { return m_vertices; }
      std::vector<Point>& vertices() { return m_vertices; }
      //
      virtual bool is_valid(const Geometric_tolerances& tolerances,
                            Roan_trail::Tovero_support::Error_param& return_error) const;
      // visitor
      virtual Solid_visitor::Visit_result accept(Solid_visitor& visitor) const { return visitor.visit(*this); }
      // other
      virtual std::string solid_class() const { return "Polyhedron"; }
      virtual Polyhedron& generalize() const { return *const_cast<Polyhedron*>(this); }
      virtual Solid& specialize(const Geometric_tolerances& tolerances) const;
    protected:
      // destructor
      virtual ~Polyhedron() {}
    private:
      // solid parameters
      std::vector<Point> m_vertices;
    };
  }
}

#endif // TOVERO_MATH_POLYHEDRON_HPP_
