// Sphere.cpp
//
// Copyright 2012-2013 Roan Trail, Inc.
//
// This file is part of Tovero.
//
// Tovero is free software; you can redistribute it and/or modify it
// under the terms of the GNU Lesser General Public License
// version 2.1 as published by the Free Software Foundation.
//
// Tovero is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
// Lesser General Public License for more details.  You should have
// received a copy of the GNU Lesser General Public License along with
// Tovero. If not, see <http://www.gnu.org/licenses/>.

#include <tovero/math/geometry/Sphere.hpp>
#include <tovero/math/geometry/Distance.hpp>
#include <tovero/math/geometry/Ellipsoid.hpp>
#include <tovero/math/geometry/Geometric_tolerances.hpp>
#include <tovero/math/geometry/Point.hpp>
#include <tovero/math/geometry/Unit_vector.hpp>
#include <tovero/math/geometry/Vector.hpp>
#include <tovero/support/common.hpp>
#include <tovero/support/error/Error.hpp>
#include <tovero/support/error/Math_error.hpp>
#include <sstream>
#include <string>

using std::string;
using std::stringstream;
using Roan_trail::Tovero_support::Error_param;
using Roan_trail::Tovero_support::Math_error;
using namespace Roan_trail::Tovero_math;

//
// Constructor
//
Sphere::Sphere(const std::string& name)
  : Ellipsoid_base(name),
    m_center(Point::O),
    m_radius(Distance(1.0 * Distance::meter))
{
}

Sphere::Sphere(const Point& center,
               const Distance& radius,
               const string& name)
  : Ellipsoid_base(name),
    m_center(center),
    m_radius(radius)
{
}

//
// Predicates
//

bool Sphere::is_valid(const Geometric_tolerances& tolerances, Error_param& return_error) const
{
  precondition(!return_error());

  bool return_value = false;

  start_error_block();

  // check for positive radius
  if (tolerances.is_distance_near_zero(m_radius)
      || (m_radius < Distance(0.0 * Distance::meter)))
  {
    const string& solid_name = name();
    stringstream diagnostic_stream;
    diagnostic_stream << "Invalid " + solid_class();
    diagnostic_stream << ((solid_name == "") ? string("") : (string(" (") + solid_name + string(")")));
    diagnostic_stream << ": the radius is non-positive";
    on_error(true, new Math_error(error_location(),
                                  Math_error::validation,
                                  diagnostic_stream.str()));
  }

  return_value = true;
  goto exit_point;

  end_error_block();

  default_error_handler_and_cleanup(return_error,
                                    return_value,
                                    false);

 exit_point:
  postcondition(return_error.is_valid_at_return(return_value));
  return return_value;
}

//
// Other
//

Ellipsoid& Sphere::generalize() const
{
  const Vector axis_a = Unit_vector::x * m_radius;
  const Vector axis_b = Unit_vector::y * m_radius;
  const Vector axis_c = Unit_vector::z * m_radius;

  Ellipsoid* ellipsoid = new Ellipsoid(m_center,
                                       axis_a,
                                       axis_b,
                                       axis_c,
                                       name());

  return *ellipsoid;
}
