// Unit_vector.hpp
//
// Copyright 2012-2013 Roan Trail, Inc.
//
// This file is part of Tovero.
//
// Tovero is free software; you can redistribute it and/or modify it
// under the terms of the GNU Lesser General Public License
// version 2.1 as published by the Free Software Foundation.
//
// Tovero is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
// Lesser General Public License for more details.  You should have
// received a copy of the GNU Lesser General Public License along with
// Tovero. If not, see <http://www.gnu.org/licenses/>.

// affine 3D vector

// TODO: class invariant, length equal to 1 (or 0) within tolerance.

#ifndef TOVERO_MATH_UNIT_VECTOR_HPP_
#define TOVERO_MATH_UNIT_VECTOR_HPP_

#include <tovero/math/geometry/Unitless.hpp>
#include <iostream>
#include <cmath>
#include <cstddef>

namespace Roan_trail
{
  namespace Tovero_math
  {
    class Unit_vector
    {
    public:
      // constructors
      inline Unit_vector();
      Unit_vector(const Unitless& x,
                  const Unitless& y,
                  const Unitless& z);
      inline Unit_vector(const Unit_vector& v);
      static Unitless normalize(const Unitless& x,
                                const Unitless& y,
                                const Unitless& z,
                                Unit_vector& return_u);
      // operators
      inline Unit_vector& operator=(const Unit_vector& v);
      const Unitless& operator[](size_t index) const { return m_v[index]; }
      void set_at(size_t index, const Unitless& value);
      inline Unit_vector operator+(const Unit_vector& v) const;
      inline Unit_vector operator-(const Unit_vector& v) const;
      inline Unit_vector& operator+=(const Unit_vector& v);
      inline Unit_vector& operator-=(const Unit_vector& v);
      // functions
      const Unitless length() const { return m_v[0] * m_v[0] + m_v[1] * m_v[1] + m_v[2] * m_v[2]; }
      inline const Unitless dot(const Unit_vector& v) const;
      inline Unit_vector cross(const Unit_vector& v) const;
      inline Unit_vector perpendicular() const;
      // predefined vectors
      static const Unit_vector x;
      static const Unit_vector y;
      static const Unit_vector z;
      static const Unit_vector zero;
    private:
      Unitless m_v[3];
    };

    //
    // Free functions
    //

    inline std::ostream& operator<<(std::ostream& os, const Unit_vector& v);

    //
    // Inline definitions
    //

    //
    //   Constructors
    //

    inline Unit_vector::Unit_vector()
    {
      m_v[0] = Unitless(0.0);
      m_v[1] = Unitless(0.0);
      m_v[2] = Unitless(0.0);
    }

    inline Unit_vector::Unit_vector(const Unit_vector& v)
    {
      m_v[0] = v.m_v[0];
      m_v[1] = v.m_v[1];
      m_v[2] = v.m_v[2];
    }

    //
    //   Operators
    //

    inline Unit_vector& Unit_vector::operator=(const Unit_vector& v)
    {
      if (this != &v)
      {
        m_v[0] = v.m_v[0];
        m_v[1] = v.m_v[1];
        m_v[2] = v.m_v[2];
      }
      return *this;
    }

    inline Unit_vector Unit_vector::operator+(const Unit_vector& v) const
    {
      return Unit_vector(m_v[0] + v.m_v[0],
                         m_v[1] + v.m_v[1],
                         m_v[2] + v.m_v[2]);
    }

    inline Unit_vector Unit_vector::operator-(const Unit_vector& v) const
    {
      return Unit_vector(m_v[0] - v.m_v[0],
                         m_v[1] - v.m_v[1],
                         m_v[2] - v.m_v[2]);
    }

    inline Unit_vector& Unit_vector::operator+=(const Unit_vector& v)
    {
      m_v[0] += v.m_v[0];
      m_v[1] += v.m_v[1];
      m_v[2] += v.m_v[2];

      normalize(m_v[0],
                m_v[1],
                m_v[2],
                *this);

      return *this;
    }

    inline Unit_vector& Unit_vector::operator-=(const Unit_vector& v)
    {
      m_v[0] -= v.m_v[0];
      m_v[1] -= v.m_v[1];
      m_v[2] -= v.m_v[2];

      normalize(m_v[0],
                m_v[1],
                m_v[2],
                *this);

      return *this;
    }

    //
    //   Functions
    //

    inline const Unitless Unit_vector::dot(const Unit_vector& v) const
    {
      const Unitless return_value(m_v[0].value() * v.m_v[0].value()
                                  + m_v[1].value() * v.m_v[1].value()
                                  + m_v[2].value() * v.m_v[2].value());

      return return_value;
    }

    inline Unit_vector Unit_vector::cross(const Unit_vector& v) const
    {
      return Unit_vector(Unitless(m_v[1].value() * v.m_v[2].value() - m_v[2].value() * v.m_v[1].value()),
                         Unitless(m_v[2].value() * v.m_v[0].value() - m_v[0].value() * v.m_v[2].value()),
                         Unitless(m_v[0].value() * v.m_v[1].value() - m_v[1].value() * v.m_v[0].value()));
    }

    // TODO: there is a faster, better way to do this...
    inline Unit_vector Unit_vector::perpendicular() const
    {
      Unit_vector return_value(Unitless(0.0),
                               Unitless(0.0),
                               Unitless(0.0));

      // find the axis closest to 90 deg to
      // this vector (minimum absolute value of the dot products)
      const Unitless abs_dot_x = dot(Unit_vector::x);
      const Unitless abs_dot_y = dot(Unit_vector::y);
      const Unitless abs_dot_z = dot(Unit_vector::z);

      Unit_vector axis = Unit_vector::x;
      if (abs_dot_x < abs_dot_y)
      {
        if (abs_dot_z < abs_dot_x)
        {
          axis = Unit_vector::z;
        }
      }
      else
      {
        if (abs_dot_y < abs_dot_z)
        {
          axis = Unit_vector::y;
        }
        else
        {
          axis = Unit_vector::z;
        }
      }

      return_value = cross(axis);

      return return_value;;
    }

    //
    //   Free functions
    //

    inline std::ostream& operator<<(std::ostream& os, const Unit_vector& v)
    {
      os << "[" << v[0].value() << ", " << v[1].value() << ", " << v[2].value() << "]";

      return os;
    }
  }
}

#endif // TOVERO_MATH_UNIT_VECTOR_HPP_
