/* Vgrid - Virtual grid program for radiology
   Copyright (C) 2020, 2021 Sonia Diaz Pacheco.

   This program is free software: you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/
/*
   Exit status: 0 for a normal exit, 1 for environmental problems
   (file not found, invalid flags, I/O errors, etc), 2 to indicate a
   corrupt or invalid input file, 3 for an internal consistency error
   (e.g. bug) which caused vgrid to panic.
*/

#define _FILE_OFFSET_BITS 64

#include <algorithm>
#include <cctype>
#include <cerrno>
#include <climits>
#include <csignal>
#include <cstdio>
#include <cstdlib>
#include <cstring>
#include <new>
#include <string>
#include <vector>
#include <unistd.h>
#if defined __MSVCRT__ || defined __OS2__ || defined __DJGPP__
#include <io.h>
#if defined __MSVCRT__
#define strtoull std::strtoul
#define SIGHUP SIGTERM
#endif
#endif

#include "arg_parser.h"
#include "vgrid.h"
#include "matrix.h"

#ifndef O_BINARY
#define O_BINARY 0
#endif

#if CHAR_BIT != 8
#error "Environments where CHAR_BIT != 8 are not supported."
#endif

int verbosity = 0;

namespace {

const char * const program_name = "vgrid";
const char * const program_year = "2021";
const char * invocation_name = program_name;		// default value
const char * const mem_msg = "Not enough memory.";

enum Mode { m_compare, m_copy, m_info, m_rawcopy, m_recontrast, m_reduce,
            m_subtract, m_test };

/* Variables used in signal handler context.
   They are not declared volatile because the handler never returns. */
std::string output_filename;
FILE * outfile = 0;
bool delete_output_on_interrupt = false;


void show_help()
  {
  std::printf( "Vgrid is a virtual grid program for radiology which improves image contrast\n"
               "by means of Adaptive Windowing.\n"
               "\nAdaptive Windowing is based exclusively on linear transformations of pixel\n"
               "values so that the contrast in the neighborhood of each pixel is expanded\n"
               "until it fills the full range of values available. The effect is similar to\n"
               "passing a 'contrast magnifier' over the image.\n"
               "\nVgrid reads images in PNG greyscale or RGB format and saves images in PNG\n"
               "format using by default the same bit depth as the input image.\n"
               "Vgrid accepts images with a bit depth of 8 or 16 bits.\n"
               "\nUsage: %s [options] [files]\n", invocation_name );
  std::printf( "\nOptions:\n"
               "  -h, --help                 display this help and exit\n"
               "  -V, --version              output version information and exit\n"
               "  -b, --bit-depth=<n>        force bit depth of images produced (8 or 16)\n"
               "  -c, --stdout               write to standard output\n"
               "  -C, --copy                 copy input to output\n"
               "  -e, --expand               expand pixel values to fill image range\n"
               "  -f, --force                overwrite existing output files\n"
               "  -i, --invert               invert image levels (white <--> black)\n"
               "  -I, --info                 show image information\n"
               "  -j, --color                process color (RGB) images\n"
               "  -m, --median=<radius>      run result through median filter of side 2*r+1\n"
               "  -o, --output=<file>        write the output to <file>\n"
               "  -p, --compare=<file>       compare each image with <file>\n"
               "  -q, --quiet                suppress all messages\n"
               "  -R, --raw-copy=w,h,bd,ch   convert raw pixel data to PNG output\n"
               "  -s, --suffix=<suf>         add suffix <suf> instead of '_vg' to <file>\n"
               "  -S, --subtract=<file>      subtract each image from <file> pixel per pixel\n"
               "      --self-test            run self-tests (debug)\n"
               "  -v, --verbose              be verbose (a 2nd -v gives more)\n"
               "  -Z, --reduce=<scale>       reduce image to 1/scale of original size per side\n"
               "\nIf no file names are given, or if a file is '-', vgrid reads an image\n"
               "from standard input and writes the result to standard output.\n"
               "\nExit status: 0 for a normal exit, 1 for environmental problems (file\n"
               "not found, invalid flags, I/O errors, etc), 2 to indicate a corrupt or\n"
               "invalid input file, 3 for an internal consistency error (e.g. bug) which\n"
               "caused vgrid to panic.\n"
               "\nReport bugs to vgrid-bug@nongnu.org\n"
               "Vgrid home page: http://www.nongnu.org/vgrid/vgrid.html\n" );
  }


void show_version()
  {
  std::printf( "%s %s\n", program_name, PROGVERSION );
  std::printf( "Copyright (C) %s Sonia Diaz Pacheco.\n", program_year );
  std::printf( "License GPLv2+: GNU GPL version 2 or later <http://gnu.org/licenses/gpl.html>\n"
               "This is free software: you are free to change and redistribute it.\n"
               "There is NO WARRANTY, to the extent permitted by law.\n" );
  }


// Recognized formats: <num>[YZEPTGM][i], <num>k, <num>Ki
//
long long getnum( const char * const ptr,
                  const long long llimit = -LLONG_MAX,
                  const long long ulimit = LLONG_MAX,
                  const char ** const tailp = 0 )
  {
  char * tail;
  errno = 0;
  long long result = strtoll( ptr, &tail, 0 );
  if( tail == ptr )
    {
    show_error( "Bad or missing numerical argument.", 0, true );
    std::exit( 1 );
    }

  if( !errno && tail[0] )
    {
    char * const p = tail++;
    int factor = 1000;				// default factor
    int exponent = -1;				// -1 = bad multiplier
    switch( *p )
      {
      case 'Y': exponent = 8; break;
      case 'Z': exponent = 7; break;
      case 'E': exponent = 6; break;
      case 'P': exponent = 5; break;
      case 'T': exponent = 4; break;
      case 'G': exponent = 3; break;
      case 'M': exponent = 2; break;
      case 'K': if( tail[0] == 'i' ) { ++tail; factor = 1024; exponent = 1; } break;
      case 'k': if( tail[0] != 'i' ) exponent = 1; break;
      default : if( tailp ) { tail = p; exponent = 0; }
      }
    if( exponent > 1 && tail[0] == 'i' ) { ++tail; factor = 1024; }
    if( exponent < 0 || ( !tailp && tail[0] != 0 ) )
      {
      show_error( "Bad multiplier in numerical argument.", 0, true );
      std::exit( 1 );
      }
    for( int i = 0; i < exponent; ++i )
      {
      if( LLONG_MAX / factor >= llabs( result ) ) result *= factor;
      else { errno = ERANGE; break; }
      }
    }
  if( !errno && ( result < llimit || result > ulimit ) ) errno = ERANGE;
  if( errno )
    {
    show_error( "Numerical argument out of limits." );
    std::exit( 1 );
    }
  if( tailp ) *tailp = tail;
  return result;
  }


// Recognized formats: <width>,<height>,<bit_depth>,<channels>
//
void parse_raw( const char * const ptr, Raw_params & raw_params )
  {
  if( std::isdigit( ptr[0] ) )
    {
    const char * tail = ptr;
    raw_params.width = getnum( ptr, 1, INT_MAX, &tail );
    if( tail[0] == ',' )
      {
      raw_params.height = getnum( tail + 1, 1, INT_MAX, &tail );
      if( tail[0] == ',' )
        {
        raw_params.bit_depth = getnum( tail + 1, 1, 16, &tail );
        if( tail[0] == ',' )
          {
          raw_params.channels = getnum( tail + 1, 1, 4, &tail );
          if( tail[0] == 0 ) return;
          }
        }
      }
    }
  show_error( "Bad or missing argument in option '--raw-copy'.", 0, true );
  std::exit( 1 );
  }


void parse_bit_depth( const std::string & arg, int & cl_bit_depth )
  {
  if( arg == "8" ) cl_bit_depth = 8;
  else if( arg == "16" ) cl_bit_depth = 16;
  else
    {
    show_error( "Invalid bit depth. Must be 8 or 16." );
    std::exit( 1 );
    }
  }


void set_mode( Mode & program_mode, const Mode new_mode )
  {
  if( program_mode != m_recontrast && program_mode != new_mode )
    {
    show_error( "Only one operation can be specified.", 0, true );
    std::exit( 1 );
    }
  program_mode = new_mode;
  }


bool binary_output( const Mode program_mode )
  {
  switch( program_mode )
    {
    case m_compare: case m_info: case m_test: return false;
    case m_copy: case m_rawcopy: case m_recontrast: case m_reduce:
    case m_subtract: return true;
    }
  return true;				// avoid compiler warning
  }


void set_outname( const std::string & name, const char * const output_suffix,
                  const Mode program_mode )
  {
  output_filename = name;
  const unsigned i = output_filename.rfind( '.' );
  if( i > 0 && i < output_filename.size() - 1 )		// name has extension
    {
    const std::string ext( name, i + 1, name.size() - i - 1 );
    if( program_mode == m_rawcopy && ext == "raw" )
      { output_filename.resize( i + 1 ); output_filename += "png"; }
    else output_filename.insert( i, output_suffix );
    }
  else { output_filename += output_suffix; output_filename += "png"; }
  }


FILE * open_instream( const char * const name )
  {
  FILE * infile = std::fopen( name, "rb" );
  if( !infile )
    show_file_error( name, "Can't open input file", errno );
  return infile;
  }


bool open_outstream( const bool force )
  {
  outfile = std::fopen( output_filename.c_str(), force ? "wb" : "wbx" );
  if( outfile ) delete_output_on_interrupt = true;
  else if( verbosity >= 0 )
    {
    if( errno == EEXIST )
      std::fprintf( stderr, "%s: Output file '%s' already exists, skipping.\n",
                    program_name, output_filename.c_str() );
    else
      std::fprintf( stderr, "%s: Can't create output file '%s': %s\n",
                    program_name, output_filename.c_str(), std::strerror( errno ) );
    }
  return ( outfile != 0 );
  }


void set_signals( void (*action)(int) )
  {
  std::signal( SIGHUP, action );
  std::signal( SIGINT, action );
  std::signal( SIGTERM, action );
  }


void cleanup_and_fail( const int retval )
  {
  set_signals( SIG_IGN );			// ignore signals
  if( delete_output_on_interrupt )
    {
    delete_output_on_interrupt = false;
    if( verbosity >= 0 )
      std::fprintf( stderr, "%s: Deleting output file '%s', if it exists.\n",
                    program_name, output_filename.c_str() );
    if( outfile != 0 ) { std::fclose( outfile ); outfile = 0; }
    if( std::remove( output_filename.c_str() ) != 0 && errno != ENOENT )
      show_error( "WARNING: deletion of output file (apparently) failed." );
    }
  std::exit( retval );
  }


extern "C" void signal_handler( int )
  {
  show_error( "Control-C or similar caught, quitting." );
  cleanup_and_fail( 1 );
  }


inline void set_retval( int & retval, const int new_val )
  { if( retval < new_val ) retval = new_val; }


void check_tty_in( const char * const input_filename, FILE * const infile,
                   int & retval )
  {
  if( isatty( fileno( infile ) ) )		// for example /dev/tty
    { show_file_error( input_filename,
                       "I won't read image data from a terminal." );
      set_retval( retval, 1 ); cleanup_and_fail( retval ); }
  }

bool check_tty_out( FILE * const outfile )
  {
  if( isatty( fileno( outfile ) ) )
    { show_file_error( output_filename.size() ?
                       output_filename.c_str() : "(stdout)",
                       "I won't write image data to a terminal." );
      return false; }
  return true;
  }


void close_outstream()
  {
  if( std::fclose( outfile ) != 0 )
    {
    show_error( "Error closing output file", errno );
    cleanup_and_fail( 1 );
    }
  outfile = 0;
  delete_output_on_interrupt = false;
  }


int compare_images( Matrix & image, const char * const matname,
                    const char * const second_matname, int maxval )
  {
  int retval = 0;
  try {
    FILE * const infile2 = open_instream( second_matname );
    if( !infile2 ) return 1;
    check_tty_in( second_matname, infile2, retval );
    if( !read_check_png_sig8( infile2 ) )
      { show_file_error( second_matname, "Not a PNG file." ); return 2; }
    int maxval2;		// maxval of the reference PNG image
    Matrix image2( infile2, 8, &maxval2 );
    if( maxval == maxval2 )
      { if( verbosity >= 0 )
          std::printf( "Both images have a bit depth of %u bit.\n",
                       real_bits( maxval ) ); }
    else
      {
      if( verbosity >= 0 )
        std::printf( "Bit depth of '%s' (%u bit) and '%s' (%u bit) differ.\n",
                     matname, real_bits( maxval ),
                     second_matname, real_bits( maxval2 ) ); retval = 1;
      // scale down the deeper matrix to match the lower bit depth
      if( maxval > maxval2 )
        { image *= (double)maxval2 / maxval; maxval = maxval2; }
      else { image2 *= (double)maxval / maxval2; maxval2 = maxval; }
      }
    if( image == image2 )
      { if( verbosity >= 0 )
          std::printf( "Content of '%s' and '%s' is identical.\n",
                       matname, second_matname ); }
    else
      { const bool equiv = image.equivalent( image2, 1 );
        if( verbosity >= 0 )
          {
          std::printf( "Content of '%s' and '%s' %s.\n", matname,
                       second_matname, equiv ? "is equivalent" : "differ" );
          const double mpd = ( image2 - image ).norm1() / image.size();
          std::printf( "Mean pixel difference = %g (%.3f%% of maximum)\n",
                       mpd, 100.0 * ( mpd / maxval ) );
          }
        if( !equiv ) retval = 1; }
    }
  catch( std::bad_alloc & )
    { show_file_error( second_matname, mem_msg ); retval = 1; }
  catch( Error & e ) { show_file_error( second_matname, e.msg ); retval = 2; }
  return retval;
  }


int subtract_images( Matrix & image, const char * const second_matname,
                     int maxval )
  {
  int retval = 0;
  try {
    FILE * const infile2 = open_instream( second_matname );
    if( !infile2 ) return 1;
    check_tty_in( second_matname, infile2, retval );
    if( !read_check_png_sig8( infile2 ) )
      { show_file_error( second_matname, "Not a PNG file." ); return 2; }
    int maxval2;		// maxval of the reference PNG image
    Matrix image2( infile2, 8, &maxval2 );
    const long rows = std::max( image.height(), image2.height() );
    const long cols = std::max( image.width(), image2.width() );
    image.resize( rows, cols );		// pad images to same size with zeros
    image2.resize( rows, cols );

    // scale down the deeper matrix to match the lower bit depth
    if( maxval > maxval2 )
      { image *= (double)maxval2 / maxval; maxval = maxval2; }
    else if( maxval < maxval2 )
      { image2 *= (double)maxval / maxval2; maxval2 = maxval; }

    const unsigned bit_depth = ( maxval > 255 ) ? 16 : 8;
    const unsigned zero = ( maxval + 1 ) / 2;
    ( ( image - image2 ) + zero ).write_png( outfile, bit_depth );
    }
  catch( std::bad_alloc & )
    { show_file_error( second_matname, mem_msg ); retval = 1; }
  catch( Error & e ) { show_file_error( second_matname, e.msg ); retval = 2; }
  return retval;
  }


int process_image( FILE * const infile, const char * const second_matname,
                   const Pretty_print & pp, const Mode program_mode,
                   const Raw_params & raw_params, const int cl_bit_depth,
                   const int median_radius, const int scale, const bool expand,
                   const bool invert, const bool keep_color )
  {
  int retval = 0;

  try {
    if( program_mode == m_rawcopy )
      return raw_to_png( pp.name(), infile, outfile, raw_params );
    if( !read_check_png_sig8( infile ) )
      { show_file_error( pp.name(), "Not a PNG file." ); return 2; }
    if( program_mode == m_info ) return show_png_info( pp.name(), infile, 8 );
    int maxval;			// maxval of the original PNG image
    Color_info color_info;	// used to restore color when saving image
    Matrix image( infile, 8, &maxval, keep_color ? &color_info : 0, invert );
    const unsigned bit_depth = cl_bit_depth ? cl_bit_depth : real_bits( maxval );
    if( verbosity >= 1 - ( program_mode == m_test ) )
      pp( 0, binary_output( program_mode ) ? stderr : stdout );
    switch( program_mode )
      {
      case m_compare: retval = compare_images( image, pp.name(),
                               second_matname, maxval ); break;
      case m_reduce: image = image.reduce( scale, color_info );	// fall through
      case m_copy:
        if( maxval != ( 1 << bit_depth ) - 1 )	// adjust to bit_depth
          image *= (double)( ( 1 << bit_depth ) - 1 ) / maxval;
        if( median_radius > 0 ) image = image.median_filter( median_radius );
        retval = !image.write_png( outfile, bit_depth, color_info, expand ); break;
      case m_info: break;			// avoid compiler warning
      case m_rawcopy: break;			// avoid compiler warning
      case m_recontrast: retval = recontrast( image, outfile, maxval, bit_depth,
                                    median_radius, color_info, expand ); break;
      case m_subtract: retval = subtract_images( image, second_matname, maxval );
        break;
      case m_test: retval = self_test2( image, maxval, program_name ); break;
      }
    }
  catch( std::bad_alloc & ) { pp( mem_msg ); retval = 1; }
  catch( Error & e ) { pp(); show_error( e.msg ); retval = 2; }
  if( verbosity >= 1 && retval == 0 && binary_output( program_mode ) )
    std::fputs( "done\n", stderr );
  return retval;
  }

} // end namespace


void Pretty_print::operator()( const char * const msg, FILE * const f ) const
  {
  if( verbosity >= 0 )
    {
    if( first_post )
      {
      first_post = false;
      std::fputs( padded_name.c_str(), f );
      if( !msg ) std::fflush( f );
      }
    if( msg ) std::fprintf( f, "%s\n", msg );
    }
  }


void show_error( const char * const msg, const int errcode, const bool help )
  {
  if( verbosity < 0 ) return;
  if( msg && msg[0] )
    std::fprintf( stderr, "%s: %s%s%s\n", program_name, msg,
                  ( errcode > 0 ) ? ": " : "",
                  ( errcode > 0 ) ? std::strerror( errcode ) : "" );
  if( help )
    std::fprintf( stderr, "Try '%s --help' for more information.\n",
                  invocation_name );
  }


void show_file_error( const char * const filename, const char * const msg,
                      const int errcode )
  {
  if( verbosity >= 0 )
    std::fprintf( stderr, "%s: %s: %s%s%s\n", program_name, filename, msg,
                  ( errcode > 0 ) ? ": " : "",
                  ( errcode > 0 ) ? std::strerror( errcode ) : "" );
  }


void internal_error( const char * const msg )
  {
  if( verbosity >= 0 )
    std::fprintf( stderr, "%s: internal error: %s\n", program_name, msg );
  std::exit( 3 );
  }


int main( const int argc, const char * const argv[] )
  {
  std::string default_output_filename;
  std::vector< std::string > filenames;
  const char * output_suffix = "_vg";
  const char * second_matname = 0;
  int cl_bit_depth = 0;		// if != 0 override input image bit depth
  int median_radius = 0;	// if > 0 append median filter of side 2*r+1
  int scale = 0;
  Mode program_mode = m_recontrast;
  Raw_params raw_params;
  bool expand = false;
  bool force = false;
  bool invert = false;
  bool keep_color = false;
  bool to_stdout = false;
  if( argc > 0 ) invocation_name = argv[0];

  enum { opt_st = 256 };
  const Arg_parser::Option options[] =
    {
    { 'b', "bit-depth",     Arg_parser::yes },
    { 'c', "stdout",        Arg_parser::no  },
    { 'C', "copy",          Arg_parser::no  },
    { 'e', "expand",        Arg_parser::no  },
    { 'f', "force",         Arg_parser::no  },
    { 'h', "help",          Arg_parser::no  },
    { 'i', "invert",        Arg_parser::no  },
    { 'I', "info",          Arg_parser::no  },
    { 'j', "color",         Arg_parser::no  },
    { 'm', "median",        Arg_parser::yes },
    { 'o', "output",        Arg_parser::yes },
    { 'p', "compare",       Arg_parser::yes },
    { 'q', "quiet",         Arg_parser::no  },
    { 'R', "raw-copy",      Arg_parser::yes },
    { 's', "suffix",        Arg_parser::yes },
    { 'S', "subtract",      Arg_parser::yes },
    { 'v', "verbose",       Arg_parser::no  },
    { 'V', "version",       Arg_parser::no  },
    { 'Z', "reduce",        Arg_parser::yes },
    { opt_st, "self-test",  Arg_parser::no  },
    {  0 , 0,               Arg_parser::no  } };

  const Arg_parser parser( argc, argv, options );
  if( parser.error().size() )				// bad option
    { show_error( parser.error().c_str(), 0, true ); return 1; }

  int argind = 0;
  for( ; argind < parser.arguments(); ++argind )
    {
    const int code = parser.code( argind );
    if( !code ) break;					// no more options
    const std::string & sarg = parser.argument( argind );
    const char * const arg = sarg.c_str();
    switch( code )
      {
      case 'b': parse_bit_depth( sarg, cl_bit_depth ); break;
      case 'c': to_stdout = true; break;
      case 'C': set_mode( program_mode, m_copy ); break;
      case 'e': expand = true; break;
      case 'f': force = true; break;
      case 'h': show_help(); return 0;
      case 'i': invert = true; break;
      case 'I': set_mode( program_mode, m_info ); break;
      case 'j': keep_color = true; break;
      case 'm': median_radius = getnum( arg, 1, 1000 ); break;
      case 'o': if( sarg == "-" ) to_stdout = true;
                else { default_output_filename = sarg; } break;
      case 'p': set_mode( program_mode, m_compare ); second_matname = arg;
                break;
      case 'q': verbosity = -1; break;
      case 'R': set_mode( program_mode, m_rawcopy );
                parse_raw( arg, raw_params ); break;
      case 's': output_suffix = arg; break;
      case 'S': set_mode( program_mode, m_subtract ); second_matname = arg;
                break;
      case 'v': if( verbosity < 4 ) ++verbosity; break;
      case 'V': show_version(); return 0;
      case 'Z': set_mode( program_mode, m_reduce );
                scale = getnum( arg, 2, 32767 ); break;
      case opt_st: set_mode( program_mode, m_test ); break;
      default : internal_error( "uncaught option." );
      }
    } // end process options

#if defined __MSVCRT__ || defined __OS2__ || defined __DJGPP__
  setmode( STDIN_FILENO, O_BINARY );
  setmode( STDOUT_FILENO, O_BINARY );
#endif

  if( program_mode == m_test )
    {
    const int ret = self_test1();
    if( ret != 0 )
      { show_error( "Fatal failure in basic operation." ); return ret; }
    }

  bool filenames_given = false;
  for( ; argind < parser.arguments(); ++argind )
    {
    filenames.push_back( parser.argument( argind ) );
    if( filenames.back() != "-" ) filenames_given = true;
    }
  if( filenames.empty() ) filenames.push_back("-");

  const bool binout = binary_output( program_mode );
  if( !binout ) to_stdout = false;		// apply overrides
  if( !binout || to_stdout ) default_output_filename.clear();

  if( to_stdout && binout )			// check tty only once
    { default_output_filename.clear();		// -c overrides -o
      outfile = stdout; if( !check_tty_out( outfile ) ) return 1; }
  else
    outfile = 0;

  const bool to_file = !to_stdout && binout && default_output_filename.size();
  if( !to_stdout && binout && ( filenames_given || to_file ) )
    set_signals( signal_handler );

  Pretty_print pp( filenames );

  int failed_tests = 0;
  int retval = 0;
  const bool one_to_one = !to_stdout && binout && !to_file;
  bool stdin_used = false;
  for( unsigned i = 0; i < filenames.size(); ++i )
    {
    std::string input_filename;
    FILE * infile;

    pp.set_name( filenames[i] );
    if( filenames[i] == "-" )
      {
      if( stdin_used ) continue; else stdin_used = true;
      infile = stdin;
      check_tty_in( pp.name(), infile, retval );
      if( one_to_one ) { outfile = stdout; output_filename.clear(); }
      }
    else
      {
      input_filename = filenames[i];
      infile = open_instream( input_filename.c_str() );
      if( !infile ) { set_retval( retval, 1 ); continue; }
      check_tty_in( pp.name(), infile, retval );
      if( one_to_one )			// open outfile after verifying infile
        {
        set_outname( input_filename, output_suffix, program_mode );
        if( !open_outstream( force ) )
          { set_retval( retval, 1 ); std::fclose( infile ); continue; }
        }
      }

    if( one_to_one && !check_tty_out( outfile ) )
      { set_retval( retval, 1 ); return retval; }	// don't delete a tty

    if( to_file && outfile == 0 )	// open outfile after verifying infile
      {
      output_filename = default_output_filename;
      if( !open_outstream( force ) || !check_tty_out( outfile ) )
        { std::fclose( infile ); return 1; }	// check tty only once and don't try to delete a tty
      }

    int tmp = process_image( infile, second_matname, pp, program_mode,
                             raw_params, cl_bit_depth, median_radius, scale,
                             expand, invert, keep_color );
    if( std::fclose( infile ) != 0 )
      {
      show_file_error( pp.name(), "Error closing input file", errno );
      if( tmp < 1 ) tmp = 1;
      }
    set_retval( retval, tmp );
    if( tmp )
      { if( binout ) cleanup_and_fail( retval ); else ++failed_tests; }
    if( delete_output_on_interrupt && one_to_one ) close_outstream();
    }
  if( delete_output_on_interrupt ) close_outstream();		// -o
  else if( outfile != 0 && std::fclose( outfile ) != 0 )	// -c
    {
    show_error( "Error closing stdout", errno );
    set_retval( retval, 1 );
    }
  if( failed_tests > 0 && verbosity >= 1 && filenames.size() > 1 )
    std::fprintf( stderr, "%s: warning: %d %s failed the test.\n",
                  program_name, failed_tests,
                  ( failed_tests == 1 ) ? "file" : "files" );
  return retval;
  }
