/* Vgrid - Virtual grid program for radiology
   Copyright (C) 2020, 2021 Sonia Diaz Pacheco.

   This program is free software: you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

class Matrix
  {
  std::vector< std::vector< double > > data;		// real values

  void compare_size( const Matrix & m ) const;
  static void test_size( const long rows, const long cols );

public:
  // create a Matrix with all elements set to val
  Matrix( const long rows, const long cols, const double val = 0.0 );

  // create a Matrix with elements in range [0, maxval] from a PNG file
  Matrix( FILE * const f, const int sig_read, int * const maxvalp = 0,
          Color_info * const color_infop = 0, const bool invert = false );

  // create a Matrix from a two-dimensional vector or array
  Matrix( const std::vector< std::vector< double > > & d );
  Matrix( const long rows, const long cols, const double array[] );

  static Matrix magic5() { const double magic[25] =
    { 17, 24,  1,  8, 15,
      23,  5,  7, 14, 16,
       4,  6, 13, 20, 22,
      10, 12, 19, 21,  3,
      11, 18, 25,  2,  9 }; return Matrix( 5, 5, magic ); }

  double get_element( const long row, const long col ) const
    { return data[row][col]; }
  void set_element( const long row, const long col, const double val )
    { data[row][col] = val; }

  long height() const { return data.size(); }
  long width() const { return data[0].size(); }
  long size() const { return height() * width(); }
  Matrix & resize( const long rows, const long cols, const double val = 0.0 );
  // reduce image to 1/scale of original size per side
  Matrix reduce( const int scale, Color_info & color_info ) const;

  double max() const;			// maximum element in matrix
  double min() const;			// minimum element in matrix
  double mean() const;			// average of elements in matrix
  double norm1() const;			// 1-norm of *this reshaped as vector
  double eunorm() const;	// Euclidean norm of *this reshaped as vector

  double median( const long row, const long col, const int radius ) const;

  Matrix & operator+=( const Matrix & m );
  Matrix & operator+=( const double val );
  Matrix & operator-=( const Matrix & m );
  Matrix & operator-=( const double val );
  Matrix & operator*=( const double val );
  Matrix & operator/=( const double val );

  Matrix operator+( const Matrix & m ) const
    { Matrix tmp( *this ); return tmp += m; }
  Matrix operator+( const double val ) const
    { Matrix tmp( *this ); return tmp += val; }
  Matrix operator-( const Matrix & m ) const
    { Matrix tmp( *this ); return tmp -= m; }
  Matrix operator-( const double val ) const
    { Matrix tmp( *this ); return tmp -= val; }

  bool equivalent( const Matrix & m, const double epsilon ) const;
  bool operator==( const Matrix & m ) const { return equivalent( m, 0 ); }
  bool operator!=( const Matrix & m ) const { return !( *this == m ); }

  // filter using a square neighborhood of '2 * radius + 1' pixels per side
  Matrix median_filter( const int radius ) const;

  void print() const;			// print matrix to standard output
  bool write_png( FILE * const f, const unsigned bit_depth = 16,
                  const Color_info & color_info = Color_info(),
                  const bool expand = false ) const;
  bool write_png( const std::string & filename, int n = -1,
                  const unsigned bit_depth = 16,
                  const bool expand = false ) const;
  };
