/* Copyright (C) 2002 Asfand Yar Qazi.

 This file is part of XBobble.

    XBobble is free software; you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    XBobble is distributed in the hope that it will be useful, but
    WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with XBobble; if not, write to the Free Software Foundation,
    Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA */

/*
15/10/2002: Changed string terminal symbol from alphanum+ to '.+'.
Fixed parser appropriately.
*/

/** @file Config_Parser.cc Implementation of members in
    Config_Parser.hh */

#include "Config_Parser.hh"

#include <cctype>

namespace XBobble
{

Config_Lexer::Config_Lexer()
 : Parent(), last_string(""), stmt_num(1)
{
}

Config_Lexer::Config_Lexer(std::istream& arg_istream)
 : Parent(arg_istream), last_string(""), stmt_num(1)
{
}

Token
Config_Lexer::next()
{
	char c;

	try
	{
		c = find_next();
	}
	catch(End_Of_Stream)
	{
		return EOF_TOKEN;
	}

	switch(c) {
	case '=':
		return ASSIGNMENT_TOKEN;

	case ';':
		++stmt_num;
		return END_STATEMENT_TOKEN;

	case '\'':
		last_string.clear();
		do {
			streamin->get(c);
			if(!istream_ok())
				return INCOMPLETE_STRING_TOKEN;
			switch(c) {
			case '\n':
				++line_num;
				return INCOMPLETE_STRING_TOKEN;
			case '\'':
				break;
			default:
				last_string += c;
			} // switch(c)
		} while(c != '\'');
		return STRING_TOKEN;

	} // switch(c)

	// Still here, so char sequence wasn't recognised...
	return UNKNOWN_TOKEN;
}

std::string
Config_Lexer::get_last_string() const
{
	return last_string;
}

void
Config_Lexer::set_istream(std::istream& arg_istream)
{
	Parent::set_istream(arg_istream);
	stmt_num = 1;
}

int
Config_Lexer::get_stmt_num() const
{
	return stmt_num;
}




Config_Parser::Config_Parser()
 : lookahead(UNKNOWN_TOKEN), lexer(), symtab(0), streamout(0)
{
}

Token
Config_Parser::get_lookahead() const
{
	return lookahead;
}

const Config_Lexer&
Config_Parser::get_lexer() const
{
	return lexer;
}

void
Config_Parser::process(std::istream& in, std::ostream& out,
		       Symtab& arg_symtab, const std::string& error_prefix)
	throw(Bad_Stream)
{
	symtab = &arg_symtab;
	lexer.set_istream(in);
	streamout = &out;

	if(!lexer.istream_ok())
		throw Bad_Stream("istream in Config_Parser::process()");

	if(!streamout->good())
		throw Bad_Stream("ostream in Config_Parser::process()");

	this->error_prefix = error_prefix;

	lookahead = lexer.next();

	stmt_list();

} // Config_Parser::process()

void
Config_Parser::match(Token t) throw(Unmatched)
{
	if(get_lookahead() != t)
		throw Unmatched();

	// else...
	lookahead = lexer.next();
}

void
Config_Parser::stmt_list()
{
/*
	stmt(); stmt_list();

	Optimised to avoid recursion.
*/

	while(1) {
		if(get_lookahead() == EOF_TOKEN)
			return;

		stmt();
	}
}

void
Config_Parser::stmt()
{
	std::string s1, s2;

	try
	{
		if(get_lookahead() == STRING_TOKEN)
			s1 = lexer.get_last_string();

		match(STRING_TOKEN); match(ASSIGNMENT_TOKEN);

		if(get_lookahead() == STRING_TOKEN)
			s2 = lexer.get_last_string();

		match(STRING_TOKEN); match(END_STATEMENT_TOKEN);

		(*symtab)[s1] = s2;
	}
	catch(Unmatched)
	{
		*streamout << error_prefix
			   << "Error - line " << lexer.get_line_num()
			   << ", statement " << lexer.get_stmt_num() << ": ";

		while(1)
		{
			if(get_lookahead() == EOF_TOKEN) {
				*streamout << error_prefix
					   << "Premature EOF\n";
				return;
			}

			switch(get_lookahead()) {
			case INCOMPLETE_STRING_TOKEN:
				*streamout << error_prefix
					   << "Incomplete string\n";
				break;

			case END_STATEMENT_TOKEN:
				lookahead = lexer.next();
				*streamout << error_prefix
					   << "Syntax error\n";
				return;

			case UNKNOWN_TOKEN:
				*streamout << error_prefix
					   << "Unknown token\n";
				break;

			default:
				break;

			} // switch...

			lookahead = lexer.next();
		}

	} // catch(Unmatched)

} // Config_Parser::stmt()

} // namespace XBobble


