/* Copyright (C) 2002 Asfand Yar Qazi.

 This file is part of XBobble.

    XBobble is free software; you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    XBobble is distributed in the hope that it will be useful, but
    WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with XBobble; if not, write to the Free Software Foundation,
    Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA */

/** @file Game.cc

    @see Game.hh
    @see Game_Elements_Accessor.hh */

#include "Game_Elements_Accessor.hh"
#include "Game.hh"
#include "System.hh"
#include "Game_Manager.hh"
#include "State.hh"
#include "Grid.hh"
#include "Launcher.hh"
#include "Chooser.hh"
#include "Timeout_Meter.hh"
#include "Grid_Shaker.hh"
#include "Ambient_Effects_Manager.hh"
#include "Launched_Ball_Manager.hh"
#include "Popped_Balls_Manager.hh"
#include "Out_Of_Game_State.hh"
#include "Gameplay_State.hh"
#include "Finished_Level_State.hh"
#include "Game_Over_State.hh"
#include "Video_Data.hh"
#include "gameplay_variables.hh"
#include <iostream>
#include <iomanip>
#include <sstream>
#include <memory>


namespace XBobble
{

namespace
{

/// @internal This class counts fpses.
class FPSCounter : public Video_Data
{
public:
	/// Init some variables and numbers display list
	FPSCounter()
	{
	}

	/// Call this each frame, as it appends the given delta to
	/// 'timestore' and creates a new 'last' fps measurement if
	/// necessary.
	void
	append_delta(uint32_t delta)
	{
		++frames;
		timestore += delta;
		if(timestore > 1000) {
				// TEMP until we get graphics working
			// Output '\b' for number of chars stored in last
			std::string tmp = last.str();
			tmp.replace(0, tmp.size(), tmp.size(), '\b');
			std::cout << tmp;
			tmp.replace(0, tmp.size(), tmp.size(), ' ');
			std::cout << tmp;
			tmp.replace(0, tmp.size(), tmp.size(), '\b');
			std::cout << tmp;
			// Calculate fps AT THIS POINT and reset the
			// members used to calculate it.
			last.str("");
			last << std::fixed << std::showpoint << std::setw(8)
			     << std::setfill(' ') << std::setprecision(2)
			     << static_cast<double>(frames)*1000/timestore
			     << "     ";
			std::cout << last.str(); std::cout.flush();

			timestore = 0;
			frames = 0;
		}
	}

private:
	/// Draw the fps counter on the screen
	void
	draw_handler() const
	{
		// @todo here
	}

	/// Keeps track of all the deltas
	uint32_t timestore;

	/// Tracks frames past
	uint32_t frames;

	/// The last fps calculated is stored here for easy rendering.
	std::ostringstream last;

}; // class FPSCounter


} // namespace

/// Impl for Game
class Game::Impl
{
	/// Derive from class "Game_Elements_Accessor" to have
	/// access to Game elements
	///
	/// @see Game_Elements_Accessor
	friend class Game_Elements_Accessor;

	friend class Game;

	/// Init, with game we belong to
	explicit
	Impl(Game& arg_game)
	 : game(arg_game),
	   grid(*this, Vars::grid_rows, Vars::grid_cols, Vars::ball_radius,
		"level1.grid"),
	   chooser(*this),
	   launcher(*this),
	   timeout_meter(*this),
	   grid_shaker(*this),
	   ambient_effects_manager(*this),
	   launched_ball_manager(*this),
	   popped_balls_manager(*this),

	   out_of_game_state(*this),
	   gameplay_state(*this),
	   game_over_state(*this),
	   finished_level_state(*this)
	{
	}

	FPSCounter game_fps;

	/// Game object we belong to
	Game& game;

	//@{
	/// Game element
	Grid grid;
	Chooser chooser;
	Launcher launcher;
	Timeout_Meter timeout_meter;
	Grid_Shaker grid_shaker;
	Ambient_Effects_Manager ambient_effects_manager;
	Launched_Ball_Manager launched_ball_manager;
	Popped_Balls_Manager popped_balls_manager;
	//@}

	//@{
	/// Game state
	Out_Of_Game_State out_of_game_state;
	Gameplay_State gameplay_state;
	Game_Over_State game_over_state;
	Finished_Level_State finished_level_state;
	//@}

}; // class Game::Impl

Game::Game(Game_Manager& arg_gmgr)
 : game_manager(arg_gmgr),
   impl(new Impl(*this)),
   grid(impl->grid),
   chooser(impl->chooser),
   launcher(impl->launcher),
   timeout_meter(impl->timeout_meter),
   grid_shaker(grid_shaker),
   ambient_effects_manager(impl->ambient_effects_manager),
   launched_ball_manager(impl->launched_ball_manager),
   popped_balls_manager(impl->popped_balls_manager),

   out_of_game_state(impl->out_of_game_state),
   gameplay_state(impl->gameplay_state),
   game_over_state(impl->game_over_state),
   finished_level_state(impl->finished_level_state)
{
	std::auto_ptr<Impl> tmp_impl(impl);
	current = &out_of_game_state;
	current->entry();

	// LASTLASTLAST
	tmp_impl.release();
}

Game::~Game()
{
	delete impl;
}

namespace
{

void
o_sc_msg(const char* msg) throw()
{
	std::cout << "Changed state to " << msg << '\n';
}

} // namespace

void
Game::set_state(int32_t arg)
{
	if(arg == NONE)
		throw Bad_State_Error("Game::set_state() - arg = NONE");

	current->exit();

	switch(arg) {
	case OUT_OF_GAME:
		current = &out_of_game_state; o_sc_msg("Out_Of_Game");
		// Reset the 'finished level' state object so that it
		// starts incrementing stuff from lvl1 again.  I don't
		// know where else to put this.
		finished_level_state.reset();
		break;

	case GAME_OVER:
		current = &game_over_state; o_sc_msg("Game_Over");
		break;

	case FINISHED_LEVEL:
		current = &finished_level_state; o_sc_msg("Finished_Level");
		break;

	case GAMEPLAY:
		current = &gameplay_state; o_sc_msg("Gameplay");
		break;

	default:
		break;
	} // switch(arg) {

	current->entry();
}

void
Game::tock(uint32_t tock_no)
{
	if(game_manager.system.user_input_manager.get_escape())
		game_manager.system.signal_exit();
	current->tock(tock_no);
}

void
Game::tick(uint32_t delta)
{
	impl->game_fps.append_delta(delta);
	current->tick(delta);
}

void
Game::sync_tick(uint32_t delta)
{
	current->sync_tick(delta);
}

Game_Elements_Accessor::Game_Elements_Accessor(Game::Impl& arg)
 : game_impl(arg),
   game(game_impl.game),
   game_manager(game.game_manager)
{
}

#define GAME_ELEMENTS_ACCESSOR_FIMPL(EBig, esmall) \
EBig& \
Game_Elements_Accessor::get_ ## esmall() \
{ \
	return game_impl.esmall; \
} \
const EBig& \
Game_Elements_Accessor::get_ ## esmall() const \
{ \
	return game_impl.esmall; \
}


GAME_ELEMENTS_ACCESSOR_FIMPL(Launcher, launcher)
GAME_ELEMENTS_ACCESSOR_FIMPL(Chooser, chooser)
GAME_ELEMENTS_ACCESSOR_FIMPL(Timeout_Meter, timeout_meter)
GAME_ELEMENTS_ACCESSOR_FIMPL(Grid_Shaker, grid_shaker)
GAME_ELEMENTS_ACCESSOR_FIMPL(Ambient_Effects_Manager, ambient_effects_manager)
GAME_ELEMENTS_ACCESSOR_FIMPL(Launched_Ball_Manager, launched_ball_manager)
GAME_ELEMENTS_ACCESSOR_FIMPL(Popped_Balls_Manager, popped_balls_manager)
GAME_ELEMENTS_ACCESSOR_FIMPL(Grid, grid)

#undef GAME_ELEMENTS_ACCESSOR_FIMPL
	;


} // namespace XBobble

