/* Copyright (C) 2002 Asfand Yar Qazi.

 This file is part of XBobble.

    XBobble is free software; you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    XBobble is distributed in the hope that it will be useful, but
    WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with XBobble; if not, write to the Free Software Foundation,
    Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA */

/** @file Game_Element.hh All Game elements derive from this. */

#ifndef XBOBBLE_GAME_ELEMENT_HH
#define XBOBBLE_GAME_ELEMENT_HH

#include <ayq/stdint.h>
#include "Game_Elements_Accessor.hh"
#include "util.hh"

namespace XBobble
{

class Game;

/** Base class of game elements.  Automatically dispatches tock func
    calls to the correct handler depending on whether or not the
    object is enabled.  Child classes only have to specify handlers,
    not the enable/disable logic.  Since each game element must
    interact with the others, it must have access to them. */
class Game_Element : public Game_Elements_Accessor
{
public:
	/// Init, giving game we belong to
	Game_Element(Game::Impl& arg)
	 : Game_Elements_Accessor(arg), tock_funcs_enabled(false)
	{
	}

	/// Because It Should Be There (tm)
	virtual
	~Game_Element()
	{
	}

	//@{

	/// Call appropriate '_handler()' if this object is enabled
	virtual void
	tock(uint32_t arg)
	{
		if(tock_funcs_enabled)
			tock_handler(arg);
	}

	virtual void
	tick(uint32_t arg)
	{
		if(tock_funcs_enabled)
			tick_handler(arg);
	}

	virtual void
	sync_tick(uint32_t arg)
	{
		if(tock_funcs_enabled)
			sync_tick_handler(arg);
	}

	//@}

	/// Enable tock func processing
	void
	enable_tock_funcs(bool arg)
	{
		tock_funcs_enabled = arg;
		this->enable_tock_funcs_handler(arg);
	}

protected:
	//@{

	/// Handler of appropriate tock func.  Only called if this
	/// object is enabled
	virtual void
	tock_handler(uint32_t tocks_passed)
	{
	}

	virtual void
	tick_handler(uint32_t delta)
	{
	}

	virtual void
	sync_tick_handler(uint32_t delta)
	{
	}

	//@}

	/// Override to be called with appropriate value when
	/// tock_funcs_processing status changed
	virtual void
	enable_tock_funcs_handler(bool)
	{
	}

private:
	/// Is tock func processing enabled?  Initially 'false'
	bool tock_funcs_enabled;

}; // class Game_Element



} // namespace XBobble


#endif // #define XBOBBLE_GAME_ELEMENT_HH
