/* Copyright (C) 2002 Asfand Yar Qazi.

 This file is part of XBobble.

    XBobble is free software; you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    XBobble is distributed in the hope that it will be useful, but
    WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with XBobble; if not, write to the Free Software Foundation,
    Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA */

/** @file Game_Manager.cc @see Game_Manager.hh */

#include <pwd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <string>
#include <iostream>
#include <sstream>
#include <fstream>
#include <SDL.h>
#include "Game_Manager.hh"
#include "Game.hh"
#include "gameplay_variables.hh"

namespace XBobble
{


Tock_Handler_Defs::Tock_Handler_Defs(Game_Manager& arg_gm)
 : game_manager(arg_gm)
{
}

// /////TEMP
// static Sound_Data snd1, snd2, snd3;

void
Tock_Handler_Defs::tock_handler(uint32_t tock_no)
{
	// /////TEMP
// 	static bool firsttime = true;
// 	if(firsttime) {
// 		game_manager.system.sound_output_manager.load_sound(
// 			"sounds/goat.voc", snd1);
// 		game_manager.system.sound_output_manager.load_sound(
// 			"sounds/goat.voc", snd2);
// 		game_manager.system.sound_output_manager.load_sound(
// 			"sounds/heartbeat.voc", snd3);	
// 	firsttime=false;
// 	}
// 	snd1.play_sound();
// 	if(!snd3.is_playing()) snd2.play_sound();
// 	snd3.play_sound();

	game_manager.game.tock(tock_no);
}

void
Tock_Handler_Defs::tick_handler(uint32_t delta)
{
	game_manager.game.tick(delta);
	game_manager.system.video_output_manager.clear_screen();
	game_manager.system.video_output_manager.draw_screen();
}

void
Tock_Handler_Defs::sync_tick_handler(uint32_t delta)
{
	User_Input_Manager& ui = game_manager.system.user_input_manager;
//	Video_Output_Manager& vo = game_manager.system.video_output_manager;

	SDL_Event ev;

	while(SDL_PollEvent(&ev)) {
		ui.process_events(ev);

		switch(ev.type) {
		case SDL_ACTIVEEVENT:
			if(ev.active.state != SDL_APPINPUTFOCUS) break;
			if(ev.active.gain == 0) // lost input focus
				ui.reset_kbd_keys();
			break;

// 		case SDL_VIDEORESIZE: {
// 			std::ostringstream ostr;
// 			ostr << ev.resize.w;
// 			game_manager.set_option("width", ostr.str());
// 			ostr.str("");
// 			ostr << ev.resize.h;
// 			game_manager.set_option("height", ostr.str());
// 			vo.resize();

// 			break; } // case VIDEO_RESIZE:

		case SDL_QUIT:
			game_manager.finish();
			break;

		default:
			break;
		} // switch(ev.type)
	} // while(SDL_PollEvent(&ev))

	game_manager.game.sync_tick(delta);
}


namespace
{

typedef Config_Parser::Symtab::iterator SymtabIt;

/// @internal default options that can be overriden
const std::string default_init_string =
"'width'='640';'height'='480';"
"'fullscreen'='0';"
"'GL_TEXTURE_MIN_FILTER'='GL_LINEAR';"
"'GL_TEXTURE_MAG_FILTER'='GL_LINEAR';"
;

/// @internal Parse a file and add options to given symtab
void
parse_file(const std::string& filename, Config_Parser::Symtab& symtab,
	   const std::string& error_string)
{
	Config_Parser parser;

	std::ifstream tmpin(filename.c_str());
	if(tmpin.good())
		parser.process(tmpin, std::cerr, symtab,
			       error_string);
} // parse_file()

} // namespace

Game_Manager::Option_Getter::Option_Getter(const std::string& more_options)
{
	Config_Parser parser;

	std::istringstream def_init_istream(default_init_string);
	/// First the default init string
	parser.process(def_init_istream, std::cerr,
		       options, "[XBobble Default Init String] ");

	// Get environment variable 'XBOBBLE1_OPTIONS and send it to
	// Config_Parser
	char* envtmp = std::getenv("XBOBBLE1_OPTIONS");
	std::string envstring = (envtmp != 0) ? envtmp : "";

	// Then the environment
	std::istringstream env_istream(envstring);
	parser.process(env_istream, std::cerr,
		       options, "[XBobble Env String] ");

	// Then get the global config file (if it exists) and load in
	// stuff from there.
	SymtabIt global_fname = options.find("global_cfg_file");
	if(global_fname != options.end())
		parse_file(global_fname->second, options,
			   "[XBobble Global cfg file] ");

	// Do the same for the user config file (if it exists)
	struct passwd* pwdline = getpwuid(getuid());
	std::string homedir_file
		= std::string(pwdline->pw_dir)
		+ "/.xbobble1.cfg";
	parse_file(homedir_file, options, "[XBobble Home Dir cfg file] ");

	// Now parse the given arguments (if needed)
	std::istringstream moreopts_istream(more_options);
	parser.process(moreopts_istream, std::cerr,
		       options, "[XBobble Command Line] ");

	// Did default data path get included?  No?  Then exit with a
	// heinous error
	if(options.find("data_root") == options.end())
		throw Init_Error("Game_Manager() - data_root not specified!");
}


Game_Manager::Game_Manager(const std::string& more_options)
try
 : Singleton_Class("Game_Manager"),
   option_getter(more_options), system(*this), func_defs(*this),
   tock_manager(Vars::tock_size, func_defs), game(*this)
{
#	ifndef NDEBUG
	const Config_Parser::Symtab& opts = option_getter.options;
	typedef std::map<std::string, std::string>::const_iterator It;
	It it=opts.begin(), itend=opts.end();
	for(; it != itend; ++it)
		std::cout << '"' << it->first << "\" = \""
			  << it->second << "\";\n";
#	endif // #ifndef NDEBUG

	tock_manager.start_loop();
}
catch(std::logic_error& e)
{
	std::cerr << "Bug found!\n";
	std::cerr << "Exception occured: " << typeid(e).name() << '\n'
		  << '\t' << e.what() << '\n';
	std::terminate();
}


std::string
Game_Manager::get_option(const std::string& name) const
{
	Config_Parser::constSymtab& opts(option_getter.options);
	Config_Parser::Symtab::const_iterator it = opts.find(name);
	if(it == opts.end())
		return std::string("");
	else
		return it->second;
}

void
Game_Manager::set_option(const std::string& name, const std::string& value)
{
	option_getter.options[name] = value;
}

} // namespace XBobble

