/* Copyright (C) 2002 Asfand Yar Qazi.

 This file is part of XBobble.

    XBobble is free software; you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    XBobble is distributed in the hope that it will be useful, but
    WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with XBobble; if not, write to the Free Software Foundation,
    Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA */

/** @file Game_Manager.hh Manages the whole game thing. */

#ifndef XBOBBLE_GAME_MANAGER_HH
#define XBOBBLE_GAME_MANAGER_HH

#include <string>
#include <ayq/stdint.h>
#include <ayq/models/Tock.hh>
#include "Config_Parser.hh"
#include "util.hh"
#include "System.hh"
#include "Game.hh"

namespace XBobble
{

class Game_Manager;

/// The tock handler definitions for the Tocks model
class Tock_Handler_Defs
{
public:
	/// Init with owning game_manager
	Tock_Handler_Defs(Game_Manager& arg_gm);

	/// Process tock
	void
	tock_handler(uint32_t);

	/// Process ticks
	void
	tick_handler(uint32_t);

	/// Synchronise tock and tick data
	void
	sync_tick_handler(uint32_t);

private:
	/// The owning object
	Game_Manager& game_manager;

}; // class Tock_Handler_Defs

/// Father of the whole game (more or less) - manages everything,
/// starts things going and stuff.  Singleton.
class Game_Manager : public Singleton_Class
{
public:
	/// A class that, on construction, gets all options for
	/// Game_Manager from the various sources.  If a compulsory
	/// option is not set, will throw an exception.  This will
	/// ensure none of Game_Manager's owned classes are
	/// constructed if necessary options are not set.
	class Option_Getter
	{
	public:
		/// Store options in the argument
		Option_Getter(const std::string& more_options);

		/// The config options.
		Config_Parser::Symtab options;

	}; // class Option_Getter

	/// Pass a string containing config options obtained at
	/// run-time (typically from the command line.)
	Game_Manager(const std::string& more_options);

	/// Returns the value of an option ("" if the option hasn't
	/// been set.)
	std::string
	get_option(const std::string& name) const;

	/// Set a value for an option parameter (ie first arg is the
	/// key, second is the value)
	void
	set_option(const std::string& name, const std::string& value);

	/// As above, except takes chars instead of std::strings
	void
	set_option(const char* arg1, const char* arg2)
	{
		std::string s1(arg1), s2(arg2);
		set_option(s1, s2);
	}

	/// And another one...
	void
	set_option(const char* arg1, const std::string& arg2)
	{
		const std::string s1(arg1);
		set_option(s1, arg2);
	}

	/// Get LOD value
	uint32_t
	get_lod() const
	{
		int lodval = atoi(get_option("lod").c_str());
		if(lodval < 1) lodval = 1;
		else if(lodval > 3) lodval = 3;
		return static_cast<uint32_t>(lodval);
	}

	/// Returns a const reference to option_getter.options, so as
	/// to support ierating through the list for printing out,
	/// etc.
	const Config_Parser::Symtab&
	get_options_map() const
	{
		return option_getter.options;
	}

	/// Set the tock_manager to finish.  Exits the game.
	void
	finish()
	{
		tock_manager.set_finish();
	}

private:
	/// First owned object.  Parses options on construction, or
	/// fails with an exception, so ensuring that no other owned
	/// objects get constructed.
	Option_Getter option_getter;

public:
	/// The system object.
	System system;

private:
	/// Tock function definitions.
	Tock_Handler_Defs func_defs;

	typedef AYQ::One::Tock_Manager<System,
				       Tock_Handler_Defs> Tock_Manager_T;

	/// The Tock_Manager.  Groovy.
	Tock_Manager_T tock_manager;

public:
	/// Game object
	Game game;

}; // class Game_Manager


} // namespace XBobble


#endif // #define XBOBBLE_GAME_MANAGER_HH
