/* Copyright (C) 2002 Asfand Yar Qazi.

 This file is part of XBobble.

    XBobble is free software; you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    XBobble is distributed in the hope that it will be useful, but
    WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with XBobble; if not, write to the Free Software Foundation,
    Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA */

/** @file Gameplay_State.cc @see Gameplay_State.hh */

#include <stdexcept>
#include "Gameplay_State.hh"
#include "Game.hh"
#include "User_Input_Manager.hh"
#include "Game_Manager.hh"
#include "Grid.hh"
#include "Chooser.hh"
#include "Launcher.hh"
#include "Ambient_Effects_Manager.hh"
#include "Popped_Balls_Manager.hh"
#include "Launched_Ball_Manager.hh"
#include <SDL_opengl.h>

namespace XBobble
{

Gameplay_State::Gameplay_State(Game::Impl& arg)
 : State(arg), substate(NONE)
{
}

void
Gameplay_State::entry()
{
	game.game_manager.system.user_input_manager.reset_joy();
	game.game_manager.system.user_input_manager.reset_kbd_keys();
	set_substate(WAITING);
	get_grid().enable_drawing(true);
	get_chooser().enable_drawing(true); get_chooser().reset();
	get_launcher().enable_tock_funcs(true); get_launcher().reset();
	get_ambient_effects_manager().enable_tock_funcs(true);
	get_popped_balls_manager().enable_tock_funcs(true);

	glClearColor(0.4, 0.1, 0.1, 0);
}

void
Gameplay_State::exit()
{
	set_substate(NONE);
}

void
Gameplay_State::set_substate(int arg)
{
	if(arg == substate)
		return;

	switch(arg) {
	case WAITING:
	case LAUNCHING:
	case NONE:
		this->subexit();
		substate = arg;
		this->subentry();
		break;

	default:
		throw Bad_State_Error("Gameplay_State::set_substate()");
		break;

	} // switch(arg)

} // Gameplay_State::set_substate()

void
Gameplay_State::subentry()
{
	switch(substate) {
	case WAITING:
		game.game_manager.system.user_input_manager.reset_kbd_keys();
		game.game_manager.system.user_input_manager.reset_joy();
		break;

	case LAUNCHING:
		get_launcher().launch();
		get_launched_ball_manager().enable_tock_funcs(true);
		break;

	case NONE:
		break;

	default:
		// We should never be here!
		throw Bad_State_Error("Gameplay_State::entry() - "
				      "invalid current substate");
		break;
	} // switch(substate)

} // Gameplay_State::subentry()

void
Gameplay_State::subexit()
{
	switch(substate) {
	case WAITING:
		break;

	case LAUNCHING:
		get_launched_ball_manager().enable_tock_funcs(true);
		break;

	case NONE:
		break;

	default:
		throw Bad_State_Error("Gameplay_State::exit() - "
				      "invalid current substate");
		break;

	} // switch(substate)

} // Gameplay_State::subexit()

void
Gameplay_State::tock(uint32_t tocks_passed)
{
	User_Input_Manager& ui = game.game_manager.system.user_input_manager;

	get_ambient_effects_manager().tock(tocks_passed);
	get_launcher().tock(tocks_passed);

	switch(substate) {
	case WAITING:
		if(ui.get_spacebar() || ui.get_button()) {
			this->set_substate(LAUNCHING);
		} // if(ui....)
		break;

	case LAUNCHING:
		if(!get_launched_ball_manager().get_ball_stopped())
			break;
		switch(get_grid().get_condition()) {
		case Grid::FINISHED_LEVEL:
			game.set_state(Game::FINISHED_LEVEL);
			return;

		case Grid::GAME_OVER:
			game.set_state(Game::GAME_OVER);
			return;

		case Grid::OK:
			set_substate(WAITING);
			break;

		} // switch(get_grid().get_condition()) {
		break;

	case NONE:
		break;

	default:
		throw Bad_State_Error("Gameplay_State::tock() - "
				      "invalid current substate");

	} // switch(substate)

} // Gameplay_State::tock()

void
Gameplay_State::tick(uint32_t delta)
{
	get_ambient_effects_manager().tick(delta);
	get_popped_balls_manager().tick(delta);

	switch(substate) {
	case WAITING:
		break;

	case LAUNCHING:
		break;

	case NONE:
		break;

	default:
		throw Bad_State_Error("Gameplay_State::tick() - "
				      "invalid current substate");
	} // switch(substate)

} // Gameplay_State::tick()

void
Gameplay_State::sync_tick(uint32_t delta)
{
	get_ambient_effects_manager().sync_tick(delta);

	switch(substate) {
	case WAITING:
		break;

	case LAUNCHING:
		get_launched_ball_manager().sync_tick(delta);
		break;

	case NONE:
		break;

	default:
		throw Bad_State_Error("Gameplay_State::sync_tick() - "
				      "invalid current substate");
	} // switch(substate)

} // Gameplay_State::sync_tick()


} // namespace XBobble

