/* Copyright (C) 2002 Asfand Yar Qazi.

 This file is part of XBobble.

    XBobble is free software; you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    XBobble is distributed in the hope that it will be useful, but
    WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with XBobble; if not, write to the Free Software Foundation,
    Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA */

/** @file Lexer_Base.hh Simple base class for lexers to derive from.
    Provides a dynamic input stream assignment policy, and partial
    next-token finding support. */

#ifndef XBOBBLE_LEXER_BASE_HH
#define XBOBBLE_LEXER_BASE_HH

#include <iostream>
#include <iomanip>
#include <string>
#include <stdexcept>

namespace XBobble
{


/// Simple base class that supports changing input streams to read
/// tokens from.  Skips all whitespace.
class Basic_Lexer
{
public:
	/// Init with no stream connected
	Basic_Lexer()
	 : streamin(0), line_num(1)
	{
	}

	/// Init with argument stream connected
	Basic_Lexer(std::istream& arg_istream)
	 : streamin(0), line_num(1)
	{
		set_istream(arg_istream);
	}

	/// Exception thrown by find_next() if EOF was the next
	/// non-whitespace char found.
	class End_Of_Stream : public std::runtime_error
	{
	public:
		End_Of_Stream()
		 : std::runtime_error("")
		{
		}
	}; // class End_Of_Stream

	/// Sets the current input stream to the argument.  If it is
	/// not a valid istream ready for reading, the internal
	/// istream is set to NULL.
	void
	set_istream(std::istream& arg_istream)
	{
		streamin = &arg_istream;

		if(!istream_ok())
			streamin = 0;

		line_num = 1;
	}

protected:
	/// Returns the first non-whitespace (!std::isspace(c)) char
	/// it finds in the stream (uses get(), so the stream position
	/// is incremented), or throws End_Of_Stream (which must be
	/// caught by the calling function.)  Designed to be called by
	/// children classes.
	char
	find_next() throw(End_Of_Stream)
	{
		if(!istream_ok())
			throw End_Of_Stream();

		char c;

		do {
			c = ' ';
			streamin->get(c);
			if(c == '\n')
				++line_num;
		} while(istream_ok() && std::isspace(c));

		if(!istream_ok())
			throw End_Of_Stream();

		return c;
	}

public:
	/// Returns whether the current istream is in a valid state,
	/// which includes not having reached end-of-file.  If it is
	/// not, it will have to be reset with set_istream.
	bool
	istream_ok() const
	{
		if(!streamin)
			return false;

		if(!streamin->good())
			return false;

		return true;
	}

	/// Returns the current line number.  Note that the current
	/// line number is reset to 1 after loading a new istream.
	int
	get_line_num() const
	{
		return line_num;
	}

protected:
	/// Internal istream
	std::istream* streamin;

	/// Current line number is returned (with-respect-to the
	/// number of carriage returns ("\n") passed so far)
	int line_num;

}; // class Basic_Lexer


} // namespace XBobble


#endif // #define XBOBBLE_LEXER_BASE_HH
